/*
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 * 
 *  Gabber
 *  Copyright (C) 1999-2001 Dave Smith & Julian Missig
 */

#include "GabberApp.hh"
#include "GabberWin.hh"
#include "GabberUtility.hh"
#include "GladeHelper.hh"
#include "MessageViews.hh"
#include "AddContactDruid.hh"
#include "ContactInfoInterface.hh"
#include "ContactInterface.hh"
#include "ChatView.hh"
#include "MessageManager.hh"
#include "ErrorManager.hh"
#include "GabberGPG.hh"
#include "ConfigManager.hh"
#include "GCInterface.hh"

// GtkText extensions
#include "gtkspell.h"
#include "gtkurl.h"

extern "C" {
#ifdef GABBER_WINICON
#include <libgnomeui/gnome-window-icon.h>
#endif
}

using namespace jabberoo;
using namespace GabberUtil;

// -------------------------------------------------------------------
//
// Basic Message View
//
// -------------------------------------------------------------------
MessageView::MessageView(const jabberoo::Message& m, ViewMap& vm, bool only_userhost)
     : _jid(m.getFrom()), _only_userhost(only_userhost), _registrar(vm)
{
     // Register with the registrar
     if (_only_userhost)
     {
	  // only register the user@host portion of the JID
	  _registrar.insert(make_pair(JID::getUserHost(_jid), this));
     }
     else
     {
	  // register the entire JID
	  _registrar.insert(make_pair(_jid, this));
     }
}

MessageView::MessageView(const string& jid, ViewMap& vm, bool only_userhost)
     : _jid(jid), _only_userhost(only_userhost), _registrar(vm)
{
     // Register with the registrar
     if (_only_userhost)
     {
	  // only register the user@host portion of the JID
	  _registrar.insert(make_pair(JID::getUserHost(_jid), this));
     }
     else
     {
	  // register the entire JID
          _registrar.insert(make_pair(_jid, this));
     }
}

MessageView::~MessageView()
{
     // Remove the entry for this object
     if (_only_userhost)
     {
	  // remove only a user@host jid
	  _registrar.erase(JID::getUserHost(_jid));
     }
     else
     {
	  // remove the full JID
	  _registrar.erase(_jid);
     }
}

// -------------------------------------------------------------------
//
// Normal Message View
//
// -------------------------------------------------------------------
NormalMessageView::NormalMessageView(const jabberoo::Message& m, ViewMap& vm)
     : MessageView(m, vm), BaseGabberWindow("Message_win")
{
     init();

     // If this is a reply to a one-on-one chat (it's possible),
     if (m.getType() == Message::mtChat)
          // set send as ooochat
          _chkOOOChat->set_active(true);

     // Set the title
     _thisWindow->set_title(_("Gabber: Message from ") + _nickname);
     _entSendTo->set_text(fromUTF8(_entSendTo, JID::getUserHost(m.getFrom())));
     _entResource->set_text(fromUTF8(_entResource, JID::getResource(m.getFrom())));
     _lblTimeLbl->show();
     _lblTime->show();
     _lblTime->set_text(m.getDateTime());

     // Display: nickname (jabberid)
     string displaynick = _nickname + " (" + m.getFrom() + ")";

     if (!m.getError().empty()) // If there's an error message, go ahead and parse it
     {
	  // Hide received parts
	  set_received(false);
	  // Set the label appropriately
	  _frmReply->set_label(fromUTF8(_frmReply, (_("Message to: ") + displaynick)));
	  // Show the error
	  on_error(_jid, m.getErrorCode(), _("Error"), m.getBody());
     }
     else // There's no error message, so this is a reply
     {
	  // Show needed parts
	  set_received(true);
	  // Set the label appropriately
	  _frmReply->set_label(fromUTF8(_frmReply, (_("Reply to: ") + displaynick)));
	  // Parse out the message
	  render(m);
     }

     // Set the subject, if it exists
     const string& subj = m.getSubject();
     if (!subj.empty())
     {
        _thisWindow->set_title(_("Gabber: Message - ") + subj);
        // Hrmm... if there's a Re:, re:, or RE:, we should do nothing.
        // but if not, should we add a re: ?
	// I think so.
        _entSubject->set_text(fromUTF8(_entSubject, subj));
     }

     // Grab the thread
     _thread = m.getThread();

     // Display
     show();
     _thisWindow->set_default_size(400, 350);
     _thisWindow->show();
}

NormalMessageView::NormalMessageView(const string& jid, ViewMap& vm)
     : MessageView(jid, vm), BaseGabberWindow("Message_win")
{
     init();

     // Disable unneeded parts
     set_received(false);

     // If no JID is specified, be sure that the advanced options
     // are automatically displayed
     if (jid.empty())
     {
          _tglbtnAdvncdOpts->set_active(true);
          _frmAdvncdOpts->show();
	  
     }

     // Set the title
     _thisWindow->set_title(_("Gabber: Message to ") + _nickname);
     _entSendTo->set_text(fromUTF8(_entSendTo, JID::getUserHost(jid)));
     _entResource->set_text(fromUTF8(_entResource, JID::getResource(jid)));

     // Display: nickname (jabberid)
     string displaynick = _nickname + " (" + jid + ")";

     _frmReply->set_label(fromUTF8(_frmReply, (_("Message to: ") + displaynick)));

     // Display
     show();
     _thisWindow->set_default_size(400, 350);
     _thisWindow->show();
}

NormalMessageView::~NormalMessageView()
{
}

void NormalMessageView::init()
{
     _thisWindow->realize();

     _onRoster = true;
     // default is the username
     _nickname = JID::getUser(_jid);
     try {
          _nickname = G_App->getSession().roster()[JID::getUserHost(_jid)].getNickname();
     } catch (Roster::XCP_InvalidJID& e) {
          // Special handling for a groupchat ID -- use the resource as the nickname
          if (G_App->isGroupChatID(_jid))
               _nickname = JID::getResource(_jid);
          // Clear the onRoster flag
          _onRoster = false;
     }

     list<string> resources;
     try {
	  // Walk the list of resources and add them to the combo box
	  PresenceDB::range r = G_App->getSession().presenceDB().equal_range(_jid);
	  for (PresenceDB::const_iterator it = r.first; it != r.second; it++)
	  {
	       const Presence& p = *it;

               // If this presence is a NA presence, then skip it
	       if (p.getType() == Presence::ptUnavailable)
		    continue;

               // Extract the resource
               const string& res = JID::getResource(p.getFrom());
               resources.push_back(res);
	  }
     } catch (PresenceDB::XCP_InvalidJID& e) {
	  // No presences from any resources
     }

     // Connect buttons to handler
     getButton("Message_Send_btn")->clicked.connect(slot(this, &NormalMessageView::on_Send_clicked));
     getButton("Message_Cancel_btn")->clicked.connect(slot(this, &NormalMessageView::on_Cancel_clicked));
     _btnQuote = getButton("Message_Quote_btn");
     _btnQuote->clicked.connect(slot(this, &NormalMessageView::on_Quote_clicked));
     _btnAddUser = getButton("Message_AddUser_btn");
     _btnAddUser->clicked.connect(slot(this, &NormalMessageView::on_AddUser_clicked));
     _btnAddUser->set_sensitive(!_onRoster);
     Gtk::Button* b = getButton("Message_EditUser_btn");
     b->clicked.connect(slot(this, &NormalMessageView::on_EditUser_clicked));
     getButton("Message_History_btn")->clicked.connect(slot(this, &NormalMessageView::on_History_clicked));

     _thisWindow->key_press_event.connect(slot(this, &NormalMessageView::on_window_key_press));
     _thisWindow->delete_event.connect(slot(this, &NormalMessageView::on_window_delete));
//   In the current implementation we don't want to connect error messages to a normal message view
//     G_App->getErrorManager().errorNormal.connect(slot(this, &NormalMessageView::on_error));

     // Setup widgets
     _tglbtnAdvncdOpts = getWidget<Gtk::ToggleButton>("Message_AdvancedOptions_tglbtn");
     _tglbtnAdvncdOpts ->toggled.connect(slot(this, &NormalMessageView::on_AdvncdOpts_toggled));
     _tglbtnReceived   = getWidget<Gtk::ToggleButton>("Message_Received_tglbtn");
     _tglbtnReceived   ->toggled.connect(slot(this, &NormalMessageView::on_Received_toggled));
     _frmAdvncdOpts    = getWidget<Gtk::Frame>("Message_AdvancedOptions_frame");
     _frmReceived      = getWidget<Gtk::Frame>("Message_Received_frame");
     // Toolbar
     getWidget<Gtk::Toolbar>("Message_toolbar")->set_style(GTK_TOOLBAR_ICONS);

     // Figure out if advanced options should be shown
     if (G_App->getCfg().getBoolValue("Messages/ShowAdvncdOpts=false"))
     {
          _tglbtnAdvncdOpts->set_active(true);
          _frmAdvncdOpts->show();
     }

     // Set the reply to frame
     _frmReply = getWidget<Gtk::Frame>("Message_Reply_frame");

     // Initialize pointer to "next" button (should be insensitive by default)
     _btnNext = getButton("Message_ReadNext_btn");
     _btnNext->clicked.connect(slot(this, &NormalMessageView::on_Next_clicked));
     _btnNext->set_sensitive(false);

     // Initialize pointers
     _entSender        = getEntry("Message_Sender_txt");
     _lblTimeLbl       = getLabel("Message_TimeLabel_lbl");
     _lblTime          = getLabel("Message_Time_lbl");
     _memIncomingBody  = getWidget<Gtk::Text>("Message_Body_txt");
     _memIncomingBody  ->set_word_wrap(true);
     gtkurl_attach(_memIncomingBody->gtkobj()); // Attach gtkurl to this - thanks bratislav!
     _memOutgoingBody  = getWidget<Gtk::Text>("Message_ReplyBody_txt");
     _memOutgoingBody  ->set_word_wrap(true);
     if (gtkspell_running())
	  gtkspell_attach(_memOutgoingBody->gtkobj()); // Attach gtkspell for spell checking
     _entSendTo        = getEntry("Message_SendTo_txt");
     _entSendTo        ->changed.connect(slot(this, &NormalMessageView::on_SendTo_changed));
     _entResource      = getEntry("Message_Resource_txt");
     _entResource      ->changed.connect(slot(this, &NormalMessageView::on_SendTo_changed));
     _cboResource      = getWidget<Gtk::Combo>("Message_Resource_cmbo");
     if (!resources.empty())
	  _cboResource ->set_popdown_strings(resources);
     _entSubject       = getEntry("Message_Subject_txt");
     _chkOOOChat       = getCheckButton("Message_OOOChat_chk");
     _pixGPG           = getWidget<Gnome::Pixmap>("Message_GPG_pix");
     _evtGPG           = getWidget<Gtk::EventBox>("Message_GPG_evt");
     _gpgInfo          = GabberGPG::gpgNone;

     _render_message = true;

     // Pixmaps
     string pix_path = ConfigManager::get_PIXPATH();
#ifdef GABBER_WINICON
     string window_icon = pix_path + "gnome-message.xpm";
     gnome_window_icon_set_from_file(_thisWindow->gtkobj(),window_icon.c_str());
     gnome_window_icon_init();
#endif

     // If gpg is diabled disable the encrypt button otherwise turn of the encryption button, it will be turned on if necessary
     // in OnSendToChanged()
     GabberGPG& gpg = G_App->getGPG();
     // _useGPG will be set when sendto is set or when a message is rendered
     _useGPG = false;
     if (!gpg.enabled())
          getWidget<Gtk::ToggleButton>("Message_Encrypt_tglbtn")->set_sensitive(false);
     else
	  getWidget<Gtk::ToggleButton>("Message_Encrypt_tglbtn")->set_active(false);

     // Add button_press handler for GPGInfo Dialog
     _evtGPG->button_press_event.connect(slot(this, &NormalMessageView::on_GPGInfo_button_press));
}

void NormalMessageView::render(const jabberoo::Message& m)
{
     if (_render_message)
     {
	  set_received(true);
          _entSender->set_text(fromUTF8(_entSender, _nickname));
          int i = 0;
          if (m.getBody().length() > 0)
	  {
               _memIncomingBody->insert_text(fromUTF8(_memIncomingBody, m.getBody()).c_str(),
                                             fromUTF8(_memIncomingBody, m.getBody()).length(), &i);
	       gtkurl_check_all(_memIncomingBody->gtkobj()); // Highlight the URLs
	  }
          // Hrmm, this doesn't seem to scroll it back to the beginning
          _memIncomingBody->set_point(0);

	  // change to Encryption icon to unlock until we know whether this message has any encryption
	  _pixGPG->load(string(ConfigManager::get_SHAREDIR()) + string("gpg-unencrypted.xpm"));
	  // Set the tooltip
	  _tips.set_tip(*_evtGPG, _("This message is not encrypted"));

	  // reset useGPG to false, if the message is encrypted or signed it will be changed
	  // back to true to enable signing of outgoing messages
	  _useGPG = false;
          Tag* x = m.getBaseTag().getTag("x", "jabber:x:signed");
	  GabberGPG& gpg = G_App->getGPG();
          if (x != NULL && gpg.enabled())
	  {
	       string message = m.getBody();
	       string encrypted = x->getData();
	       GPGInterface::SigInfo info;
	       GPGInterface::Error err;

               if ((err = gpg.verify(info, encrypted, message)) != GPGInterface::errOK)
               {
                    if (err == GPGInterface::errPubKey)
                    {
                         cerr << "FIXME: need a way to get users Public Key" << endl;
                    }
		    // Set icon to invalid signature
		    _pixGPG->load(string(ConfigManager::get_SHAREDIR()) + string("gpg-badsigned.xpm"));
		    // Set the tooltip
		    _tips.set_tip(*_evtGPG, _("Signature on message is invalid"));
		    _gpgInfo = GabberGPG::gpgInvalidSigned;
               }
	       else
	       {
		    // It was a valid signature so add the KeyID to the KeyMap
		    gpg.add_jid_keyid(m.getFrom(), info.get_key().get_keyid());
		    // Set icon to valid signature icon
	            _pixGPG->load(string(ConfigManager::get_SHAREDIR()) + string("gpg-signed.xpm"));
		    // Set the tooltip
		    _tips.set_tip(*_evtGPG, _("Signature on message is valid"));
		    _gpgInfo = GabberGPG::gpgValidSigned;
		    // received a signed message so reply with signed messages
		    _useGPG = true;
	       }
          }
	  x = m.getBaseTag().getTag("x", "jabber:x:encrypted");
	  if (x != NULL && gpg.enabled()) {
	       string message;
	       string encrypted = x->getData();
	       GPGInterface::DecryptInfo info;
	       GPGInterface::Error err;

	       if ((err = gpg.decrypt(info, encrypted, message)) != GPGInterface::errOK)
	       {
		    if (err == GPGInterface::errPubKey)
		    {
		         cerr << "FIXME: need a way to get the user's Public Key" << endl;
		    }
		    // If the decryption failed, there's not much we can do with the message
		    return;
	       }
	       // Need to figure out if it's a WinJab encrypted message and make a second pass over the
	       // decrypted text to verify the signature
	       bool winjab_sig = false;
	       if (strncmp(message.c_str(), "-----BEGIN PGP", 14) == 0)
	       {
	            // It is a winjab message so we need to run it through gpg again
	            if ((err = gpg.verify_clear((GPGInterface::SigInfo &) info.get_sig(), message, message)) != GPGInterface::errOK)
	            {
	                 cerr << "FIXME: don't have user's public key" << endl;
	                 return; 
	            }
	            winjab_sig = true;
	       }
	       // Set the message text to what was just decrypted
	       _memIncomingBody->delete_text(0, -1);
	       i = 0;
	       _memIncomingBody->insert_text(fromUTF8(_memIncomingBody, message).c_str(), fromUTF8(_memIncomingBody, message).length(), &i);
	       _memIncomingBody->set_point(0);
	       // If the encryption had a signature (which it should) add the keyid to the KeyMap
	       if ((info.has_sig() || winjab_sig) && info.get_sig().valid())
	            gpg.add_jid_keyid(m.getFrom(), info.get_sig().get_key().get_keyid());
	       _pixGPG->load(string(ConfigManager::get_SHAREDIR()) + string("gpg-encrypted.xpm"));
	       // Set the tooltip
	       _tips.set_tip(*_evtGPG, _("This message is encrypted"));
	       _gpgInfo = GabberGPG::gpgValidEncrypted;
	       // This message was encrypted for enable encryption for outgoing messages
	       getWidget<Gtk::ToggleButton>("Message_Encrypt_tglbtn")->set_active(true);
	       _useGPG = true;
	  }
          _render_message = false;
     }
     else
     {
          // Add the message to the _Messages queue
          _Messages.push_back(m);
          // Enable the next button, if necessary
          _btnNext->set_sensitive(true);
     }
}

void NormalMessageView::on_Send_clicked()
{
     
     // Compose message from text in the text box
     if (!_memOutgoingBody->get_chars(0, -1).empty() && !_entSendTo->get_text().empty())
     {
          // Ensure the message is not editable while we're sending it...
          _memOutgoingBody->set_sensitive(false);

          // Get the JID to send to
          string jidto;
          if (!_entResource->get_text().empty())
               jidto = toUTF8(_entSendTo, _entSendTo->get_text()) + "/" + toUTF8(_entResource, _entResource->get_text());
          else
               jidto = toUTF8(_entSendTo, _entSendTo->get_text());

          // If "Send as OOOChat" is picked, send the messages w/ a OOOChat flag
          Message::Type mtype;
          if (_chkOOOChat->get_active())
               mtype = Message::mtChat;
          else
               mtype = Message::mtNormal;

	  // message contents
	  string body = toUTF8(_memOutgoingBody, _memOutgoingBody->get_chars(0,-1));
          // Construct the message
	  Message m(jidto, body, mtype);

	  MessageManager::Encryption enc = MessageManager::encNone;
	  // If the encryption toggle button is active encrypt the message
	  // send_message will check if gpg is enabled
	  if (getWidget<Gtk::ToggleButton>("Message_Encrypt_tglbtn")->get_active())
	  {
	       // encrypt the message
	       enc = MessageManager::encEncrypt;
	  }
	  // If we aren't encrypting then sign the message
	  else if (_useGPG)
	  {
	       // sign the message
	       enc = MessageManager::encSign;
	  }

          // If they set a subject, add it to the message
          string subject = toUTF8(_entSubject, _entSubject->get_text());
          if (!subject.empty())
               m.setSubject(subject);

	  // Set the thread
	  if (!_thread.empty())
	       m.setThread(_thread);

          // Actually send the message
          bool message_sent = G_App->getMessageManager().send_message(m, enc);

          // If the message wasn't sent, allow them to edit it again.
          if (!message_sent)
               _memOutgoingBody->set_sensitive(true);

          // If this window is allowed to close BUT there are messages
          else if (!_Messages.empty())
          {
               _memOutgoingBody->delete_text(0, -1);
               on_Next_clicked();
               _memOutgoingBody->set_sensitive(true);
          }
          else
          {
               // Save whether or not advanced options should be displayed
               G_App->getCfg().putValue("Messages/ShowAdvncdOpts", _tglbtnAdvncdOpts->get_active());
               G_App->getCfg().sync();

               // And we're done... close
               close();
          }
     }
}

void NormalMessageView::on_Cancel_clicked()
{
     // If there are more messages in the queue,
     // act as if they pressed the next button
     // (prevents loss of messages)
     if (!_Messages.empty())
          on_Next_clicked();
     else
          close();
}

gint NormalMessageView::on_window_delete(GdkEventAny* e)
{
     on_Cancel_clicked();
     return 0;
}

gint NormalMessageView::on_window_key_press(GdkEventKey* e)
{
     // If Ctrl-Enter is pressed, send the thing
     if ( (e->keyval == GDK_Return) &&
          (e->state & GDK_CONTROL_MASK))
          on_Send_clicked();
     // If Shift-Enter is pressed, insert a newline
     else if ( (e->keyval == GDK_Return) && (e->state & GDK_SHIFT_MASK) )
	  e->state ^= GDK_SHIFT_MASK;
     else if (e->keyval == GDK_space && gtkspell_running())
	  gtkspell_check_all(_memOutgoingBody->gtkobj());
     return 0;
}

void NormalMessageView::on_error(const string& concerning, int errorcode, const string& errormsg, const string& body)
{
     // If it's not for this view, ignore
     if (concerning != _jid)
	  return;

     // Display the error
     string error = _("Error");
     if (errorcode != 0)
     {
	  char* errnum;
	  errnum = g_strdup_printf(" %d", errorcode);
	  error += errnum;
	  g_free(errnum);
     }
     if (!errormsg.empty())
     {
	  error += ": ";
	  error += errormsg;
     }
     // Set the error label
     Gtk::Label* l = getLabel("Message_Error_lbl");
     l->set_text(fromUTF8(l, error));
     // Display the error frame
     getWidget<Gtk::Frame>("Message_Error_frm")->show();
     
     // Set the message body to be the message they were working on
     int i = 0;
     if (body.length() > 0)
     {
	  _memOutgoingBody->insert_text(fromUTF8(_memOutgoingBody, body).c_str(),
					fromUTF8(_memOutgoingBody, body).length(), &i);
     }
}

void NormalMessageView::on_Next_clicked()
{
     // If there are more messages in the queue..
     if (!_Messages.empty())
     {
          // Toggle the Received Message display button
          _tglbtnReceived->set_active(true);
          // Show the Received Message frame
          _frmReceived->show();
          // Clear out the body text
          _memIncomingBody->delete_text(0, -1);
          // Render the new message
          _render_message = true;
          render(_Messages.front());
          // Release the message from the queue
          _Messages.pop_front();
     }
     // Determine if the button should remain sensitive
     if (_Messages.empty())
          _btnNext->set_sensitive(false);
}

void NormalMessageView::on_AdvncdOpts_toggled()
{
     if (_tglbtnAdvncdOpts->get_active())
          _frmAdvncdOpts->show();
     else
          _frmAdvncdOpts->hide();
}

void NormalMessageView::on_Received_toggled()
{
     if (_tglbtnReceived->get_active())
          _frmReceived->show();
     else
          _frmReceived->hide();
}

void NormalMessageView::set_received(bool has_received)
{
     if (has_received)
     {
          _frmReceived->show();
          _tglbtnReceived->set_sensitive(true);
	  _tglbtnReceived->set_active(true);
	  _btnQuote->set_sensitive(true);
     }
     else
     {
	  _frmReceived->hide();
	  _tglbtnReceived->set_sensitive(false);
	  _tglbtnReceived->set_active(false);
	  _btnQuote->set_sensitive(false);
     }
}

void NormalMessageView::on_SendTo_changed()
{
     _sendChanged = true;

     // If it's a transport, use the transport nickname
     string displaynick;
     if (G_App->isTransport(_entSendTo->get_text()))
     {
          string transName = _entSendTo->get_text().substr(0, _entSendTo->get_text().find('.'));
          displaynick = transName + " Transport";
     }
     // Else use nickname (jabberid)
     else
     {
          if (!_entResource->get_text().empty())
               displaynick = _entSendTo->get_text() + "/" + _entResource->get_text();
          else
               displaynick = _entSendTo->get_text();
     }

     // Set the labels appropriately
     if (!_btnNext->is_visible())
          _frmReply->set_label(_("Message to: ") + displaynick);
     else
          _frmReply->set_label(_("Reply to: ") + displaynick);

     // If it contains @ - a very basic test, oh well
     string sendto = _entSendTo->get_text();
     string found = sendto.substr(0, sendto.find('@'));
     // Then display the add user button
     if (sendto != found && !sendto.empty() && _btnNext->is_visible())
          _btnAddUser->set_sensitive(true);
     else
          _btnAddUser->set_sensitive(false);

     // Do GPG Checks
     // If gpg is enabled, check for a signed presence for the jid
     if (G_App->getGPG().enabled() && !sendto.empty())
     {
	  try {
	       string resource = _entResource->get_text();
	       PresenceDB::range r = G_App->getSession().presenceDB().equal_range(sendto);
	       PresenceDB::const_iterator it = r.first;
	       // We only want online resources that have the same resource as the one we are sending to
               while (it != r.second && !resource.empty() && (it->getType() == Presence::ptUnavailable || JID::getResource(it->getFrom()) != resource))
	            it++;

	       Tag* x = NULL;
	       if ((it == r.second) || !(x = it->getBaseTag().getTag("x", "jabber:x:signed")))
	       {
		    // No presence or no signature on the presence so turn off encryption
		    getWidget<Gtk::ToggleButton>("Message_Encrypt_tglbtn")->set_active(false);
	       }
	       else
	       {
		    // Have a presence with a signature.  Shoul we verify the signature here or is it enough that
		    // it exists?
		    getWidget<Gtk::ToggleButton>("Message_Encrypt_tglbtn")->set_active(true);
		    _useGPG = true;
	       }
          } catch (PresenceDB::XCP_InvalidJID& e) {
	       getWidget<Gtk::ToggleButton>("Message_Encrypt_tglbtn")->set_active(false);
          }
     }
}

void NormalMessageView::on_Quote_clicked()
{
     // Prepare the text
     string quotedtext = _nickname + _(" wrote:\n");
     quotedtext += _memIncomingBody->get_chars(0, -1);
     // from Benedikt.Roth@bratislav.de
     // eMail-like quoting: Every newline ('\n') is followed by "> "
     for(unsigned pos=0; (pos=quotedtext.find("\n", pos)) != string::npos; )
	  quotedtext.insert( ++pos, "> ");

     // For comfortable message typing make sure to start in a new line
     if (quotedtext.at(quotedtext.length()-1) != '\n' )
	  quotedtext += "\n";

     // Insert it
     int i = 0;
     _memOutgoingBody->insert_text(quotedtext.c_str(),
                                   quotedtext.length(), &i);
}

void NormalMessageView::on_AddUser_clicked()
{
     string jabberid = toUTF8(_entSendTo, _entSendTo->get_text());
     // Start AddContactDruid with that JID
     if (!jabberid.empty())
          AddContactDruid::display(JID::getUserHost(jabberid));
}

void NormalMessageView::on_EditUser_clicked()
{

     string jabberid = toUTF8(_entSendTo, _entSendTo->get_text());
     string resource = toUTF8(_entResource, _entResource->get_text());
     // Edit User Info for that JID
     if (!jabberid.empty())
     {
	  if (!resource.empty())
	       jabberid += "/" + resource;

          if (_onRoster)
               ContactInfoDlg::display(jabberid);
          else
               ContactInfoDlg::display(jabberid, Roster::rsNone);
     }
}

void NormalMessageView::on_History_clicked()
{
     if (!_entSendTo->get_text().empty())
          //HistoryDlg::display(_txtSendTo->get_text());
          gnome_url_show(G_App->getLogFile(_entSendTo->get_text()));
}

int NormalMessageView::on_GPGInfo_button_press(GdkEventButton* e)
{
     bool valid = false;

     if (_gpgInfo == GabberGPG::gpgNone)
	  return 1;
     else if (_gpgInfo == GabberGPG::gpgValidSigned || _gpgInfo == GabberGPG::gpgValidEncrypted)
	  valid = true;

     try {
          string keyid = G_App->getGPG().find_jid_key(_jid); 
	  manage(new GPGInfoDialog(keyid, valid));
     } catch (GabberGPG::GPG_InvalidJID& e) {
          manage(new GPGInfoDialog("", valid));
     }
     return 0;
}

// -------------------------------------------------------------------
//
// Chat (One-on-One) Message View
//
// -------------------------------------------------------------------
ChatMessageView::ChatMessageView(const jabberoo::Message& m, ViewMap& vm)
     : MessageView(m, vm), BaseGabberWindow("OOOChat_win"),
       _pix_path(ConfigManager::get_PIXPATH())
{
     // Basic initialization
     init();

     // If this is a reply to a normal message (it's possible),
     if (m.getType() == Message::mtNormal)
          // set send as ooochat
          _chkMessage->set_active(true);

     // Set the resource
     if (!JID::getResource(_jid).empty())
          _entResource->set_text(fromUTF8(_entResource, JID::getResource(_jid)));
     _cboResource->hide();

     // Grab the subject
     _subject = m.getSubject();

     // Grab the thread
     _thread = m.getThread();

     // Add the message
     render(m);

     // Display
     show();

     // Raise
     _lblSender->get_window().show();
     _lblSender->get_window().raise();
}

ChatMessageView::ChatMessageView(const string& jid, ViewMap& vm)
     : MessageView(jid, vm), BaseGabberWindow("OOOChat_win"),
       _pix_path(ConfigManager::get_PIXPATH())
{
     // Basic initialization
     init();

     // Set the resource
     if (!JID::getResource(_jid).empty())
          _entResource->set_text(fromUTF8(_entResource, JID::getResource(_jid)));
     list<string> resources;
     try {
	  // Walk the list of resources and add them to the combo box
	  PresenceDB::range r = G_App->getSession().presenceDB().equal_range(_jid);
	  for (PresenceDB::const_iterator it = r.first; it != r.second; it++)
	  {
	       const Presence& p = *it;

               // If this presence is a NA presence, then skip it
	       if (p.getType() == Presence::ptUnavailable)
		    continue;

               // Extract the resource
               const string& res = JID::getResource(p.getFrom());
               resources.push_back(fromUTF8(_cboResource, res));
	  }
     } catch (PresenceDB::XCP_InvalidJID& e) {
	  // No presences from any resources
     }
     if (!resources.empty())
	  _cboResource->set_popdown_strings(resources);
//     _cboResource->show();
     // FIXME: Allowing resource selection the first time the user opens a new blank ooochat
     // The only problem is that the MessageView was already created for another resource
     // And the message views are supposed to be per resource
     // So this ooochat needs to change the jid it was created for
     _cboResource->hide();

     // Display
     show();
}

ChatMessageView::~ChatMessageView()
{
     delete _chatview;
}

void ChatMessageView::init()
{
     _thisWindow->realize();

     // Lookup nickname
     _onRoster = true;
     // default is the username
     _nickname = JID::getUser(_jid);
     try {
          _nickname = G_App->getSession().roster()[JID::getUserHost(_jid)].getNickname();
     } catch (Roster::XCP_InvalidJID& e) {
          // Special handling for a groupchat ID -- use the resource as the nickname
          if (G_App->isGroupChatID(_jid))
               _nickname = JID::getResource(_jid);
          // Else try looking up their nickname in their vCard - except I don't see a need for this here... dunno
          // Clear the onRoster flag
          _onRoster = false;
     }

     // Create chat view
     _chatview = new ChatView(_thisWindow, getWidget<Gtk::HBox>("OOOChat_Chatview"), false);

     // Get widgets
     _chkMessage  = getCheckButton("OOOChat_Message_chk");
     _btnAddUser  = getButton("OOOChat_AddUser_btn");
     _lblSender   = getLabel("OOOChat_Sender_lbl");
     _memMsg      = getWidget<Gtk::Text>("OOOChat_Message_txt");
     if (gtkspell_running())
	  gtkspell_attach(_memMsg->gtkobj()); // Spell check
     _entResource = getEntry("OOOChat_Resource_txt");
     _cboResource = getWidget<Gtk::Combo>("OOOChat_Resource_cmbo");
     _cboResource ->hide();
     _pixGPG      = getWidget<Gnome::Pixmap>("OOOChat_GPG_pix");
     _evtGPG      = getWidget<Gtk::EventBox>("OOOChat_GPG_evt");
     _pixShow     = getWidget<Gnome::Pixmap>("OOOChat_Show_pix");
     _evtShow     = getWidget<Gtk::EventBox>("OOOChat_Show_evt");

     // Connect events
     _btnAddUser->clicked.connect(slot(this, &ChatMessageView::on_AddUser_clicked));
     _thisWindow->key_press_event.connect(slot(this, &ChatMessageView::on_window_key_press));
     _thisWindow->delete_event.connect(slot(this, &ChatMessageView::on_window_delete));

     G_App->getSession().evtPresence.connect(slot(this, &ChatMessageView::on_presence));
     G_App->getErrorManager().errorChat.connect(slot(this, &ChatMessageView::on_error));

//   Current implementation lacks such buttons
//     getButton("OOOChat_Send_btn")->clicked.connect(slot(this, &ChatMessageView::OnSend));
//     getButton("OOOChat_Close_btn")->clicked.connect(slot(this, &ChatMessageView::on_Close_clicked));
     getButton("OOOChat_EditUser_btn")->clicked.connect(slot(this, &ChatMessageView::on_EditUser_clicked));
     getButton("OOOChat_History_btn")->clicked.connect(slot(this, &ChatMessageView::on_History_clicked));

     // Tweak widgets
     _btnAddUser->set_sensitive(!_onRoster);
     _memMsg->set_word_wrap(true);

     getWidget<Gtk::Toolbar>("OOOChat_toolbar")->set_style(GTK_TOOLBAR_ICONS);

     // Set the sender frame, sender entry, and window title
     // If it's a transport, use the transport nickname
     string displaynick;
     // Else if they have a resource
     if (!JID::getResource(_jid).empty() && !G_App->isTransport(_jid))
          displaynick = _nickname + "/" + JID::getResource(_jid);
     // Else simply use the nickname
     else
          displaynick = _nickname;

     // Ensure it's not too long
     if (displaynick.length() > 100)
	  displaynick = displaynick.substr(0, 100) + "...";

     // Resource
     _resource = JID::getResource(_jid);

     // All of that just to set these labels
     _lblSender->set_text(displaynick);
     _thisWindow->set_title(_("Gabber: Chat with ") + _nickname);

#ifdef GABBER_WINICON
     string window_icon = string(ConfigManager::get_PIXPATH()) + "gnome-ooochat.xpm";
     gnome_window_icon_set_from_file(_thisWindow->gtkobj(),window_icon.c_str());
     gnome_window_icon_init();
#endif

     // Get user status
     get_status();

     // Add button press handler for GPG_evt
     _evtGPG->button_press_event.connect(slot(this, &ChatMessageView::on_GPGInfo_button_press));
     _gpgInfo = GabberGPG::gpgNone;

     _thisWindow->set_default_size(300, 300);
     _thisWindow->show();

}

void ChatMessageView::get_status()
{
     try {
	  // Get the presence
	  const Presence& p = *G_App->getSession().presenceDB().find(_jid);
	  
	  // Set the labels
	  string show_pix = _pix_path + p.getShow_str() + ".xpm";
	  _pixShow->load(show_pix);
	  _tips.set_tip(*_evtShow, getShowName(p.getShow()) + ": " + fromUTF8(_evtShow, p.getStatus()));
	  _evtShow->show();

	  // Turn on the Encryption toggle if gpg is enabled, otherwise disable it
	  GabberGPG& gpg = G_App->getGPG();
	  _useGPG = false;
	  if (!gpg.enabled())
	       getWidget<Gtk::ToggleButton>("OOOChat_Encrypt_tglbtn")->set_sensitive(false);
	  else
	  {
	       PresenceDB::range r = G_App->getSession().presenceDB().equal_range(_jid);
	       PresenceDB::const_iterator it = r.first;
	       // We only want online resources that have the same resource as the one we are sending to
	       while (it != r.second && !_resource.empty() && (it->getType() == Presence::ptUnavailable || JID::getResource(it->getFrom()) != _resource))  
		    it++;
	       
	       Tag* x = NULL;
	       if ((it != r.second) && (x = it->getBaseTag().getTag("x", "jabber:x:signed")))
	       {
		    // Have a presence with a signature.  Shoul we verify the signature here or is it enough that
		    // it exists?
		    getWidget<Gtk::ToggleButton>("OOOChat_Encrypt_tglbtn")->set_active(true);
		    _useGPG = true;
	       }
	  }
     } catch (PresenceDB::XCP_InvalidJID& e) {
	  // No presences from any resources
	  _evtShow->hide();
     }
}

void ChatMessageView::render(const jabberoo::Message& m)
{
     if (_cboResource->is_visible())
	  _cboResource->hide();
     // change the GPG pixmap to unlock until we have determine if the message is encrypted
     _pixGPG->load(string(ConfigManager::get_SHAREDIR()) + string("gpg-unencrypted.xpm"));
     // Set the tooltip
     _tips.set_tip(*_evtGPG, _("This message is not encrypted"));

     // Grab the subject and thread, since they may have changed
     _subject = m.getSubject();
     _thread = m.getThread();

     string message = m.getBody();
     // reset useGPG so signed messages are only sent if the received message was signed or encrypted
     _useGPG = false;
     // Check for a signature
     Tag* x = m.getBaseTag().getTag("x", "jabber:x:signed");
     GabberGPG& gpg = G_App->getGPG();
     if (x != NULL && gpg.enabled())
     {
	  string encrypted = x->getData();
	  GPGInterface::SigInfo info;
	  GPGInterface::Error err;

          if ((err = gpg.verify(info, encrypted, message)) == GPGInterface::errOK)
	  {
	       // The signature was valid so add the keyid to the KeyMap
	       gpg.add_jid_key(m.getFrom(), info.get_key().get_keyid());
	       // Change the pixmap
	       _pixGPG->load(string(ConfigManager::get_SHAREDIR()) + string("gpg-signed.xpm"));
	       // Set the tooltip
	       _tips.set_tip(*_evtGPG, _("Signature on message is valid"));
	       _gpgInfo = GabberGPG::gpgValidSigned;
	       _useGPG = true;
	  }
	  else
	  {
	       // Set the GPG pixmap to bad signature
	       _pixGPG->load(string(ConfigManager::get_SHAREDIR()) + string("gpg-badsigned.xpm"));
	       // Set the tooltip
	       _tips.set_tip(*_evtGPG, _("Signature on message is invalid"));
	       _gpgInfo = GabberGPG::gpgInvalidSigned;
	  }
     }

     // Check for encryption
     x = m.getBaseTag().getTag("x", "jabber:x:encrypted");
     if (x != NULL && gpg.enabled()) {
          string encrypted = x->getData();
          GPGInterface::DecryptInfo info;
          GPGInterface::Error err;

          if ((err = gpg.decrypt(info, encrypted, message)) != GPGInterface::errOK)
          {
               if (err == GPGInterface::errPubKey)
               {
                    cerr << "FIXME: need a way to get the user's Public Key" << endl;
               }
               return;
          }
	  else
	  {
	       // Need to figure out if it's a WinJab encrypted message and make a second pass over the
	       // decrypted text to verify the signature
	       bool winjab_sig = false;
	       if (strncmp(message.c_str(), "-----BEGIN PGP", 14) == 0)
	       {
		    // It is a winjab message so we need to run it through gpg again
		    if ((err = gpg.verify_clear((GPGInterface::SigInfo &) info.get_sig(), message, message)) != GPGInterface::errOK)
		    {
			 cerr << "FIXME: don't have user's public key" << endl;
			 return;
		    }
		    winjab_sig = true;
	       }
	       // If the message had a valid signature, add the keyid to the KeyMap
	       if ((info.has_sig() || winjab_sig) && info.get_sig().valid())
	            gpg.add_jid_key(m.getFrom(), info.get_sig().get_key().get_keyid());
	       _pixGPG->load(string(ConfigManager::get_SHAREDIR()) + string("gpg-encrypted.xpm"));
	       // Set the tooltip
	       _tips.set_tip(*_evtGPG, _("This message is encrypted"));
	       _gpgInfo = GabberGPG::gpgValidEncrypted;
	       // The message was encrypted so enable encryption for the reply
	       getWidget<Gtk::ToggleButton>("OOOChat_Encrypt_tglbtn")->set_active(true);
	       _useGPG = true;
	  }
     }

     // If they want timestamps, put them in
     if (G_App->getCfg().getBoolValue("Chats/OOOTime=true"))
          _chatview->render(fromUTF8(GTK_WIDGET(_chatview->_xtext), message), _nickname,
                            m.getDateTime(G_App->getCfg().getStrValue("Dates/ChatFormat=[%T] ")), RED);
     else
          _chatview->render(fromUTF8(GTK_WIDGET(_chatview->_xtext), message), _nickname,
                            "", RED);
}

void ChatMessageView::on_Send_clicked()
{
     string body = toUTF8(_memMsg, _memMsg->get_chars(0, -1));
     if(!body.empty())
     {
          // If "Send as Normal Message" is checked, send message with normal flag
          Message::Type mtype;
          if (_chkMessage->get_active())
               mtype = Message::mtNormal;
          else
               mtype = Message::mtChat;

          // Construct message
          Message m(_jid, body, mtype);

	  // Set the subject
	  if (!_subject.empty())
	       m.setSubject(_subject);

	  // Set the thread
	  if (!_thread.empty())
	       m.setThread(_thread);

	  // Do Encryption
	  MessageManager::Encryption enc = MessageManager::encNone;
          if (getWidget<Gtk::ToggleButton>("OOOChat_Encrypt_tglbtn")->get_active())
          {
	       // Encrypt the message
	       enc = MessageManager::encEncrypt;
          }
	  // Default to signing if GPG is enabled but we aren't encrypting
          else if (_useGPG)
          {
	       // sign the message
	       enc = MessageManager::encSign;
          }

          // Transmit the message and render it in the dialog, only if the message was sent
	  if (_cboResource->is_visible())
	       _cboResource->hide();
          bool message_sent = G_App->getMessageManager().send_message(m, enc);

	  // only render the message if it was actually sent
	  if (message_sent)
	  {
               // Render the message
               // If they want timestamps, put them in
               if (G_App->getCfg().getBoolValue("Chats/OOOTime=true"))
                    _chatview->render(fromUTF8(GTK_WIDGET(_chatview->_xtext), body), G_App->getCfg().get_nick(),
                                      m.getDateTime(G_App->getCfg().getStrValue("Dates/ChatFormat=[%T] ")), BLUE2);
               else
                    _chatview->render(fromUTF8(GTK_WIDGET(_chatview->_xtext), body), G_App->getCfg().get_nick(),
                                      "", BLUE2);
               // Reset the message box
               _memMsg->delete_text(0, -1);

	       // Update encryption pixmap
	       if (enc == MessageManager::encNone)
	       {
		    _pixGPG->load(string(ConfigManager::get_SHAREDIR()) + string("gpg-unencrypted.xpm"));
		    // set the tooltip
		    _tips.set_tip(*_evtGPG, _("This message is not encrypted"));
	       }
	       else if (enc == MessageManager::encSign)
	       {
		    _pixGPG->load(string(ConfigManager::get_SHAREDIR()) + string("gpg-signed.xpm"));
		    // set the tooltip
		    _tips.set_tip(*_evtGPG, _("Signature on message is valid"));
	       }
	       else if (enc == MessageManager::encEncrypt)
	       {
		    _pixGPG->load(string(ConfigManager::get_SHAREDIR()) + string("gpg-encrypted.xpm"));
		    // set the tooltip
		    _tips.set_tip(*_evtGPG, _("This message is encrypted"));
	       }
          }
     }
}

void ChatMessageView::on_Close_clicked()
{
     close();
}

gint ChatMessageView::on_window_delete(GdkEventAny* e)
{
     on_Close_clicked();
     return 0;
}

gint ChatMessageView::on_window_key_press(GdkEventKey* e)
{
     if (e->keyval == GDK_Return)
     {
	  //enter a newline if shift-return is used
	  if (e->state & GDK_SHIFT_MASK)
	  {
	       //unset the shift bit. shift-return seems to have a special meaning for the widget
	       e->state ^= GDK_SHIFT_MASK;
	       return 0;
	  }

	  // Ctrl-Return sends the message so we don't need to check for it  
          on_Send_clicked();
          gtk_signal_emit_stop_by_name(GTK_OBJECT(_thisWindow->gtkobj()), "key_press_event");
     }
     else if (e->keyval == GDK_space && gtkspell_running())
	  gtkspell_check_all(_memMsg->gtkobj());
     return 0;
}

void ChatMessageView::on_AddUser_clicked()
{
     // Start AddContactDruid with that JID
     AddContactDruid::display(JID::getUserHost(_jid));
}

void ChatMessageView::on_EditUser_clicked()
{
     if (_onRoster)
          ContactInfoDlg::display(_jid);
     else
          ContactInfoDlg::display(_jid, Roster::rsNone);
}

void ChatMessageView::on_History_clicked()
{
     //HistoryDlg::display(JID::getUserHost(_jid));
     gnome_url_show(G_App->getLogFile(_jid));
}

void ChatMessageView::on_presence(const Presence& p, const Presence::Type prev)
{
     // Display a notification message if this presence packet
     // is from the JID associated with this dialog
     if (p.getFrom() != _jid)
	  return;

     // If this is an error, send it to the error manager
     if (p.getBaseTag().getAttrib("type") == "error")
	  G_App->getErrorManager().add(p);

     // Set the labels
     string show_pix = _pix_path + p.getShow_str() + ".xpm";
     _pixShow->load(show_pix);
     _tips.set_tip(*_evtShow, getShowName(p.getShow()) + ": " + fromUTF8(_evtShow, p.getStatus()));
     _evtShow->show();
}

void ChatMessageView::on_error(const string& concerning, int errorcode, const string& errormsg, const string& body)
{
     // If it's not for this view, ignore
     if (concerning != _jid)
	  return;

     // Display the error
     string nick = _("Error");
     if (errorcode != 0)
     {
	  char* errnum;
	  errnum = g_strdup_printf(" %d", errorcode);
	  nick += errnum;
	  g_free(errnum);
     }
     _chatview->render_error(fromUTF8(GTK_WIDGET(_chatview->_xtext), errormsg), nick,
			     "", RED);
}

int ChatMessageView::on_GPGInfo_button_press(GdkEventButton* e)
{
     bool valid = false;

     if (_gpgInfo == GabberGPG::gpgNone)
	  return FALSE;
     else if (_gpgInfo == GabberGPG::gpgValidSigned || _gpgInfo == GabberGPG::gpgValidEncrypted)
	  valid = true;

     try {
	  string keyid = G_App->getGPG().find_jid_key(_jid);
	  manage(new GPGInfoDialog(keyid, valid));
     } catch (GabberGPG::GPG_InvalidJID& e) {
	  manage(new GPGInfoDialog("", false));
     }
     return TRUE;
}

// -------------------------------------------------------------------
//
// GC (GroupChat) Message View
//
// -------------------------------------------------------------------
GCMessageView::GCMessageView(const jabberoo::Message& m, ViewMap& vm)
     : MessageView(m, vm, true), BaseGabberWindow("GC_win"),
       _room(JID::getUserHost(m.getFrom())), _nick(JID::getResource(m.getFrom()))
{
     // Initialize the window
     init();

     // display the message
     render(m);
}

GCMessageView::GCMessageView(const string& jid, ViewMap& vm)
     : MessageView(jid, vm, true), BaseGabberWindow("GC_win"),
       _room(JID::getUserHost(jid)), _nick(JID::getResource(jid))
{
     init();
}

GCMessageView::~GCMessageView()
{
     if (G_Win && G_Win->is_connected())
     {
	  // Notify GC server that we're ready to leave
	  G_App->getSession() << Presence(_room + "/" + _nick, Presence::ptUnavailable);
     }

     delete _chatview;
}

void GCMessageView::init()
{
     _thisWindow->realize();

     // Setup pointers
     _memMessage = getWidget<Gtk::Text>("GC_Message_txt");
     if (gtkspell_running())
	  gtkspell_attach(_memMessage->gtkobj());
     _entSubject = getEntry("GC_Subject_txt");
     _lstUsers   = getWidget<Gtk::CList>("GC_Users_list");
     _optShow    = getWidget<Gtk::OptionMenu>("GC_Show_opt");
     _btnHistory = getButton("GC_History_btn");
//     _btnClose   = getButton("GC_Close_btn");
     _tglUsers   = getWidget<Gtk::ToggleButton>("GC_Users_toggle");
     _hpane      = getWidget<Gtk::Paned>("GC_hpane");
     _lblUserCount = getLabel("GC_MemberNum_lbl");
     _baseGCU    = manage(new BaseGabberWidget("GCUser_menu", "GC_win"));
     _menuGCUser = static_cast<Gtk::Menu*>(_baseGCU->get_this_widget());

     bool indent = true;
     // If they want timestamps, disable indentation
     if (G_App->getCfg().getBoolValue("Chats/GroupTime=false"))
          indent = false;

     // Initialize chat view
     _chatview = new ChatView(_thisWindow, getWidget<Gtk::HBox>("GC_Chatview"), indent);

     // Setup window handler
     _thisWindow->delete_event.connect(slot(this, &GCMessageView::on_window_delete));

     // Setup event handlers
     _memMessage->key_press_event.connect(slot(this, &GCMessageView::on_Message_key_press));
     _entSubject->key_press_event.connect(slot(this, &GCMessageView::on_Subject_key_press));
     _lstUsers->button_press_event.connect(slot(this, &GCMessageView::on_Roster_button_press));
     _btnHistory->clicked.connect(slot(this, &GCMessageView::on_History_clicked));
//     _btnClose->clicked.connect(slot(this, &GCMessageView::on_Close_clicked));
     _tglUsers->toggled.connect(slot(this, &GCMessageView::on_Users_toggled));
     getButton("GC_Invite_btn")->clicked.connect(slot(this, &GCMessageView::on_Invite_clicked));
     // Right-click menu
     _baseGCU->getMenuItem("GCUser_Message_item")->activate.connect(slot(this, &GCMessageView::on_Message_activate));
     _baseGCU->getMenuItem("GCUser_OOOChat_item")->activate.connect(slot(this, &GCMessageView::on_OOOChat_activate));
     _baseGCU->getMenuItem("GCUser_SendContacts_item")->activate.connect(slot(this, &GCMessageView::on_SendContacts_activate));
     _baseGCU->getMenuItem("GCUser_ViewInfo_item")->activate.connect(slot(this, &GCMessageView::on_ViewInfo_activate));

     // Setup the default pane settings
     _posHpane = 360;
     _tglUsers->set_active(true);
     _hpane->set_position(360);

     // Connect to the session disconnect event
     G_App->getSession().evtDisconnected.connect(slot(this, &GCMessageView::on_session_disconnected));
     G_App->getSession().evtPresence.connect(slot(this, &GCMessageView::on_session_presence));
     G_Win->evtMyPresence.connect(slot(this, &GCMessageView::on_presence_changed));
     G_App->getErrorManager().errorGroupchat.connect(slot(this, &GCMessageView::on_error));

     // Setup option menu
//FIXME: See GCMessageView::on_Show_selected
//     _menuShow.add_presence_items();
//     _menuShow.finish_items();
//     _menuShow.selected.connect(slot(this, &GCMessageView::on_Show_selected));
//     _optShow->set_menu(_menuShow.menu);
//     _optShow->set_history(0);

     // We want icon-only toolbar for now
     getWidget<Gtk::Toolbar>("GC_toolbar")->set_style(GTK_TOOLBAR_ICONS);

     // Set the window title and frame title
     string roomname = JID::getUser(_room);
     string fullname = JID::getUser(_room);
     string::size_type percent = fullname.find('%');
     if (percent != string::npos)
          fullname = roomname.substr(0, percent) + " on " + roomname.substr(percent + 1);
     else
          fullname = roomname + " on " + JID::getHost(_room);
     _thisWindow->set_title(_("Gabber: Group Chat in ") + roomname);
     Gtk::Label* l = getLabel("GC_Group_lbl");
     l->set_text(fromUTF8(l, (fullname)));

     // Pixmaps
     string pix_path = ConfigManager::get_PIXPATH();
#ifdef GABBER_WINICON
     string window_icon = pix_path + "gnome-groupchat.xpm";
     gnome_window_icon_set_from_file(_thisWindow->gtkobj(),window_icon.c_str());
     gnome_window_icon_init();
#endif

     // Word wrap
     _memMessage->set_word_wrap(true);

     // Ensure roster resizes properly, does auto-sort and has a proper
     // column height
     _lstUsers->set_column_auto_resize(0, true);
     _lstUsers->set_row_height(18);
     _lstUsers->set_compare_func(&GabberUtil::strcasecmp_clist_items);

     // Setup tab-completion buffer
     _completer = g_completion_new(NULL);

     // Join room on the server
     G_App->getSession() << Presence(_room + "/" + _nick, Presence::ptAvailable, G_App->getCfg().get_show(), G_App->getCfg().get_status());

     // Display
     _thisWindow->set_default_size(500, 300);
     _thisWindow->show();
}

void GCMessageView::render(const jabberoo::Message& m)
{
     // Convert from UTF8
     string fromnick = fromUTF8(GTK_WIDGET(_chatview->_xtext), JID::getResource(m.getFrom()));
     string body = fromUTF8(GTK_WIDGET(_chatview->_xtext), m.getBody());

     // A message from myself
     if (fromnick == fromUTF8(GTK_WIDGET(_chatview->_xtext), _nick))
     {
          // If they want timestamps, put them in
          if (G_App->getCfg().getBoolValue("Chats/GroupTime=false"))
               _chatview->render(body, fromnick, m.getDateTime(G_App->getCfg().getStrValue("Dates/GCFormat=[%T] ")), BLUE2);
          else
               _chatview->render(body, fromnick, "", BLUE2);
     }
     // A system message
     else if (fromnick.empty())
     {
          // If they want timestamps, put them in
          if (G_App->getCfg().getBoolValue("Chats/GroupTime=false"))
               _chatview->render(body, fromnick, m.getDateTime(G_App->getCfg().getStrValue("Dates/GCFormat=[%T] ")), AQUA);
          else
               _chatview->render(body, fromnick, "", AQUA);
     }
     // A message from someone else
     else
     {
          // If they want timestamps, put them in
          if (G_App->getCfg().getBoolValue("Chats/GroupTime=false"))
               _chatview->render(body, fromnick, m.getDateTime(G_App->getCfg().getStrValue("Dates/GCFormat=[%T] ")), RED);
          else
               _chatview->render(body, fromnick, "", RED);
     }

     const string& subj = m.getSubject();
     if (!subj.empty())
     {
        _thisWindow->set_title(_("Gabber: Group Chat - ") + subj);
        _entSubject->set_text(fromUTF8(_entSubject, subj));
     }
}

gint GCMessageView::on_window_delete(GdkEventAny* e)
{
     close();
     return 0;
}

gint GCMessageView::on_Subject_key_press(GdkEventKey* e)
{
     if (e->keyval == GDK_Return)
     {
          string subject = toUTF8(_entSubject, _entSubject->get_text());
          if (!subject.empty())
          {
               // Setup a message object
               Message m(_room, _("/me has changed the subject to: ") + subject, Message::mtGroupchat);
               m.setSubject(subject);
               // Transmit the subject change
               G_App->getSession() << m;
          }
     }
     return 0;
}

int GCMessageView::on_Roster_button_press(GdkEventButton* e)
{
     if (!_lstUsers->get_selection_info(e->x, e->y, &_row, &_col))
	  return 0;

     _lstUsers->row(_row).select();

     switch (e->type) {
     case GDK_2BUTTON_PRESS: // Double-click
     {
          // Check for control key..
          GdkEventButton* ex = (GdkEventButton*)e;
          if (ex->state & GDK_CONTROL_MASK)
               G_App->getMessageManager().display(_room + "/" + _lstUsers->cell(_row, _col).get_text(), MessageManager::mtChat);
          else
	  {
	       if (G_App->getCfg().getBoolValue("Messages/SendMsgs=true"))
		    G_App->getMessageManager().display(_room + "/" + _lstUsers->cell(_row, _col).get_text(), MessageManager::mtNormal);
	       else
		    G_App->getMessageManager().display(_room + "/" + _lstUsers->cell(_row, _col).get_text(), MessageManager::mtChat);
	  }
	  break;
     }
     case GDK_BUTTON_PRESS: // Single-click
	  // Check for rt-click
	  if (e->button == 3) 
	  {
	       _menuGCUser->show_all();
	       _menuGCUser->popup(e->button, e->time);
	  }
	  break;
     default:
	  break;
     }

     return 0;
}

void GCMessageView::on_Message_activate()
{
     string gcuser = toUTF8(_lstUsers, _lstUsers->cell(_row, _col).get_text());
     G_App->getMessageManager().display(_room + "/" + gcuser, MessageManager::mtNormal);
}

void GCMessageView::on_OOOChat_activate()
{
     string gcuser = toUTF8(_lstUsers, _lstUsers->cell(_row, _col).get_text());
     G_App->getMessageManager().display(_room + "/" + gcuser, MessageManager::mtChat);
}

void GCMessageView::on_SendContacts_activate()
{
     string gcuser = toUTF8(_lstUsers, _lstUsers->cell(_row, _col).get_text());
     manage(new ContactSendDlg(_room + "/" + gcuser));
}

void GCMessageView::on_ViewInfo_activate()
{
     string gcuser = toUTF8(_lstUsers, _lstUsers->cell(_row, _col).get_text());
     ContactInfoDlg::display(_room + "/" + gcuser, Roster::rsNone);
}

void GCMessageView::on_History_clicked()
{
     //HistoryDlg::display(_room);
     gnome_url_show(G_App->getLogFile(_room));
}

void GCMessageView::on_Close_clicked()
{
     close();
}

void GCMessageView::on_Users_toggled()
{
     // If they want to see the users, set it back to the old position
     if (_tglUsers->get_active())
          _hpane->set_position(_posHpane);
     else
     {
          // if they don't, remember old position, and set to a big number. xchat uses 1200
          _posHpane = _hpane->gtkobj()->handle_xpos;
          _hpane->set_position(1200);
     }
}

gint GCMessageView::on_Message_key_press(GdkEventKey* e)
{
     if (e->keyval == GDK_Return)
     {
	  if (e->state & GDK_SHIFT_MASK)
	  {
	       e->state ^= GDK_SHIFT_MASK;
	       return 0;
	  }

          string msg = toUTF8(_memMessage, _memMessage->get_chars(0, -1));
          if ( (!msg.empty()) && (!process_command(msg)) )
          {
               // Transmit the message
               G_App->getSession() << Message(_room, msg, Message::mtGroupchat);
          }
          // Clear display
          _memMessage->delete_text(0, -1);
          // HACK --  to stop the signal from continuing, and the return from being left over in the message
          gtk_signal_emit_stop_by_name(GTK_OBJECT(_memMessage->gtkobj()), "key_press_event");
     }
     // Handle tab-completion
     else if (e->keyval == GDK_Tab)
     {
          string msg = toUTF8(_memMessage, _memMessage->get_chars(0, -1));
          // Look for the last word
          if (!msg.empty())
          {
               string lastword;
               // Search for the last whitespace
               string::size_type n = msg.find_last_of(" ");
               if (n != string::npos)
                    lastword = msg.substr(n+1);
               else
                    lastword = msg;

               // Try and autocomplete
               gchar* prefix;
               g_completion_complete(_completer, (char*)lastword.c_str(), &prefix);
               if (prefix != NULL)
               {
                    // Replace the last word in the message and update
                    if (n != string::npos)
                         msg.replace(msg.find_last_of(" ")+1, msg.length(), prefix);
                    else
                         msg = string(prefix);
                    g_free(prefix);

                    // Update text
                    _memMessage->delete_text(0, -1);
                    _memMessage->insert(fromUTF8(_memMessage, msg));
               }
               // HACK --  to stop the signal from continuing, and the entry from losing focus
               gtk_signal_emit_stop_by_name (GTK_OBJECT(_memMessage->gtkobj()), "key_press_event");
               return 1;
          }
     }
     else if (e->keyval == GDK_space && gtkspell_running())
	  gtkspell_check_all(_memMessage->gtkobj());
     return 0;
}

void GCMessageView::on_session_disconnected()
{
     // Notify user that the session is disconnected and disable the
     // input
     _memMessage->set_sensitive(false);
     _chatview->render(_("Session Disconnected"), "", "", YELLOW);
}

void GCMessageView::on_session_presence(const Presence& p, const Presence::Type prev)
{
     // If this packet isn't for this room, ignore it
     if (JID::getUserHost(p.getFrom()) != _room)
          return;

     // If this is an error, send it to the error manager
     if (p.getBaseTag().getAttrib("type") == "error")
	  G_App->getErrorManager().add(p);

     // Freeze the roster list
     _lstUsers->freeze();
     _lstUsers->clear();

     // Get a ref to the config manager
     ConfigManager& cfgm = G_App->getCfg();

     // Clear the completion list
     g_completion_clear_items(_completer);

     // Setup a G_List to hold all user strings
     GList* l = NULL;

     // Define a counter to hold # of users
     int usercount = 0;

     // Grab iterators from the presencedb for this room jid
     PresenceDB::range r = G_App->getSession().presenceDB().equal_range(_room);
     for (PresenceDB::const_iterator it = r.first; it != r.second; it++, usercount++)
     {
          // Grab presence reference
          const Presence& p = *it;

          // If this presence is a NA presence, then skip it
          if (p.getType() == Presence::ptUnavailable)
               continue;

          // Create a row in the list
          const char* data[1] = {""};
          int row = _lstUsers->append(data);

          // Extract the resource
          const string& s = JID::getResource(p.getFrom());

          // Create an entry in the roster
          gtk_clist_set_foreground(_lstUsers->gtkobj(), row, cfgm.getPresenceColor(p.getShow_str()));
          gtk_clist_set_pixtext(_lstUsers->gtkobj(), row, 0, fromUTF8(_lstUsers, s).c_str(), 5,
                                cfgm.getPresencePixmap(p.getShow_str()),
                                cfgm.getPresenceBitmap(p.getShow_str()));

          // Add an entry to the completion list
          l = g_list_append(l, g_strdup(s.c_str()));
     }

     // Sort the list
     _lstUsers->sort();

     // Update user count widget
     char ucount_str[15];
     g_snprintf((char*)&ucount_str, 15, _("%d members"), usercount);
     _lblUserCount->set_text((char*)&ucount_str);

     // Add /commands into the completer list
     l = g_list_append(l, g_strdup("/nick"));
     l = g_list_append(l, g_strdup("/quit"));
     l = g_list_append(l, g_strdup("/query"));
     l = g_list_append(l, g_strdup("/msg"));
     l = g_list_append(l, g_strdup("/clear"));
     l = g_list_append(l, g_strdup("/help"));

     // Update completer
     g_completion_add_items(_completer, l);

     _lstUsers->thaw();
}

void GCMessageView::on_presence_changed(Presence::Show show, const string& status, int priority)
{
     _current_show = show;
     if (G_Win && G_Win->is_connected())
     {
	  G_App->getSession() << Presence(_room + "/" + _nick, Presence::ptAvailable, _current_show, status); 
     }
}

void GCMessageView::on_error(const string& concerning, int errorcode, const string& errormsg, const string& body)
{
     // If it's not for this view, ignore
     if (concerning != JID::getUserHost(_jid))
	  return;

     // Display the error
     string nick = _("Error");
     if (errorcode != 0)
     {
	  char* errnum;
	  errnum = g_strdup_printf(" %d", errorcode);
	  nick += errnum;
	  g_free(errnum);
     }
     _chatview->render_error(fromUTF8(GTK_WIDGET(_chatview->_xtext), errormsg), nick,
			     "", RED);
}

bool GCMessageView::process_command(const string& cmd)
{
     // Dummy check..
     if (cmd.at(0) != '/')
          return false;

     // NICK handler
     if ((cmd.substr(0,6) == "/nick ") && (cmd.length() > 6))
     {
          string newnick = cmd.substr(6);
          if (!newnick.empty())
          {
               _nick = newnick;
               G_App->getSession() << Presence(_room + "/" + _nick, Presence::ptAvailable, _current_show);
          }
          return true;
     }
     // QUERY handler
     else if ((cmd.substr(0,6) == "/query") && (cmd.length() > 7))
     {
          string user = cmd.substr(7);
          if (!user.empty())
               G_App->getMessageManager().display(_room + "/" + user, MessageManager::mtChat);
          return true;
     }
     // MSG handler
     else if ((cmd.substr(0,4) == "/msg") && (cmd.length() > 5))
     {
          string user = cmd.substr(5);
          if (!user.empty())
               G_App->getMessageManager().display(_room + "/" + user, MessageManager::mtNormal);
          return true;
     }
     // QUIT handler
     else if (cmd.substr(0,5) == "/quit")
     {
          close();
          return true;
     }
     // HELP handler
     else if (cmd.substr(0,5) == "/help")
     {
	  GnomeHelpMenuEntry help_entry = { "gabber", "msg.html#MSG-GC-MAIN" };
	  gnome_help_display (NULL, &help_entry);
	  return true;
     }
     // CLEAR handler
     else if (cmd.substr(0,6) == "/clear")
     {
	  _chatview->clearbuffer();
	  return true;
     }
     return false;
}

void GCMessageView::on_Show_selected(int show_index)
{
     // FIXME: See GabberWin::on_Show_selected and make this work like that
     // Set the global _current_show
     //_current_show = indexShow(show_index);

     // Output new presence for this jid
     //G_App->getSession() << Presence(_room + "/" + _nick, Presence::ptAvailable, _current_show);
}

void GCMessageView::join(const string& jid)
{
     // Case 1: No jid is provided -- exit the procedure
     if (jid.empty())
          return;
     // Case 2: JID is provided, but no window can be found -- create a new window
     else if (!G_App->getMessageManager().hasView(JID::getUserHost(jid), MessageManager::mtGroupchat))
          G_App->getMessageManager().display(jid, MessageManager::mtGroupchat);
     // Case 3: JID is provided and a window was found -- send new presence packet
     else
          G_App->getSession() << Presence(jid, Presence::ptAvailable);
}

void GCMessageView::on_Invite_clicked()
{
     manage(new GCIDlg(_room, toUTF8(_entSubject, _entSubject->get_text())));
}

// -------------------------------------------------------------------
//
// Join Group Chat Dialog
//
// -------------------------------------------------------------------

void GCJoinDlg::execute()
{
     manage(new GCJoinDlg());
}

GCJoinDlg::GCJoinDlg()
     : BaseGabberDialog("GCJoin_dlg")
{
     // Setup buttons
     getButton("GCJoin_OK_btn")->clicked.connect(slot(this, &GCJoinDlg::on_ok_clicked));
     getButton("GCJoin_Cancel_btn")->clicked.connect(slot(this, &GCJoinDlg::on_cancel_clicked));
     
     // Connect widget(s)
     _optProtocol = getWidget<Gtk::OptionMenu>("GCJoin_Protocol_opt");
     _thisWindow->key_press_event.connect(slot(this, &GCJoinDlg::on_key_pressed));
     getEntry("GCJoin_Nick_txt")->changed.connect(slot(this, &GCJoinDlg::changed));
     getEntry("GCJoin_Room_txt")->changed.connect(slot(this, &GCJoinDlg::changed));
     getEntry("GCJoin_Server_txt")->changed.connect(slot(this, &GCJoinDlg::changed));
     _gentNick = getWidget<Gnome::Entry>("GCJoin_Nick_gent");
     _gentRoom = getWidget<Gnome::Entry>("GCJoin_Room_gent");
     _gentServer = getWidget<Gnome::Entry>("GCJoin_Server_gent");

     // Grab the history and put it in the gnome entries (a cooler combo box)
     G_App->getCfg().loadEntryHistory(_gentNick);
     G_App->getCfg().loadEntryHistory(_gentRoom);
     G_App->getCfg().loadEntryHistory(_gentServer);
     G_App->getCfg().loadEntryHistory(getWidget<Gnome::Entry>("GCJoin_IRCServer_gent"));

     // Set the default nick
     getEntry("GCJoin_Nick_txt")->set_text(G_App->getCfg().get_nick());

     // Let's give them some rooms to join and default it to conference.jabber.org - for now
     _gentRoom->prepend_history(false, "jabber");
     _gentRoom->prepend_history(false, "jdev");
     getEntry("GCJoin_Server_txt")->set_text("conference.jabber.org");

     // Setup the option menu
     _protocol = 1;
     _menuProtocol.add_item(_("Jabber Group Chat 1.0"), 1);
     _menuProtocol.add_item(_("IRC"), 2);
     _menuProtocol.finish_items();
     _menuProtocol.selected.connect(slot(this, &GCJoinDlg::on_protocol_selected));
     _optProtocol->set_menu(_menuProtocol.get_menu());
     _optProtocol->set_history(0);

     // Pixmaps
     string pix_path = ConfigManager::get_PIXPATH();
#ifdef GABBER_WINICON
     string window_icon = pix_path + "gnome-groupchat.xpm";
     gnome_window_icon_set_from_file(_thisWindow->gtkobj(),window_icon.c_str());
     gnome_window_icon_init();
#endif

     // Poll the agents of this server and add the list of groupchat agents
     G_App->getSession().queryNamespace("jabber:iq:agents", slot(this, &GCJoinDlg::on_agents_reply), G_App->getCfg().getStrValue("Server/Server=jabber.com"));

     show();
}

void GCJoinDlg::on_ok_clicked()
{
     // Get values from UI
     Gtk::Entry* e = getEntry("GCJoin_Nick_txt");
     string handle = toUTF8(e, e->get_text());
     e = getEntry("GCJoin_Room_txt");
     string room   = toUTF8(e, e->get_text());
     e = getEntry("GCJoin_Server_txt");
     string server = toUTF8(e, e->get_text());
     e = getEntry("GCJoin_IRCServer_txt");
     string ircserver = toUTF8(e, e->get_text());

     // Transmit presence 
     string id;
     // If they're using IRC, set the irc server
     if (_protocol == 2 && !ircserver.empty())
	  id = room + "%" + ircserver + "@" + server + "/" + handle;
     else
	  id = room + "@" + server + "/" + handle;
     GCMessageView::join(id);
     // Save the gnome entry history
     G_App->getCfg().saveEntryHistory(_gentNick);
     G_App->getCfg().saveEntryHistory(_gentRoom);
     G_App->getCfg().saveEntryHistory(_gentServer);
     G_App->getCfg().saveEntryHistory(getWidget<Gnome::Entry>("GCJoin_IRCServer_gent"));
     close();
}

void GCJoinDlg::on_cancel_clicked()
{
     close();
}

gint GCJoinDlg::on_key_pressed(GdkEventKey* e)
{
     // If Enter is pressed, attempt to continue
     switch (e->keyval)
     {
     case GDK_Return:
	  on_ok_clicked();
	  break;
     case GDK_Escape:
	  on_cancel_clicked();
     }
     return FALSE;
}     

void GCJoinDlg::on_protocol_selected(int protocol)
{
     _protocol = protocol;
     switch (protocol)
     {
     case 1:
	  // Jabber
	  getWidget<Gtk::HBox>("GCJoin_IRCServer_hbox")->set_sensitive(false);
	  break;
     case 2:
	  getWidget<Gtk::HBox>("GCJoin_IRCServer_hbox")->set_sensitive(true);
	  break;
     }
}

void GCJoinDlg::changed()
{
     // If they left something out, then disable the OK button
     if (getEntry("GCJoin_Room_txt")->get_text().empty()
	 || getEntry("GCJoin_Server_txt")->get_text().empty()
	 || getEntry("GCJoin_Nick_txt")->get_text().empty())
	  getButton("GCJoin_OK_btn")->set_sensitive(false);
     else
	  getButton("GCJoin_OK_btn")->set_sensitive(true);
}

void GCJoinDlg::on_agents_reply(const Tag& iq)
{
     Tag* query = iq.getTag("query");
     ElementList::const_iterator it = query->getChildren().begin();

     if (iq.cmpAttrib("type", "result")) 
     {
          for (; it != query->getChildren().end(); it++)
          {
               Tag& agent = *static_cast<Tag*>(*it);

	       if (agent.getName() != "agent")
                    continue;

               string jid = agent.getAttrib("jid");
	       Tag* groupchat = agent.getTag("groupchat");

	       // Only add agents that have a groupchat tag
	       if (groupchat)
	       {
	            _gentServer->prepend_history(false, jid);
	       }
          }
     }
}


// -------------------------------------------------------------------
//
// Autoupdate Dialog
//
// -------------------------------------------------------------------
AutoupdateDlg::AutoupdateDlg(const jabberoo::Message& m)
     : BaseGabberDialog("Autoupdate_dlg")
{
     // Hook up the buttons
     getButton("Autoupdate_OK_btn")->clicked.connect(slot(this, &AutoupdateDlg::on_OK_clicked));
     _thisWindow->delete_event.connect(slot(this, &AutoupdateDlg::on_window_delete));


     // This is an autoupdate message
     // Get next session ID
     string id = G_App->getSession().getNextID();
     // Construct update info request
     Tag iq("iq");
     iq.putAttrib("id", id);
     string autoupdateJID = "956878967";             // Gabber's clientID on jabbercentral
     autoupdateJID += "@update.jabber.org";         // the only place to grab updates right now
     iq.putAttrib("to", autoupdateJID);
     iq.putAttrib("type", "get");
     Tag& query = iq.addTag("query");
     query.putAttrib("xmlns", "jabber:iq:autoupdate");
     // Send the update info request
     G_App->getSession() << iq.getXML().c_str();
     G_App->getSession().registerIQ(id, slot(this, &AutoupdateDlg::get_update_info));
}

void AutoupdateDlg::get_update_info(const Tag& t)
{
     if (t.hasChildren())
     {
          Tag* query = t.getTag("query");
          if (query != NULL)
          {
               Tag* release = query->getTag("release");
               string curtext;
               curtext = release->getTaggedCDATA("version");
               Gtk::Label* l = getLabel("Autoupdate_UpdateVer_lbl");
               l->set_text(fromUTF8(l, ("Gabber " + curtext)));
               curtext = release->getTaggedCDATA("desc");
               l = getLabel("Autoupdate_UpdateDesc_lbl");
               l->set_text(fromUTF8(l, curtext));
	       curtext = release->getTaggedCDATA("url");
	       Gnome::HRef* b = getWidget<Gnome::HRef>("Autoupdate_UpdateURI_href");
	       b->set_url(fromUTF8(b, curtext));
	       b->set_label(fromUTF8(b, curtext));
          }
     }
}

void AutoupdateDlg::on_OK_clicked()
{
     close();
}

gint AutoupdateDlg::on_window_delete(GdkEventAny* e)
{
     close();
     return 0;
}
