/*
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 * 
 *  Gabber
 *  Copyright (C) 1999-2001 Dave Smith & Julian Missig
 */

#include "GabberApp.hh"
#include "GabberWin.hh"
#include "MessageManager.hh"
#include "MessageViews.hh"
#include "GabberLogger.hh"
#include "ContactInterface.hh"
#include "GCInterface.hh"

#include <string>
#include <sys/types.h>
#include <dirent.h>

using namespace jabberoo;

const string MessageManager::XML_FOOTER = "</spool>";

MessageManager::MessageManager(const string& spooldir)
     : _spool(spooldir)
{
     _mlistmap = new map<string, MList> [numTypes]; 
     _mspools = new map<string, ofstream*> [numTypes];
     _mviews = new ViewMap[numTypes];

     initdir();

     // Check spool dir for existing messages; if there are
     // message, load them up
     DIR* spool = opendir(_spool.c_str()); 
     if (spool != NULL)
     {
	  struct dirent* entry = readdir(spool);
	  while (entry != NULL)
	  {
               string filename = entry->d_name;
               if ((filename != ".") && (filename != ".."))
               {
	            string fullname = g_concat_dir_and_file(_spool.c_str(), entry->d_name);
	            MessageLoader(fullname, *this);
               }
	       entry = readdir(spool);
	  }
	  closedir(spool);
     }
}

MessageManager::~MessageManager()
{
     for (unsigned int i = 0; i < numTypes; i++)
     {
          map<string, ofstream*>::iterator mit = _mspools[i].begin();

          // clean up the ofstreams
          while (mit != _mspools[i].end())
          {
	       mit->second->close();
               delete mit->second;
	       mit++;
          }
     }
     delete [] _mlistmap;
     delete [] _mspools;
     delete [] _mviews;
}

void MessageManager::add(const jabberoo::Message& m, bool spool)
{
     ConfigManager& cfgm = G_App->getCfg();
     // Figure out if the user has selected to only received OOOChats Or Messages
     MessageType mtype = translateType(m);
     string user = (mtype == mtGroupchat) ? JID::getUserHost(m.getFrom()) : m.getFrom();

     if (mtype == mtGroupchat)
	  ;
     else if (mtype == mtError)
	  mtype = mtNormal; // It was passed on to message manager, so it's probably a normal message
     else if (cfgm.getBoolValue("Messages/RecvMsgs=false") && (mtype == MessageManager::mtChat))
          mtype = mtNormal;
     else if (cfgm.getBoolValue("Messages/RecvOOOChats=false") && (mtype == MessageManager::mtNormal))
          mtype = mtChat;

     // Select a view based on the message type
     ViewMap& vm = _mviews[mtype];

     // Attempt to locate a view for this JID
     ViewMap::iterator it = vm.find(user);
     // If a view is found....
     if (it != vm.end())
     {
	  if (m.getType() == Message::mtNormal)
	       // Play RecvMessage sound
	       gnome_triggers_do(NULL, NULL, "gabber", "RecvMessage", NULL);
	  else if (m.getType() == Message::mtChat)
	       // Play RecvOOOChat sound
	       gnome_triggers_do(NULL, NULL, "gabber", "RecvOOOChat", NULL);
	  // Pass the message on to the view
	  it->second->render(m);
     }
     // No view was found, so stick this message
     // in a message list
     else
     {
	  // Lookup the message list for this JID and Message Type
	  MList& mlist = _mlistmap[mtype][user];
	  // Add the message
	  mlist.push_back(m);
	  // If this message should be spooled, do so
	  if (spool)
	  {
               streampos pos;
	       map<string, ofstream*>::iterator it = _mspools[mtype].find(user);

	       if (it == _mspools[mtype].end())
               {
		    string file = user + "-" + translateType(mtype);
		    if (file.find("/") != string::npos)
			 file.replace(file.find("/"), 1, "-");
                    char *filename = g_concat_dir_and_file(_spool.c_str(), file.c_str());
                    ofstream* spool = new ofstream(filename, ios::out | ios::ate);
                    pair<map<string, ofstream*>::iterator, bool> np = _mspools[mtype].insert(make_pair(user, spool));
		    it = np.first;

                    spool->seekp(0, ios::end);
                    // If file is empty output proper XML header
                    if (spool->tellp() == 0)
                    {
                         *spool << "<?xml version=\"1.0\" encoding=\"UTF-8\"?>" << endl;
                         *spool << "<spool xmlns=\"http://gabber.sourceforge.net/namespace/spool\">" << endl;

                         pos = spool->tellp();
                         *spool << XML_FOOTER;
                         spool->seekp(pos);
                    }
                    else
                         spool->seekp(spool->tellp() - XML_FOOTER.size());
               }
	       Message logm(m);

	       // If the message has no timestamp add one so the correct time is displayed
	       Tag* x = logm.getBaseTag().getTag("x", "jabber:x:delay");
	       if (!x)
	       {
	            time_t curtime = time(0);
	            struct tm* t = gmtime(&curtime);
	            t->tm_year += 1900;
	            t->tm_mon += 1;
	            char* time = g_strdup_printf("%04d%02d%02dT%02d:%02d:%02d", t->tm_year, t->tm_mon, t->tm_mday,
		 	       		         t->tm_hour, t->tm_min, t->tm_sec);

		    Tag& rx = logm.getBaseTag().addTag("x");
		    rx.putAttrib("xmlns", "jabber:x:delay");
		    rx.putAttrib("stamp", time);

		    g_free(time);
	       }

	       *it->second << logm.toXML();

               pos = it->second->tellp();
               *it->second << XML_FOOTER;
               it->second->seekp(pos);

               it->second->flush();
	  }

          // Add Event
          switch (mtype)
          {
          case mtNormal:
	       // Play QueueMessage sound
	       gnome_triggers_do(NULL, NULL, "gabber", "QueueMessage", NULL);
	       _events.push_back(make_pair(user, mtNormal));
               break;
          case mtChat:
	       // Play QueueOOOChat sound
	       gnome_triggers_do(NULL, NULL, "gabber", "QueueOOOChat", NULL);
	       _events.push_back(make_pair(user, mtChat));
               break;
	  case mtRoster:
	       // Sound??
	       _events.push_back(make_pair(user, mtRoster));
	       break;
	  case mtGCI:
	       _events.push_back(make_pair(user, mtGCI));
	       break;
          default:
               break;
          }
     }
}

MessageView* MessageManager::display(const string jid, const MessageType type)
{
     bool view_displayed = false;
     MessageView *view = NULL;
     string user = (type == mtGroupchat) ? JID::getUserHost(jid) : jid;

     // Locate the message list for this JID
     map<string, MList>::iterator it = _mlistmap[type].find(user);
     // If we found a message list, walk the list, rendering
     // each message
     if (it != _mlistmap[type].end())
     {
	  MList& mlist = it->second;
	  for (MList::iterator mit = mlist.begin(); mit != mlist.end(); mit++)
	       view = render(*mit);

	  if (mlist.begin() != mlist.end())
	       view_displayed = true;

	  _mlistmap[type].erase(it);
     }
     clearSpool(jid, type);
     clearEvents(jid, type);

     // make sure a View of the requested type is displayed. 
     if (!view_displayed)
          view = render(jid, type);
     return view;
}

MessageView* MessageManager::render(const string& jid, const MessageType type)
{
     ViewMap& vm = _mviews[type];
     string user = (type == mtGroupchat) ? JID::getUserHost(jid) : jid;
     ViewMap::iterator it = vm.find(user);
     MessageView* view = NULL;

     if (it == vm.end())
     {
          // Only open a new view if one doesn't exist already
	  switch (type)
	  {
  	  case mtNormal:
  	       view = manage(new NormalMessageView(jid, vm));
	       break;
	  case mtChat:
	       view = manage(new ChatMessageView(jid, vm));
	       break;
          case mtGroupchat:
	       view = manage(new GCMessageView(jid, vm));
	       break;
	  default:
	       break;
	  }
     }
     else
	  view = it->second;
     return view;
}

MessageView* MessageManager::render(const jabberoo::Message& m)
{
     MessageView *view = NULL;
     ConfigManager& cfgm = G_App->getCfg();
     // Figure out if the user has selected to only received OOOChats Or Messages
     MessageType render_type = translateType(m);

     if (render_type == mtGroupchat)
	  ;
     else if (render_type == mtError)
	  render_type = mtNormal; // It was passed on to message manager, so it's probably a normal message
     if (cfgm.getBoolValue("Messages/RecvMsgs=false") && (render_type == MessageManager::mtChat) )
          render_type = mtNormal;
     else if (cfgm.getBoolValue("Messages/RecvOOOChats=false") && (render_type == MessageManager::mtNormal))
          render_type = mtChat;

     // Get a view map for the message type we are using to render the message
     ViewMap& vm = _mviews[render_type];
     string user = (render_type == mtGroupchat) ? JID::getUserHost(m.getFrom()) : m.getFrom();
     // Attempt to locate an existing view..
     ViewMap::iterator it = vm.find(user);

     // If an existing view was found, use it to render the message...
     if (it != vm.end())
     {
	  view = it->second;
	  it->second->render(m);
     }
     // If no view was found, create a new based on the message type
     // - Also, the MessageView is given a reference to the ViewMap
     // - so that it can register it self with it and unregister itself
     // - on destruction
     else
     {	  
	  switch (render_type)
	  {
	  case mtNormal:
	       view = manage(new NormalMessageView(m, vm));
               break;
	  case mtChat:
	       view = manage(new ChatMessageView(m, vm));
               break;
	  case mtGroupchat:
	       view = manage(new GCMessageView(m, vm));
               break;
	  case mtHeadline:
	       view = manage(new ChatMessageView(m, vm));
	       break;
	  case mtRoster:
	       view = manage(new ContactRecvView(m, vm));
	       break;
	  case mtGCI:
	       view = manage(new GCIRecvView(m, vm));
	       break;
	  default:
	       break;
	  }
     }	  
     return view;
}

bool MessageManager::hasView(const string& jid, MessageType type)
{
     ViewMap& vm = _mviews[type];
     string user = (type == mtGroupchat) ? JID::getUserHost(jid) : jid;
     return (vm.find(user) != vm.end());
}

void MessageManager::initdir()
{
     // Ensure _logdir is terminated with a /
     if (_spool[_spool.length()-1] != '/')
          _spool += "/";
     // Replace ~ with $HOME
     if (_spool[0] == '~')
          _spool.replace(0, 1, g_get_home_dir());
     // See if directory already exists..
     if (!g_file_test(_spool.c_str(), G_FILE_TEST_ISDIR) &&
         (mkdir(_spool.c_str(), 0700) == -1))
     {
          g_error(("Unable to create logging directory: " + _spool).c_str());
          G_App->quit();
     }
}

void MessageManager::clearSpool(const string& jid, MessageType type)
{
     if (jid.empty())
	  return;
     string user = (type == mtGroupchat) ? JID::getUserHost(jid) : jid;
     string fileend = user + "-" + translateType(type);
     if (fileend.find("/") != string::npos)
	  fileend.replace(fileend.find("/"), 1, "-");
     string filename = _spool + fileend;
     unlink(filename.c_str());
     // Remove file from the spool map so it is properly re-created if
     // a new message comes in from the user
     map<string, ofstream*>::iterator it = _mspools[type].find(user);
     if (it != _mspools[type].end())
     {
          delete it->second;
          _mspools[type].erase(it);
     }
}

void MessageManager::clearEvents(const string& jid, MessageType type)
{
     if (jid.empty())
	  return;
     string user = (type == mtGroupchat) ? JID::getUserHost(jid) : jid;
     EventList::iterator it = _events.begin();
     for ( ; it != _events.end(); )
     {
          if (it->first == user && it->second == type)
               _events.erase(it++);
	  else
	       it++;
     }
     // If the user has no more events, clear the events on the roster
     if (getEvent(user) == _events.end())
	  G_Win->clear_event(user);
}

// Find the first event in the list (the oldest event) for a user
MessageManager::EventList::iterator MessageManager::getEvent(const string& jid)
{
     EventList::iterator it = _events.begin();
     // FIXME: there might be issues with this and flashing events
     // but I can't think of a way to fix it.  Need to check if problems
     // actually occur when receiving msgs from same jid but different resources.  
     while (it != _events.end() && it->first != jid)
          it++;
     return it;
}

// Send a message, properly logging it and encrypting or signing it
// on return encryption is the actual level of encryption
bool MessageManager::send_message(Message m, Encryption& encryption)
{
     G_App->getLogger().log(m.getTo(), m);

     GabberGPG& gpg = G_App->getGPG();
     if (encryption == encSign && gpg.enabled())
     {
	  // sign the message
          string message;
          GPGInterface::Error err;

          if ((err = gpg.sign(GPGInterface::sigDetach, m.getBody(), message)) != GPGInterface::errOK)
          {
	       // Couldn't sign message.  Verify the message should still be
	       // sent
               Gnome::Dialog* d;
               string question = _("An error occured trying to sign your message.\nSend the message unsigned?");
               d = manage(Gnome::Dialogs::question_modal(question));

               gint button = d->run_and_close();
               // They pressed No so don't send the message
               if (button == 1)
                    return false;
	       // no encryption was used
	       encryption = encNone;
          }
          else
          {
               // Add the signature to the message
               Tag& x = m.addX();
               x.addCDATA(message.c_str(), message.length());
               x.putAttrib("xmlns", "jabber:x:signed");
          }
     }
     else if (encryption == encEncrypt && gpg.enabled())
     {
	  // encrypt the message
	  string message;
	  string keyid;

          try {
               // Lookup the recipient's jid in the keymap to get their keyid
               keyid = gpg.find_jid_key(m.getTo());
               GPGInterface::Error err;

               if ((err = gpg.encrypt(keyid, m.getBody(), message, true)) != GPGInterface::errOK)
               {
		    // couldn't encrypt message.  ask if it should be sent 
		    // unencrypted 
                    Gnome::Dialog* d;
                    string question = _("An error occured trying to encrypt your message.\nSend the message unencrypted?");
                    d = manage(Gnome::Dialogs::question_modal(question));

                    gint button = d->run_and_close();
                    // They pressed No so don't send the message. 
                    if (button == 1)
                         return false;
		    // message us being sent without encryption
		    encryption = encNone;
               }
	       else
	       {
	            // Change the body of the message to indicate the message is encrypted and 
	            // add the encrypted part to the message
                    m = Message(m.getTo(), "This Message Is Encrypted. ", m.getType());
                    Tag& x = m.addX();
                    x.addCDATA(message.c_str(), message.length());
                    x.putAttrib("xmlns", "jabber:x:encrypted");
               }
	  } catch(GabberGPG::GPG_InvalidJID& e) {
               cerr << "FIXME: don't have Public Key.  Need a way to get it." << endl;

               Gnome::Dialog* d;
               string question = _("An error occured trying to encrypt your message.\nSend the message unencrypted?");
               d = manage(Gnome::Dialogs::question_modal(question));

               gint button = d->run_and_close();
               // They pressed No so don't send the message
               if (button == 1)
                    return false;
	       // message is being sent without encryption
	       encryption = encNone;
          }
     }
     else
     {
	  // the message wasn't encrypted
	  encryption = encNone;
     }

     // Send the message
     G_App->getSession() << m;

     return true;
}

MessageManager::MessageType MessageManager::translateType(const Message& m)
{
     // scan for extensions first
     if (m.getBaseTag().getTag("x", "jabber:x:roster"))
	  return mtRoster;
     else if (m.getBaseTag().getTag("x", "jabber:x:conference"))
	  return mtGCI;

     switch (m.getType())
     {
     case Message::mtNormal:
	  return mtNormal;
     case Message::mtError:
	  return mtError;
     case Message::mtChat:
	  return mtChat;
     case Message::mtGroupchat:
	  return mtGroupchat;
     case Message::mtHeadline:
	  return mtHeadline;
     }
     return mtNormal;
}

string MessageManager::translateType(MessageType type)
{
     switch (type)
     {
     case mtNormal:
	  return "normal";
     case mtError:
	  return "error";
     case mtChat:
	  return "chat";
     case mtGroupchat:
	  return "groupchat";
     case mtHeadline:
	  return "headline";
     case mtRoster:
	  return "roster";
     case mtURL:
	  return "url";
     case mtGCI:
	  return "gci";
     default:
	  return "";
     }
     return "";
}

// --------------------------------------
//
// MessageLoader
//
// --------------------------------------
MessageLoader::MessageLoader(const string& filename, MessageManager& mm)
     : _manager(mm)
{
     // Open an istream 
     ifstream spoolfile(filename.c_str());
     if (spoolfile)
     { 
	  // Feed all istream data into the parser
	  char buf[4096];
	  spoolfile.getline(buf, 4096);
	  while (spoolfile.gcount() != 0)
	  {
	       push(buf);
	       spoolfile.getline(buf, 4096);
	  }
     }
}

MessageLoader::~MessageLoader()
{
}

void MessageLoader::OnTag(Tag& t)
{
     // Ensure we've got a message tag..
     if (t.getName() == "message")
     {
	  Message m(t);
	  _manager.add(m, false);
     }
}
