/*
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 * 
 *  Gabber
 *  Copyright (C) 1999-2001 Dave Smith & Julian Missig
 */



// General headers
#include "GabberApp.hh"
#include "GladeHelper.hh"
#include "Gabber.hh"
#include "GabberWin.hh"
#include "RosterView.hh"
#include "GabberUtility.hh"
#include "BaseGabberWindow.hh"

// Dialogs
#include "AgentInterface.hh"
#include "PrefsInterface.hh"
#include "AddContactDruid.hh"
#include "StatusInterface.hh"
#include "WelcomeDruid.hh"
#include "ContactInfoInterface.hh"
#include "FilterInterface.hh"
#include "DebugInterface.hh"
#include "MessageManager.hh"
#include "MessageViews.hh"
#include "IgnoreInterface.hh"

// GPG
#include "GabberGPG.hh"

using namespace jabberoo;
using namespace GabberUtil;

GabberWin* G_Win = NULL;

GabberWin::GabberWin()
     : BaseGabberWindow("Gabber_win"),
       connected(false),
       _pix_path(ConfigManager::get_PIXPATH())
{
     // Initialize G_Win
     G_Win = this;

     _thisWindow->realize();

     MainWin = getWidget<Gnome::App>("Gabber_win");

     // Setup roster
     _roster = new RosterView(getWidget<Gtk::CTree>("Gabber_RosterTree"),G_App->getSession());

     // GABBER *
     // * SHOW AS *
     _m_Presence = getPixmapMenuItem("Gabber_Showas_menu");
     // * LOGIN *
     _m_Login = getMenuItem("Gabber_Login_item");
     _m_Login->activate.connect(slot(&LoginDlg::execute));
     // * LOGOUT *
     _m_Logout = getMenuItem("Gabber_Logout_item");
     _m_Logout->activate.connect(slot(this, &GabberWin::on_disconnect));
     _m_Logout->set_sensitive(false);
     // * HIDE OFFLINE USERS *
     _m_HideOffline = getWidget<Gtk::CheckMenuItem>("Gabber_HideOffline_item");
     _m_HideOffline->toggled.connect(slot(this, &GabberWin::on_HideOfflineUsers_toggled));
     // * HIDE AGENTS *
     _m_HideAgents = getWidget<Gtk::CheckMenuItem>("Gabber_HideAgents_item");
     _m_HideAgents->toggled.connect(slot(this, &GabberWin::on_HideOfflineUsers_toggled));
     // * HIDE SPECIFIC GROUPS *
     //getMenuItem("Gabber_HideGroups_item")->set_sensitive(false);
     // * USER INFO *
     _m_UserInfo = getMenuItem("Gabber_UserInfo_item");
     _m_UserInfo->activate.connect(slot(&MyContactInfoWin::execute));
     _m_UserInfo->set_sensitive(false);
     // * PREFS *
     getMenuItem("Gabber_Preferences_item")->activate.connect(slot(&PrefsWin::execute));
     // * EXIT *
     getMenuItem("Gabber_Exit_item")->activate.connect(slot(this, &GabberWin::on_Exit_activate));

     // SERVICES *
     _m_Roster = getMenuItem("Gabber_Services_menu");
     _m_Roster->set_sensitive(false);
     // * BLANK MESSAGE *
     getMenuItem("Gabber_BlankMessage_item")->activate.connect(slot(this, &GabberWin::on_BlankMessage_activate));
     // * ADD USER *
     getMenuItem("Gabber_AddUser_item")->activate.connect(slot(this, &GabberWin::on_AddUser_activate));
     // * ADD GROUP *
     getMenuItem("Gabber_AddGroup_item")->activate.connect(slot(this, &GabberWin::on_AddGroup_activate));
     // * TRANSPORTS *
     getMenuItem("Gabber_Agents_item")->activate.connect(slot(&AgentBrowserDlg::execute));
     // * GROUP CHAT *
     getMenuItem("Gabber_GroupChat_item")->activate.connect(slot(&GCJoinDlg::execute));
     // * FILTERS *
     getMenuItem("Gabber_Filters_item")->activate.connect(slot(&FilterListView::execute));
     // * SHOW HEADLINES *
     _m_ShowHeadlines = getWidget<Gtk::CheckMenuItem>("Gabber_ShowHeadlines_item");
     _m_ShowHeadlines->toggled.connect(slot(this, &GabberWin::on_ShowHeadlines_toggled));
     // * IGNORE LIST */
     getMenuItem("Gabber_IgnoreList_item")->activate.connect(slot(&IgnoreDlg::execute));

     // HELP *
     // * MANUAL *
     getMenuItem("Gabber_Manual_item")->activate.connect(slot(this, &GabberWin::on_Manual_activate));
     // * ABOUT *
     getMenuItem("Gabber_About_item")->activate.connect(slot(this, &GabberWin::on_About_activate));
     // * DEBUG *
     // ** RAW XML INPUT
     getMenuItem("Gabber_RawXML_item")->activate.connect(slot(&RawXMLInput::execute));

     // * DELETE *
     _thisWindow->delete_event.connect(slot(this, &GabberWin::on_window_delete));


     _baseMU = manage(new BaseGabberWidget("ManageUser_menu", "Gabber_win"));
     // USER MENU*
     _menuUser = static_cast<Gtk::Menu*>(_baseMU->get_this_widget());
     // * MESSAGE *
     _baseMU->getMenuItem("ManageUser_Message_item")->activate.connect(slot(_roster, &RosterView::on_Message_activate));
     // * OOOCHAT *
     _baseMU->getMenuItem("ManageUser_OOOChat_item")->activate.connect(slot(_roster, &RosterView::on_OOOChat_activate));
     // * SEND ROSTER ITEM *
     _baseMU->getMenuItem("ManageUser_SendRoster_item")->activate.connect(slot(_roster, &RosterView::on_SendRoster_activate));
     // * RE-REQUEST S10N *
     _baseMU->getMenuItem("ManageUser_Rerequest_item")->activate.connect(slot(_roster, &RosterView::on_S10nRequest_activate));
     // * EDIT USER *
     _baseMU->getMenuItem("ManageUser_EditUser_item")->activate.connect(slot(_roster, &RosterView::on_EditUser_activate));
     // * EDIT GROUPS *
     _baseMU->getMenuItem("ManageUser_EditGroups_item")->activate.connect(slot(_roster, &RosterView::on_EditGroups_activate));
     // * USER HISTORY *
     _baseMU->getMenuItem("ManageUser_History_item")->activate.connect(slot(_roster, &RosterView::on_History_activate));
     // * DELETE USER *
     _baseMU->getMenuItem("ManageUser_DeleteUser_item")->activate.connect(slot(_roster, &RosterView::on_DeleteUser_activate));
     // * ADD USER (for NIL Users) *
     _baseMU->getMenuItem("ManageUser_AddUsertoRoster_item")->activate.connect(slot(_roster, &RosterView::on_AddUser_activate));
     // hide by default
     _baseMU->getMenuItem("ManageUser_AddUsertoRoster_item")->hide();

     // TRANSPORT MENU*
     _baseMA = manage(new BaseGabberWidget("ManageAgent_menu", "Gabber_win"));
     _menuAgent = static_cast<Gtk::Menu*>(_baseMA->get_this_widget());
     // * MESSAGE *
     _baseMA->getMenuItem("ManageAgent_Message_item")->activate.connect(slot(_roster, &RosterView::on_Message_activate));
     // * OOOCHAT *
     _baseMA->getMenuItem("ManageAgent_OOOChat_item")->activate.connect(slot(_roster, &RosterView::on_OOOChat_activate));
     // * LOGIN TRANS *
     _baseMA->getMenuItem("ManageAgent_ALogin_item")->activate.connect(slot(_roster, &RosterView::on_LoginTransport_activate));
     // * LOGOUT TRANS *
     _baseMA->getMenuItem("ManageAgent_ALogout_item")->activate.connect(slot(_roster, &RosterView::on_LogoutTransport_activate));
     // * TRANSPORT INFO *
     _baseMA->getMenuItem("ManageAgent_AInfo_item")->activate.connect(slot(_roster, &RosterView::on_TransInfo_activate));
     // * EDIT GROUPS
     _baseMA->getMenuItem("ManageAgent_AEditGroups_item")->activate.connect(slot(_roster, &RosterView::on_EditGroups_activate));
     // * REMOVE TRANS *
     _baseMA->getMenuItem("ManageAgent_ARemove_item")->activate.connect(slot(_roster, &RosterView::on_DeleteUser_activate));

     // DOCKLET MENU *
     _baseD = manage(new BaseGabberWidget("Docklet_menu", "Gabber_win"));
     _menuDocklet = static_cast<Gtk::Menu*>(_baseD->get_this_widget());
     // Status
     _md_Presence = _baseD->getPixmapMenuItem("Docklet_Show_menu");
     // * LOGIN *
     _md_Login = _baseD->getMenuItem("Docklet_Login_item");
     _md_Login->activate.connect(slot(&LoginDlg::execute));
     // * LOGOUT *
     _md_Logout = _baseD->getMenuItem("Docklet_Logout_item");
     _md_Logout->activate.connect(slot(this, &GabberWin::on_disconnect));
     _md_Logout->set_sensitive(false);
     // * PREFERENCES *
     _baseD->getMenuItem("Docklet_Preferences_item")->activate.connect(slot(&PrefsWin::execute));
     // * ABOUT *
     _baseD->getMenuItem("Docklet_About_item")->activate.connect(slot(this, &GabberWin::on_About_activate));
     // * HIDE WINDOW *
     _md_Hide = _baseD->getMenuItem("Docklet_Hide_item");
     _md_Hide->activate.connect(slot(this, &GabberWin::toggle_visibility));
     // * SHOW WINDOW *
     _md_Show = _baseD->getMenuItem("Docklet_Show_item");
     _md_Show->activate.connect(slot(this, &GabberWin::toggle_visibility));
     _md_Show->set_sensitive(false);

     // Toolbar
     getWidget<Gtk::Toolbar>("Gabber_Common_toolbar")->set_style(GTK_TOOLBAR_ICONS);
     // - Add User
     getButton("Gabber_AddUser_btn")->clicked.connect(slot(this, &GabberWin::on_AddUser_activate));
     // - Group Chat
     getButton("Gabber_GroupChat_btn")->clicked.connect(slot(&GCJoinDlg::execute));
     // - Browse Agents
     getButton("Gabber_BrowseAgents_btn")->clicked.connect(slot(&AgentBrowserDlg::execute));
     // - Manual
     getButton("Gabber_Manual_btn")->clicked.connect(slot(this, &GabberWin::on_Manual_activate));

     // Presence menu
     _menuPresence = getPixmapMenuItem("Gabber_Presence_menu");
     // Main Presence Menu
     _mbuildPresence.main_selected.connect(slot(this, &GabberWin::on_Show_selected));
     _menuPresence->set_submenu(*_mbuildPresence.get_menu());
     _menuPresence->set_sensitive(false);
     // Gabber Presence Menu
     _mbuildGabberPresence.main_selected.connect(slot(this, &GabberWin::on_Show_selected));
     _m_Presence->set_submenu(*_mbuildGabberPresence.get_menu());
     _m_Presence->set_sensitive(false);
     // Docklet Presence Menu
     _mbuildDockletPresence.main_selected.connect(slot(this, &GabberWin::on_Show_selected));
     _md_Presence->set_submenu(*_mbuildDockletPresence.get_menu());
     _md_Presence->set_sensitive(false);

     // StatusBar
     StatusBar = getWidget<Gnome::AppBar>("Gabber_Status_bar");
//     MainWin->set_menus(*getWidget<Gtk::MenuBar>("Gabber_Menu_bar"));
//     MainWin->set_statusbar(*StatusBar);
     MainWin->install_menu_hints();
     MainWin->enable_layout_config(true);

     // Connect to session events
     G_App->getSession().evtConnected.connect(slot(this, &GabberWin::on_session_connected));
     G_App->getTransmitter().evtDisconnected.connect(slot(this, &GabberWin::on_transmitter_disconnected));

     // Hide toolbar if people (DizzyD ;) hate it
     _dockToolbar = getWidget<Gnome::DockItem>("Gabber_Toolbar_dockitem");
     _dockStatusbar = getWidget<Gnome::DockItem>("Gabber_Status_dockitem");

     // Set the hide offline users toggle and run the check
     _m_HideOffline->set_active(G_App->getCfg().getBoolValue("Roster/HideOffline=false"));
     _m_HideAgents->set_active(G_App->getCfg().getBoolValue("Roster/HideAgents=false"));
     on_HideOfflineUsers_toggled();
     // Do the same for show headlines
     _m_ShowHeadlines->set_active(G_App->getCfg().getBoolValue("Headlines/DisplayTicker=false"));
     on_ShowHeadlines_toggled();

     // Set window defaults
     if (G_App->getCfg().getBoolValue("Window/SaveSize=true"))
	  _thisWindow->set_default_size(G_App->getCfg().getIntValue("Window/width=185"), G_App->getCfg().getIntValue("Window/height=270"));
     else
	  _thisWindow->set_default_size(185, 270);
     if (G_App->getCfg().getBoolValue("Window/SavePos=true"))
	  _thisWindow->set_uposition(G_App->getCfg().getIntValue("Window/x="), G_App->getCfg().getIntValue("Window/y="));
     _thisWindow->show();

     // Refresh the roster so that all the icons & colors get loaded
     refresh_roster();

     display_toolbars(G_App->getCfg().getBoolValue("Toolbars/toolbar=true"), 
		     G_App->getCfg().getBoolValue("Toolbars/statusbar=true"));
     on_ShowHeadlines_toggled();

     // Status Docklet
     DockletWin::display(Presence::stOffline, "Disconnected");
}

gint GabberWin::on_window_delete(GdkEventAny* e)
{
     G_App->getSession().disconnect();
     G_App->quit();
     return 0;
}

void GabberWin::on_disconnect()
{
     Gnome::Dialog* d;
     string question = _("Are you sure you want to disconnect from Jabber?");
     d = manage(Gnome::Dialogs::question_modal(question,
						     slot(this, &GabberWin::handle_disconnect_qry)));
}

void GabberWin::on_session_connected(const Tag& t)
{
     connected = true;

     // Play connected sound
     gnome_triggers_do(NULL, NULL, "gabber", "Connected", NULL);

     // If they didn't want to save the password, clear it
     if (!G_App->getCfg().getBoolValue("Server/SavePassword=false"))
	  G_App->getCfg().putPrivValue("Server/Password", "");

     // Disable login menuitem
     _m_Login->set_sensitive(false);
     _md_Login->set_sensitive(false);
     _m_Logout->set_sensitive(true);
     _md_Logout->set_sensitive(true);

     // Enable agents and roster menus
     _m_Roster->set_sensitive(true);
     _m_UserInfo->set_sensitive(true);

     // Update status bar
     StatusBar->pop();
     StatusBar->push(_("Logged in"));

     // Load the list of expanded roster groups
     _roster->load_expanded_groups();

     // Allow usr to change status
     _menuPresence->set_sensitive(true);
     _m_Presence->set_sensitive(true);
     _md_Presence->set_sensitive(true);

     // If they're in WelcomeDruid, move them to finish
     if (WelcomeDruid::isRunning())
	 	WelcomeDruid::Connected();

     // Send out autoupdate request
     string autoupdateJID = "956878967";             // Gabber's clientID on jabbercentral
     autoupdateJID += "@update.jabber.org/";         // the only place to grab updates right now
     autoupdateJID += ConfigManager::get_VERSION();  // Gabber's version
     G_App->getSession() << Presence(autoupdateJID, Presence::ptAvailable);

     // Grab the Show, Status, and Priority for presence
     _curShow = G_App->getCfg().get_show();
     _curStatus = G_App->getCfg().get_status();
     _curPriority = G_App->getCfg().get_priority();

     // If they want it to remember their last status, then set it
     // if (G_App->getCfg().getBoolValue("Status/save=true"))
     display_status(indexShow(_curShow), _curStatus, _curPriority);
     // else
     // displayStatus(Presence::stOnline, "Online", 0);
}

void GabberWin::on_transmitter_disconnected()
{
     // Update status display
     display_status(Presence::stOffline, "Disconnected", 0);

     // Save the list of expanded roster groups
     _roster->save_expanded_groups();

     // Clear roster..
     _roster->clear();

     // Disable logout menuitem
     _m_Login->set_sensitive(true);
     _md_Login->set_sensitive(true);
     _m_Logout->set_sensitive(false);
     _md_Logout->set_sensitive(false);

     // Disable agents and roster menus
     _m_Roster->set_sensitive(false);
     _m_UserInfo->set_sensitive(false);

     // Update status bar
     StatusBar->pop();
     StatusBar->push(_("Logged out"));

     // Disallow user to change status
     _menuPresence->set_sensitive(false);
     _m_Presence->set_sensitive(false);
     _md_Presence->set_sensitive(false);

     connected = false;
}

void GabberWin::on_AddUser_activate()
{
     AddContactDruid::display(AddContactDruid::auChoice);
}

void GabberWin::on_AddGroup_activate()
{
     Gnome::Dialog* dialog = Gnome::Dialogs::request(*_thisWindow, false, 
						     _("Please enter name of the group:"),
						     "", 0, slot(this, &GabberWin::on_AddGroup_string));
}

void GabberWin::on_AddGroup_string(string group)
{
     _roster->add_temp_group(group);
     _roster->refresh();
}

void GabberWin::on_BlankMessage_activate()
{
     G_App->getMessageManager().display("", MessageManager::mtNormal);
}

void GabberWin::on_About_activate()
{
     string info = _("Gabber: The GNOME Jabber Client.");
     info += "\n";
     info += _("More information about Jabber:\n  http://www.jabbercentral.org/\n  http://jabber.org/\n  http://jabber.com/\n\n");
     info += _("Thanks:\n");
     info += _("  Thomas Muldowney - several patches/fixes\n");
     info += _("  David Zoll - Small bug fixes, g_asserts\n");
     info += _("  Erich Zigler - Gabber FreeBSD port\n");
     info += _("  Lauris Kaplinski - e-font and e-unicode\n");
     info += _("  All of the translators\n");
     info += _("  gtkmm - especially kenelson and cactus\n");
     info += _("  artwiz and garrett - amazing icons\n");
     info += _("  gnomeicu and gaim\n");
     info += _("  All of the Jabber and GNOME people!\n");
     info += _("Gabber's website: http://gabber.sourceforge.net/ ");

     vector<string> authors(6);
     authors[0] = _("Dave \"DizzyD\" Smith <dave@jabber.org>");
     authors[1] = _("Julian \"x-virge\" Missig <julian@jabber.org>");
     authors[2] = _("Brandon \"brandon2\" Lees <brandon@aspect.net>");
     authors[3] = _("Eliot \"e-t\" Landrum <eliot@landrum.cx>");
     authors[4] = _("Konrad Podloucky <konrad@crunchy-frog.org>");
     authors[5] = _("Dave \"bigdave\" Lee");

     Gnome::About* about = new Gnome::About(ConfigManager::get_PACKAGE(), 
					    ConfigManager::get_VERSION(),
			      _("Copyright (c) 1999-2001 "
			      "Julian Missig and Dave Smith"),
			      authors, info);

     about->run();

}

void GabberWin::on_Manual_activate()
{
     GnomeHelpMenuEntry help_entry = { "gabber", "index.html" };
     gnome_help_display (NULL, &help_entry);
}

void GabberWin::on_Show_selected(int custom_index)
{
     string status = _("Online");

     switch (indexShow(custom_index))
     {
     case Presence::stOnline:
	  status = _("Online");
	  break;
     case Presence::stChat:
	  status = _("Chat with me!");
	  break;
     case Presence::stAway:
	  status = _("I'm away.");
	  break;
     case Presence::stXA:
	  status = _("I'm not available right now.");
	  break;
     case Presence::stDND:
	  status = _("Please do not disturb me right now.");
	  break;
     default:
	  display_status(G_App->getCfg().get_show(), 
			 G_App->getCfg().get_status(),
			 G_App->getCfg().get_priority());
	  return;
     }
     manage(new ShowDlg(indexShow(custom_index), status, _curPriority));
}

void GabberWin::on_Exit_activate()
{
     G_App->getSession().disconnect();
     G_App->quit();
}

void GabberWin::handle_disconnect_qry(int code)
{
     switch (code)
     {
     case 0: // Yes
          // If the Session isn't connected, disconnect the transmitter otherwise disconnect session
          if (!G_App->getSession().disconnect())
               G_App->getTransmitter().disconnect();

	  // Open Login Dialog
	  //LoginDlg::execute();
	  // Close G_Win
	  //saveSize();
	  //_thisWindow->destroy();
	  //delete G_Win;
	  //G_Win = NULL;
	  break;
     case 1: // No
	  ;
     }
}

void GabberWin::on_HideOfflineUsers_toggled()
{
     if (_m_HideOffline->is_active())
     {
	  if (_m_HideAgents->is_active())
     	       _roster->set_view_filter(RosterView::rvfOnlineOnlyNoAgents);
	  else
	       _roster->set_view_filter(RosterView::rvfOnlineOnly);
     }
     else
     {
	  if (_m_HideAgents->is_active())
	       _roster->set_view_filter(RosterView::rvfAllNoAgents);
	  else
	       _roster->set_view_filter(RosterView::rvfAll);
     }
}

void GabberWin::on_ShowHeadlines_toggled()
{
     // Show/hide the ticker
}

// -------------------------------------------------------------------
//
// Public interface to the window
//
// -------------------------------------------------------------------
void GabberWin::refresh_roster()
{
     _roster->refresh(true);
}

void GabberWin::display_status(Presence::Show show, const string& current_status, int priority, bool send_presence)
{
     char* cpriority;
     cpriority = g_strdup_printf("%d", priority);
     // We have this here in case if we want to control applet, docklet, etc.
     if (connected)
     {
	  Presence p("", Presence::ptAvailable, show, current_status, string(cpriority));
	  GabberGPG& gpg = G_App->getGPG();
	  if (gpg.enabled())
	  {
	       string sig;

	       if (gpg.sign(GPGInterface::sigDetach, current_status, sig) != GPGInterface::errOK)
		    cerr << "Couldn't sign status" << endl;
	       else
	       {
		    Tag& x = p.addX();
		    x.addCDATA(sig.c_str(), sig.length());
		    x.putAttrib("xmlns", "jabber:x:signed");
	       }
	  }
	  if (send_presence)
	       G_App->getSession() << p;
     }
     g_free(cpriority);

     // Save the priority
     _curPriority = priority;

     // Set the pixmaps of the presence menu items
     _menuPresence->remove(); // Remove the current label
     _menuPresence->add_label(getShowName(show)); // Set the label to the show name
     switch(show)
     {
     case Presence::stOffline:
	  change_pixmap(_menuPresence, _pix_path + "offline.xpm");
	  change_pixmap(_m_Presence, _pix_path + "offline.xpm");
	  change_pixmap(_md_Presence, _pix_path + "offline.xpm");
	  break;
     case Presence::stOnline:
	  change_pixmap(_menuPresence, _pix_path + "online.xpm");
	  change_pixmap(_m_Presence, _pix_path + "online.xpm");
	  change_pixmap(_md_Presence, _pix_path + "online.xpm");
	  break;
     case Presence::stChat:
	  change_pixmap(_menuPresence, _pix_path + "chat.xpm");
	  change_pixmap(_m_Presence, _pix_path + "chat.xpm");
	  change_pixmap(_md_Presence, _pix_path + "chat.xpm");
	  break;
     case Presence::stAway:
	  change_pixmap(_menuPresence, _pix_path + "away.xpm");
	  change_pixmap(_m_Presence, _pix_path + "away.xpm");
	  change_pixmap(_md_Presence, _pix_path + "away.xpm");
	  break;
     case Presence::stXA:
	  change_pixmap(_menuPresence, _pix_path + "xa.xpm");
	  change_pixmap(_m_Presence, _pix_path + "xa.xpm");
	  change_pixmap(_md_Presence, _pix_path + "xa.xpm");
	  break;
     case Presence::stDND:
	  change_pixmap(_menuPresence, _pix_path + "dnd.xpm");
	  change_pixmap(_m_Presence, _pix_path + "dnd.xpm");
	  change_pixmap(_md_Presence, _pix_path + "dnd.xpm");
	  break;
     default:
	  break;
     }
     // Update status docklet
     if (connected)
	  DockletWin::display(show, current_status);
     else
	  DockletWin::display(Presence::stOffline, _("Disconnected"));

     // Fire off the event for my presence
     evtMyPresence(show, current_status, priority);
}

void GabberWin::display_toolbars(bool displaytoolbar, bool displaystatus)
{
     // Show or hide the toolbar
     (displaytoolbar) ? _dockToolbar->show()   : _dockToolbar->hide();
     // Show or hide the status bar
     (displaystatus)  ? _dockStatusbar->show() : _dockStatusbar->hide();
}

void GabberWin::save_size()
{
     // Save the GabberWin size and position
     int width, height, x, y;
     _thisWindow->get_window().get_size(width, height);
     _thisWindow->get_window().get_root_origin(x, y);
     ConfigManager& c = G_App->getCfg();
     c.putValue("Window/x", x);
     c.putValue("Window/y", y);
     c.putValue("Window/width", width);
     c.putValue("Window/height", height);
     c.putValue("Roster/HideOffline", _m_HideOffline->is_active());
     c.putValue("Roster/HideAgents", _m_HideAgents->is_active());
     c.putValue("Headlines/DisplayTicker", _m_ShowHeadlines->is_active());
     c.sync();
}

void GabberWin::toggle_visibility()
{
     // If the window is visible
     if (_thisWindow->is_visible())
     {
	  // Save the size
	  _thisWindow->get_window().get_size(_gwin_width, _gwin_height);
	  _thisWindow->get_window().get_root_origin(_gwin_x, _gwin_y);  
	  // Hide it
	  _thisWindow->hide();
	  // Allow user to show the window
	  _md_Show->set_sensitive(true);
	  _md_Hide->set_sensitive(false);
     }
     else
     {
	  //restore the size we saved because it sometimes forgets
	  _thisWindow->set_default_size(_gwin_width, _gwin_height);
	  _thisWindow->set_uposition(_gwin_x, _gwin_y);

	  // Show it
	  _thisWindow->show();
	  // Allow user to hide the window
	  _md_Show->set_sensitive(false);
	  _md_Hide->set_sensitive(true);
     }
}

void GabberWin::logging_in()
{
     // While it's trying to login, do this stuff:

     // Disable login, enable logout
     _m_Login->set_sensitive(false);
     _md_Login->set_sensitive(false);
     _m_Logout->set_sensitive(true);
     _md_Logout->set_sensitive(true);

     // Update the status bar   
     StatusBar->pop();
     StatusBar->push(_("Logging in..."));
}

void GabberWin::toggle_roster_popup(bool inlist)
{
     if (inlist)
     {
	  // Hide Menu Items Used only in the NIL popup
	  _baseMU->getMenuItem("ManageUser_AddUsertoRoster_item")->hide();
     }
     else
     {
	  // Hide Menu Items not used in the NIL popup
	  _baseMU->getMenuItem("ManageUser_EditGroups_item")->hide();
	  _baseMU->getMenuItem("ManageUser_DeleteUser_item")->hide();
     }
}

void GabberWin::roster_event(const string& jid, MessageManager::MessageType event)
{
     bool on_roster = true;

     // default is the username
     string nickname = JID::getUser(jid);
     try {
          nickname = G_App->getSession().roster()[JID::getUserHost(jid)].getNickname();
     } catch (Roster::XCP_InvalidJID& e)
     {
          // Special handling for a groupchat ID -- use the resource as the nickname
          if (G_App->isGroupChatID(jid))
               nickname = JID::getResource(jid);
          // Clear the onRoster flag
          on_roster = false;
     }

     if (!on_roster)
     {
          Roster::Item t(jid, nickname);
          t.addToGroup(_("Not in Roster"));
          _roster->addNILItem(t);
     }
}

void GabberWin::clear_event(const string& jid)
{
     _roster->clearNILItem(jid);
}
