/*
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 * 
 *  Gabber
 *  Copyright (C) 1999-2000 Dave Smith & Julian Missig
 */


#include "GabberApp.hh"
// Windows
#include "GabberWin.hh"
#include "WelcomeDruid.hh"
// Dialogs
#include "S10nInterface.hh"
#include "PrefsInterface.hh"
#include "MessageManager.hh"
#include "MessageViews.hh"
#include "ErrorManager.hh"
// Misc
#include "TCPTransmitter.hh"
#include "GabberLogger.hh"
#include "GabberGPG.hh"
#include "AutoAway.hh"

// Spell checking
#include "gtkspell.h"

// For uname
#include <sys/utsname.h>

extern "C" {
#ifdef GABBER_WINICON
#include <libgnomeui/gnome-window-icon.h>
#endif
}

#include "jutil.hh"
using namespace jabberoo;

GabberApp* G_App;


GabberApp::GabberApp(int argc, char** argv)
     : manage_groups(true),
       _GnomeApp(ConfigManager::get_PACKAGE(), ConfigManager::get_VERSION(), 
		 argc, argv),
       _SourceDir(ConfigManager::get_GLADEDIR()),
       _Cfg(new ConfigManager(ConfigManager::get_CONFIG())),
       _share_dir(ConfigManager::get_SHAREDIR()),
       _pix_path(ConfigManager::get_PIXPATH()),
       _logging(true),
       _createUser(false)
{
     cerr << ConfigManager::get_PACKAGE() << " " << ConfigManager::get_VERSION() << endl;

     // Assign this pointer to the global application pointer
     G_App = this;

     // Initialize glade for gnome
     glade_gnome_init();

     // Set the default window icon
#ifdef GABBER_WINICON
     string window_icon = _pix_path + "gnome-gabber.xpm";
     gnome_window_icon_set_default_from_file(window_icon.c_str());
     gnome_window_icon_init();
#endif

     // Locate UI/Glade file
     if (!g_file_exists(string(_SourceDir + "Login_dlg.glade").c_str()))
          _SourceDir = "./";
     if (!g_file_exists(string(_SourceDir + "Login_dlg.glade").c_str()))
          _SourceDir = "./ui/";
     if (!g_file_exists(string(_SourceDir + "Login_dlg.glade").c_str()))
          _SourceDir = "./src/";
     if (!g_file_exists(string(_SourceDir + "Login_dlg.glade").c_str()))
     {
	  g_error("Could not find Login_dlg.glade!");
          exit(-1);
     }

     // Set the default profile
     _Cfg->setProfile(_Cfg->getStrValue("Profiles/DefaultName=gabber",false));

     // Instantiate objects
     _Session     = new Session;
     _Transmitter = new TCPTransmitter;
     _Logger      = new GabberLogger(_Cfg->getStrValue("Logs/Dir=~/.Gabber"),
				     _Cfg->get_nick());
     _MessageManager = new MessageManager(_Cfg->getStrValue("Spool/Dir=~/.Gabber-spool"));
     _ErrorManager = new ErrorManager();
     _GabberGPG = new GabberGPG;
     _GabberGPG->loadKeyMap();
     _GabberGPG->set_armor(true);
    
     _AutoAway = new AutoAway;

     // Instantiate classes
     Message::setDateTimeFormat(_Cfg->getStrValue("Dates/Format=%d %b %Y %H:%M:%S"));

     // Debugging signals     
     _Session->evtTransmitXML.connect(slot(this, &GabberApp::transmit_XML));
     _Session->evtRecvXML.connect(slot(this, &GabberApp::recv_XML));
     _Session->evtTransmitPacket.connect(slot(this, &GabberApp::transmit_packet));

     // Connect up with session signals
     _Session->evtTransmitXML.connect(slot(_Transmitter, &TCPTransmitter::send));
     _Session->evtMessage.connect(slot(this, &GabberApp::on_session_message));
     _Session->evtAuthError.connect(slot(this, &GabberApp::on_session_auth_error));
     _Session->evtPresenceRequest.connect(slot(this, &GabberApp::on_session_presence_request));
     _Session->evtUnknownPacket.connect(slot(this, &GabberApp::on_session_unknown_packet));
     _Session->evtDisconnected.connect(slot(this, &GabberApp::on_session_disconnected));
     _Session->evtOnVersion.connect(slot(this, &GabberApp::on_session_version));
     _Session->evtPresence.connect(slot(this, &GabberApp::on_session_presence));

     // Connect up transmitter signal
     _Transmitter->evtConnected.connect(slot(this, &GabberApp::on_transmitter_connected));
     _Transmitter->evtDisconnected.connect(slot(this, &GabberApp::on_transmitter_disconnected));
     _Transmitter->evtError.connect(slot(this, &GabberApp::on_transmitter_error));
     _Transmitter->evtDataAvailable.connect(slot(_Session, &Session::push));

     // Startup sound
     gnome_triggers_do(NULL, NULL, "gabber", "Startup", NULL);

     // Startup spell checking
     init_spellcheck();

     // Run the wizard if necessary
     if ((!_Cfg->getBoolValue("Wizards/WelcomeHasRun=false", false)) && 
	 (_Cfg->getBoolValue("Wizards/Firsttime=true", false)))   // This is what 0.7.0 set
     {
	  WelcomeDruid::execute();
	  _Cfg->putValue("Wizards/WelcomeHasRun", true, false);
     }
     else {
	  // Check config and auto-connect if necessary
	  if (_Cfg->getBoolValue("Server/Autologin=false",false))
	       if (_Cfg->getBoolValue("Server/SavePassword=false"))
	       {
		    init_win();

                    // FIXME: Attempt to sign something here so it loops to get the correct passphrase.  It
                    // would be better to do this when the passphrase is first needed but there is some wierd problem
	            // with Gnome::Dialog->run() and receiving the roster at the same time causing the roster to be ignored.
	            GabberGPG& gpg = G_App->getGPG();
	            string dest;
	            if (gpg.enabled() && (gpg.sign(GPGInterface::sigClear, string(""), dest) == GPGInterface::errPass))
	            {
	                 // If we couldn't get the right passphrase, disable gpg
	                 gpg.disable();
	            }

		    login();
	       }
	       else
		    // For *now* we just give them login dialog
		    // Soon this should be a simple password dialog
		    LoginDlg::execute();
	  else
	       LoginDlg::execute();
     }
}

void GabberApp::init_win()
{
     if (G_Win == NULL)
     {
	  _MainWindow  = new GabberWin;
	  
	  // Initialize G_Win
	  G_Win = _MainWindow;
     }
}

void GabberApp::init_spellcheck()
{
     if (!gtkspell_running())
     {
	  // Start up gtkspell for spell checking
	  char* spellcmd[] = { "ispell", "-a", NULL };
	  if (gtkspell_start(NULL, spellcmd) < 0)
	  {
	       char* spellcmd2[] = { "aspell", "pipe", NULL };
	       if (gtkspell_start(NULL, spellcmd2) < 0)
	       {
		    cerr << "Neither ispell nor aspell could run, spell-checking disabled" << endl;
	       }
	  }
     }
}

void GabberApp::stop_spellcheck()
{
     if (gtkspell_running())
     {
	  // Stop gtkspell processes
	  gtkspell_stop();
     }
}

void GabberApp::transmit_XML(const char* XML)
{
     cerr << jutil::getTimeStamp() << "<<< " << XML << endl;
}

void GabberApp::recv_XML(const char* XML)
{
     cerr << jutil::getTimeStamp() << ">>> " << XML << endl;
}

void GabberApp::transmit_packet(const Packet& p)
{
#if 0
     if (!_logging)
	  return;
     // Shouldn't we be able to use RTTI to determine if p is Message ???
     if (p.getBaseTag().getName() != "message")
          return;
     Message m = Message(p.getBaseTag());
     _Logger->log(m.getTo(), m);
#endif
}

GabberApp::~GabberApp()
{
     // Release
     delete _Session;
     // set session ponter to null so we know it was deleted if the delete of _Transmitter
     // sends out a signal
     _Session = NULL;
     delete _Transmitter;
     delete _Cfg;
     delete _Logger;
     delete _MessageManager;
     delete _ErrorManager;
     delete _GabberGPG;
     delete _AutoAway;

     // Delete the main window
     delete _MainWindow;
}

void GabberApp::run()
{
     // Let the gnome app start processing messages
     Gnome::Main::run();
}

void GabberApp::quit()
{
     // Save GabberWin's window size
     G_Win->save_size();

     _GabberGPG->saveKeyMap();

     // Stop gtk/gnome message loop
     Gnome::Main::quit();

     // Stop spell checking
     stop_spellcheck();

     delete G_App;
}

GladeXML* GabberApp::load_resource(const char* name)
{
     return glade_xml_new(string(_SourceDir + name + ".glade").c_str(), name);
}

GladeXML* GabberApp::load_resource(const char* name, const char* filename)
{
     return glade_xml_new(string(_SourceDir + filename + ".glade").c_str(), name);
}

void GabberApp::login()
{
     G_Win->logging_in();
     _Transmitter->connect(_Cfg->getStrValue("Server/Server=jabber.com"),
			   _Cfg->getIntValue("Server/Port=5222"));
}

bool GabberApp::isGroupChatID(const string& jid)
{
     return (_GroupChatIDs.find(JID::getUserHost(jid)) != _GroupChatIDs.end());
}

bool GabberApp::isTransport(const string& jid)
{
     return ((jid.find("@") == string::npos) && (jid.find("/") != string::npos));
}

const char* GabberApp::getLogFile(const string& jid) const
{
     return _Logger->getLogFile(jid).c_str();
}

void GabberApp::setLogDir(const string& dir)
{
     _Logger->setLogDir(dir);
}

void GabberApp::setLogging(bool onoff)
{
     _logging = onoff;
}

void GabberApp::setLogHTML(bool html)
{
     _Logger->setLogHTML(html);
}

// -------------------------------------
//
// Session event handlers
//
// -------------------------------------
void GabberApp::on_session_message(const Message& m)
{
     // Update group chat IDs if necessary
     if (m.getType() == Message::mtGroupchat)
	  _GroupChatIDs.insert(JID::getUserHost(m.getFrom()));

     // Route messages properly
     if (m.getType() == Message::mtError)
     {
	  _ErrorManager->add(m);
	  return;
     }
     
     if (m.getBaseTag().getTag("x", "jabber:x:autoupdate"))
	  manage(new AutoupdateDlg(m));

     if (_logging)
	  _Logger->log(m.getFrom(), m);

     // Get real message type
     MessageManager::MessageType mtype = MessageManager::translateType(m);
     if (_Cfg->getBoolValue("Messages/RecvMsgs=false") && (mtype == MessageManager::mtChat))
          mtype = MessageManager::mtNormal;
     else if (_Cfg->getBoolValue("Messages/RecvOOOChats=false") && (mtype == MessageManager::mtNormal))
          mtype = MessageManager::mtChat;

     // Pass message on to the MessageManager
     _MessageManager->add(m);
     // Here we use the message's real type to determine if it should pop up
     if ((_Cfg->getBoolValue("Messages/OpenMsgs=false") && (m.getType() == Message::mtNormal)) ||
	 (_Cfg->getBoolValue("Messages/OpenOOOChats=false") && (m.getType() == Message::mtChat)) ||
	 (mtype == MessageManager::mtHeadline) ||
	 (_MessageManager->hasView(m.getFrom(), mtype)))
     {
	  // Pop up the message
          _MessageManager->display(m.getFrom(), mtype);
     }
     else
     {
          // tell the roster to change the pixmap
          switch (mtype)
          {
          case MessageManager::mtNormal:
               G_Win->roster_event(m.getFrom(), mtype);
               break;
          case MessageManager::mtChat:
               G_Win->roster_event(m.getFrom(), mtype);
               break;
	  case MessageManager::mtRoster:
	       G_Win->roster_event(m.getFrom(), mtype);
	       break;
	  case MessageManager::mtGCI:
	       G_Win->roster_event(m.getFrom(), mtype);
          default:
               break;
          }
     }

}

void GabberApp::on_session_auth_error(int ErrorCode, const char* ErrorMsg)
{
     Gnome::Dialog* d;
  
     // Check for the password
     string password = _Cfg->getPrivStrValue("Server/Password=");
     if (password.empty())
     {
	  d = manage(Gnome::Dialogs::warning(_("Error, unable to load the password.\nPlease try logging in again.")));
	  d->set_modal(true);
	  LoginDlg::execute();
     }

     switch (ErrorCode)
     {
     case 401:
  	  // If they have yet to log in with the wizard, then give them a special dialog
	  if (!WelcomeDruid::hasTriedConnecting())
	  {
	       d = manage(Gnome::Dialogs::question_modal(_("Are you sure you would like Gabber to try\n"
								 "to register ") + 
							       _Cfg->getStrValue("Server/Username=") + "@" +
							       _Cfg->getStrValue("Server/Server=jabber.com") + _("?"),
							       slot(this, &GabberApp::handle_create_user_dlg)));
	  } 
	  else
	  {
	       d = manage(Gnome::Dialogs::question_modal(_("Gabber could not log you in.\n"
								 "Would you like Gabber to try to create a new\n"
								 "account on the selected Jabber server?"),
							       slot(this, &GabberApp::handle_create_user_dlg)));
	  }
	  break;
     case 409:
	  LoginDlg::execute();		     
	  d = manage(Gnome::Dialogs::warning(_("Attempt to create a new user failed: \n")
					     + string(ErrorMsg) 
					     + _("\n. Please select a different user name or verify your \n"
					     "login information.")));
	  d->set_modal(true);
	  break;
     case 500:
	  // If user is trying to auth using Digest, it's possible the server doesn't support it.
	  LoginDlg::execute();
	  d = manage(Gnome::Dialogs::warning(_("Login failed. It is possible this server does not\n"
					       "support digest authentication.\n"
					       "Try connecting again with digest disabled.")));
	  d->set_modal(true);
     }

      
}

void GabberApp::on_session_presence_request(const Presence& p)
{
     if (p.getType() == Presence::ptSubRequest)
     {
	  // Play SubRequest sound
	  gnome_triggers_do(NULL, NULL, "gabber", "SubRequest", NULL);

	  if (isTransport(p.getFrom()))
	  {
	       string host = JID::getHost(p.getFrom());
	       // Send jabber:iq:agent to the agent to get its name
	       string id = G_App->getSession().getNextID();
	       Tag iq("iq");
	       iq.putAttrib("id", id);
	       iq.putAttrib("to", host);
	       iq.putAttrib("type", "get");
	       Tag& query = iq.addTag("query");
	       query.putAttrib("xmlns", "jabber:iq:agent");
	       G_App->getSession() << iq.getXML().c_str();
	       G_App->getSession().registerIQ(id, slot(this, &GabberApp::on_agent_reply));
	       // Also send a jabber:iq:agents query to server to try to get some of the transports
	       // since very few of them actually reply to jabber:iq:agent queries
	       // Set the _current_agent_jid for on_agents_reply to find the right name
	       _current_agent_host = host;
	       _current_agent_jid = p.getFrom();
	       G_App->getSession().queryNamespace("jabber:iq:agents", slot(this, &GabberApp::on_agents_reply), _Cfg->getStrValue("Server/Server=jabber.com"));

	       // Add a part of the jid in case the agent doesn't reply
	       G_App->getSession() << Presence(p.getFrom(), Presence::ptSubscribed);
	       string transName = host.substr(0, host.find('.'));
	       G_App->getSession().roster() << Roster::Item(p.getFrom(), transName);
	  }
	  else
	       S10nReceiveDlg::execute(p);
     }
}

void GabberApp::on_agent_reply(const Tag& iq)
{
     Tag* query = iq.getTag("query");
     if (iq.cmpAttrib("type", "result")) 
     {
          string jid = iq.getAttrib("jid");
	  if (query != NULL)
	  {
	       string agentname = query->getTaggedCDATA("name");
	       // Rename the agent to its actual name
	       if (!agentname.empty())
		    G_App->getSession().roster() << Roster::Item(_current_agent_jid, agentname);
	  }
     }
}

void GabberApp::on_agents_reply(const Tag& iq)
{
     Tag* query = iq.getTag("query");
     if (iq.cmpAttrib("type", "result") && query != NULL)
     {
	  // Walk the children
	  ElementList::const_iterator it = query->getChildren().begin();
	  for (; it != query->getChildren().end(); it++)
          {
               Tag& agent = *static_cast<Tag*>(*it);
	       
               if (agent.getName() != "agent")
                    continue;
	       // Find the agent which matches the current agent jid
               string jid = agent.getAttrib("jid");
	       if (jid != _current_agent_host)
		    continue;
	       string agentname = agent.getTaggedCDATA("name");
	       // Rename the agent to its actual name
	       if (!agentname.empty())
		    G_App->getSession().roster() << Roster::Item(_current_agent_jid, agentname);
	  }
     }
}
	       

void GabberApp::on_session_unknown_packet(const Tag& t)
{
     _Log << "Unknown packet: " << t.getXML().c_str() << endl;
}

void GabberApp::on_session_disconnected()
{
     // Disconnect the socket...
     _Transmitter->disconnect();
}

void GabberApp::on_session_version(string& name, string& version, string& os)
{
     name = "Gabber";
     version = ConfigManager::get_VERSION();
     // Run uname
     struct utsname osinfo;
     uname(&osinfo);
     os = string(osinfo.sysname) + " " + string(osinfo.release) + " " + string(osinfo.machine);
}

// -------------------------------------
//
// Transmitter event handlers
//
// -------------------------------------
void GabberApp::on_transmitter_connected()
{
     // Check for the password
     string password = _Cfg->getPrivStrValue("Server/Password=");
     if (password.empty())
     {
	  Gnome::Dialog* d = manage(Gnome::Dialogs::warning(_("Error, unable to load the password.\n"
							      "Please try logging in again.")));
	  d->set_modal(true);
	  LoginDlg::execute();
     }
     else
     {
	  Session::AuthType atype = Session::atAutoAuth;
	  // If they don't want digest, specify plaintext
	  // otherwise allow jabberoo to try digest
	  if (!_Cfg->getBoolValue("Server/Digest=true"))
	       atype = Session::atPlaintextAuth;

	  // Tell the session to connect with whatever type was selected
	  _Session->connect(_Cfg->getStrValue("Server/Server=jabber.com"),
			    atype,
			    _Cfg->getStrValue("Server/Username="),
			    _Cfg->getStrValue("Server/Resource=Gabber"),
			    password,
			    _createUser);
     }
}

void GabberApp::on_transmitter_disconnected()
{
     // only do this if the session exists
     if (_Session)
     {
          // Tell the session to disconnect
          cerr << "DISCONNECTED" << endl;
          _Session->disconnect();
     }
}

/*void GabberApp::on_transmitter_error(TCPTransmitter::Error e)*/
void GabberApp::on_transmitter_error(const string & emsg)
{
     string errormessage = _("Transmitter error. Disconnected: ");
     errormessage += emsg;

     Gnome::Dialog* d = manage(Gnome::Dialogs::warning(errormessage));
     d->set_modal(true);

     cerr << "Transmitter error. Disconnected: " << emsg << endl;
     _Transmitter->disconnect();
}

void GabberApp::handle_create_user_dlg(int code)
{
     // Default to not creating new user
     _createUser = false;
     switch (code)
     {
     case 0:
	  // Set the "create new user" flag
	  _createUser = true;
	  // Tell the session to attempt to create a user and connect
	  on_transmitter_connected();
	  break;
     case 1:
	  // Disconnect Transmitter
	  _Transmitter->disconnect();
	  break;
     }
}

void GabberApp::on_session_presence(const jabberoo::Presence& p, const jabberoo::Presence::Type prev_type)
{
     // If the presence was unavailable from a transport make all other items on the roster
     // that are from that transport unavailable as well
     if ((p.getType() == Presence::ptUnavailable) && isTransport(p.getFrom()))
     {
	  cerr << "got unavailable presence from transport" << endl;
          Roster::iterator it = _Session->roster().begin();
          string transport = JID::getHost(p.getFrom());
          for ( ; it != _Session->roster().end(); it++)
          {
               // Check if the jid is from the transport but skip over the transport itself
               if ((transport == JID::getHost(it->getJID())) && (p.getFrom() != it->getJID()))
               {
                    // Check to see if we have an available presence from this jid
                    if (_Session->presenceDB().available(it->getJID()))
                    {
                         // Setup unavailable presence for JID
                         Presence p("", Presence::ptUnavailable);
                         p.getBaseTag().putAttrib("from", it->getJID());
                         // send the unavail presence to the presenceDB, roster and whoever else wants it
                         _Session->presenceDB().insert(p);
                         _Session->roster().update(p, Presence::ptAvailable);
		         _Session->evtPresence(p, Presence::ptAvailable);
                    }
               }
          }
     }
     // Jer doesn't like this code.  Transports are supposed to handle this for us
     // when jabberd-1.4 is released.  
#if 0
     else if ((p.getType() == Presence::ptAvailable) && isTransport(p.getFrom()))
     {
	  Roster::iterator it = _Session->roster().begin();
	  string transport = JID::getHost(p.getFrom());

	  // if the presence needs to be signed, sign it here so it is only
	  // signed once
	  char* cpriority;
	  cpriority = g_strdup_printf("%d", _Cfg->get_priority());
	  string current_status = _Cfg->get_status();
	  Presence::Show current_show = indexShow(_Cfg->get_show());
	  string signature;

          if (_GabberGPG->enabled())
          {
              if (_GabberGPG->sign(GPGInterface::sigDetach, current_status, signature) != GPGInterface::errOK)
                    cerr << "Couldn't sign status" << endl;
	  }

          for ( ; it != _Session->roster().end(); it++)
	  {
	       if ((transport == JID::getHost(it->getJID())) && (p.getFrom() != it->getJID()))
	       {
		     Presence p(it->getJID(), Presence::ptAvailable, current_show, current_status, cpriority);
		     if (_GabberGPG->enabled())
		     {
		          Tag& x = p.addX();
		          x.addCDATA(signature.c_str(), signature.length());
			  x.putAttrib("xmlns", "jabber:x:signed");
		     }

		     *_Session << p;
	       }
	  }
     }
#endif
}
