/*
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 * 
 *  Gabber
 *  Copyright (C) 1999-2001 Dave Smith & Julian Missig
 */

/*
 * ContactInterface
 * Author Brandon Lees <brandon@sci.brooklyn.cuny.edu>
 */

#include "ContactInterface.hh"
#include "GabberUtility.hh"
#include "GabberApp.hh"
#include "ContactInfoInterface.hh"

using namespace jabberoo;
using namespace GabberUtil;

ContactRecvView::ContactRecvView(const jabberoo::Message& m, ViewMap& vm)
     : MessageView(m, vm), BaseGabberDialog("ContactRecv_dlg") 
{
     init(m.getFrom());

     render(m);
}

ContactRecvView::ContactRecvView(const string& jid, ViewMap& vm)
     : MessageView(jid, vm), BaseGabberDialog("ContactRecv_dlg")
{
     init(jid);
}

void ContactRecvView::init(string jid)
{
     _ctreePeople = getWidget<Gtk::CTree>("ContactRecv_People_ctree");
     _RosterView = new SimpleRosterView(_ctreePeople, _roster, false); // Set arg 3 to true when groups are supported
     _RosterView->show_jid();

     getButton("ContactRecv_ViewInfo_btn")->clicked.connect(slot(this, &ContactRecvView::on_ViewInfo_clicked));
     getButton("ContactRecv_Remove_btn")->clicked.connect(slot(this, &ContactRecvView::on_Remove_clicked));
     getButton("ContactRecv_Add_btn")->clicked.connect(slot(this, &ContactRecvView::on_Add_clicked));
     getButton("ContactRecv_Cancel_btn")->clicked.connect(slot(this, &ContactRecvView::on_Cancel_clicked));

     // default is the username
     _nickname = JID::getUser(jid);
     string displaynick = _nickname;
     try {
          _nickname = G_App->getSession().roster()[JID::getUserHost(jid)].getNickname();
	  displaynick = _nickname;
	  if (!JID::getResource(jid).empty())
	       displaynick += "/" + JID::getResource(jid);
     } catch (Roster::XCP_InvalidJID& e) {
          // Special handling for a groupchat ID -- use the resource as the nickname
          if (G_App->isGroupChatID(jid))
	  {
               _nickname = JID::getResource(jid);
	       displaynick = _nickname;
	  }
	  else
	       displaynick = jid;
     }
     _lblRecvFrom = getLabel("ContactRecv_RecvFrom_lbl");
     _lblRecvFrom->set_text(fromUTF8(_lblRecvFrom, displaynick));
}

ContactRecvView::~ContactRecvView()
{
     delete _RosterView;
}

void ContactRecvView::render(const jabberoo::Message& m)
{
     Tag* x = m.getBaseTag().getTag("x", "jabber:x:roster");
     if (!x)
     {
	  cerr << "Got a Message without a roster extension in ContactRecvView::render?!?!" << endl;
	  return;
     }
     ElementList::const_iterator it = x->getChildren().begin();
     for (; it != x->getChildren().end(); it++)
     {
	  Tag& item = *static_cast<Tag*>(*it);

	  if (item.getName() != "item")
	       continue;
	  string jid = item.getAttrib("jid");
	  string nick = item.getAttrib("name");
	  cerr << "Got roster item " << jid << " " << nick << endl;
	  Roster::Item ritem(jid, nick);
	  ElementList::const_iterator git = item.getChildren().begin();
	  for ( ; git != item.getChildren().end(); git++)
	  {
	       Tag& group = *static_cast<Tag*>(*git);
	       if (group.getName() != "group")
		    continue;

	       ritem.addToGroup(group.getData());
	       cerr << "Got group " << group.getData() << endl;
	  }
	  if (ritem.empty())
	       ritem.addToGroup("Unfiled");
	  _roster.insert(make_pair(jid, ritem));
     }
     _RosterView->refresh();
}

void ContactRecvView::on_Remove_clicked()
{
     if (_ctreePeople->selection().begin() != _ctreePeople->selection().end())
     {
          Gtk::CTree::Row& r = *_ctreePeople->selection().begin();
	  Roster::ItemMap::iterator it = _roster.find(r[_RosterView->_colJID].get_text());
	  if (it != _roster.end())
	  {
	       _roster.erase(it);
	       _RosterView->refresh();
	  }
     }
}

void ContactRecvView::on_ViewInfo_clicked()
{
     if (_ctreePeople->selection().begin() != _ctreePeople->selection().end())
     {
          Gtk::CTree::Row& r = *_ctreePeople->selection().begin();
          // Check the roster for the jid to make sure it isn't a group
          Roster::ItemMap::iterator it = _roster.find(r[_RosterView->_colJID].get_text());
          if (it != _roster.end())
          {
	       Roster::Subscription type;
	       // Try getting the subscription
	       try {
		    type = G_App->getSession().roster()[JID::getUserHost(it->first)].getSubsType();
	       } catch (Roster::XCP_InvalidJID& e) {
		    type = Roster::rsNone;
	       }
               ContactInfoDlg::display(it->first, type);
          }
     }
}

void ContactRecvView::on_Add_clicked()
{
     for (Roster::ItemMap::iterator it = _roster.begin(); it != _roster.end(); it++)
	  G_App->getSession().roster() << it->second;

     close();
}

void ContactRecvView::on_Cancel_clicked()
{
     close();
}

///////////////////////////////////////////////////
// Contact Send Dialog
///////////////////////////////////////////////////

ContactSendDlg::ContactSendDlg(const string& jid)
     : BaseGabberDialog("ContactSend_dlg"),
       _jid(jid)
{
     _ctreePeople = getWidget<Gtk::CTree>("ContactSend_People_ctree");
     _RosterView = new SimpleRosterView(_ctreePeople, _roster, false); // Set arg 3 to true when groups are supported
     _RosterView->show_jid();
     _entResource = getEntry("ContactSend_Resource_ent");
     _cboResource = getWidget<Gtk::Combo>("ContactSend_Resource_cbo");

     getButton("ContactSend_ViewInfo_btn")->clicked.connect(slot(this, &ContactSendDlg::on_ViewInfo_clicked));
     getButton("ContactSend_Remove_btn")->clicked.connect(slot(this, &ContactSendDlg::on_Remove_clicked));
     getButton("ContactSend_Send_btn")->clicked.connect(slot(this, &ContactSendDlg::on_Send_clicked));
     getButton("ContactSend_Cancel_btn")->clicked.connect(slot(this, &ContactSendDlg::on_Cancel_clicked));

     // default is the username
     _nickname = JID::getUser(jid);
     try {
          _nickname = G_App->getSession().roster()[JID::getUserHost(jid)].getNickname();
     } catch (Roster::XCP_InvalidJID& e) {
          // Special handling for a groupchat ID -- use the resource as the nickname
          if (G_App->isGroupChatID(jid))
               _nickname = jid;
     }

     // If it's a group chat id, don't bother with resources
     if (G_App->isGroupChatID(jid))
     {
	  getWidget<Gtk::HBox>("ContactSend_Resource_hbox")->hide();
     }
     else
     {
	  list<string> resources;
	  try {
	       // Walk the list of resources and add them to the combo box
	       PresenceDB::range r = G_App->getSession().presenceDB().equal_range(JID::getUserHost(_jid));
	       for (PresenceDB::const_iterator it = r.first; it != r.second; it++)
	       {
		    const Presence& p = *it;
		    
		    // If this presence is a NA presence, then skip it
		    if (p.getType() == Presence::ptUnavailable)
			 continue;
		    
		    // Extract the resource
		    const string& res = JID::getResource(p.getFrom());
		    resources.push_back(res);
	  }
	  } catch (PresenceDB::XCP_InvalidJID& e) {
	       // No presences from any resources
	  }
	  if (!resources.empty())
	       _cboResource->set_popdown_strings(resources);
     }

     _lblSendTo = getLabel("ContactSend_SendTo_lbl");
     _lblSendTo->set_text(fromUTF8(_lblSendTo, _nickname));
}

ContactSendDlg::~ContactSendDlg()
{
     delete _RosterView;
}

void ContactSendDlg::on_ViewInfo_clicked()
{
     if (_ctreePeople->selection().begin() != _ctreePeople->selection().end())
     {
          Gtk::CTree::Row& r = *_ctreePeople->selection().begin();
	  // Check the roster for the jid to make sure it isn't a group
          Roster::ItemMap::iterator it = _roster.find(r[_RosterView->_colJID].get_text());
          if (it != _roster.end())
          {
	       ContactInfoDlg::display(it->first);
	  }
     }
}

void ContactSendDlg::on_Remove_clicked()
{
     if (_ctreePeople->selection().begin() != _ctreePeople->selection().end())
     {
          Gtk::CTree::Row& r = *_ctreePeople->selection().begin();
          Roster::ItemMap::iterator it = _roster.find(r[_RosterView->_colJID].get_text());
	  if (it != _roster.end())
	  {
	       _roster.erase(it);
	       _RosterView->refresh();
	  }
     }
}

void ContactSendDlg::on_Send_clicked()
{
     string body = _("Jabber Contacts attached:");
     // Walk the list and add the JIDs to the message
     for (Roster::ItemMap::iterator it = _roster.begin(); it != _roster.end(); it++)
     {
	  Roster::Item& ritem = it->second;
	  body += "\n" + ritem.getNickname() + " - " + ritem.getJID();
     }

     Message m(_jid, body, Message::mtNormal);
     Tag& x = m.addX();
     x.putAttrib("xmlns", "jabber:x:roster");
     // Walk the list and attach the JIDs
     for (Roster::ItemMap::iterator it = _roster.begin(); it != _roster.end(); it++)
     {
	  Roster::Item& ritem = it->second;

	  Tag& item = x.addTag("item");
	  item.putAttrib("jid", ritem.getJID());
	  item.putAttrib("name", ritem.getNickname());
	  for (Roster::Item::iterator git = ritem.begin(); git != ritem.end(); git++)
	       item.addTaggedCDATA("group", *git);
     }
     G_App->getSession() << m;
     close();
}

void ContactSendDlg::on_Cancel_clicked()
{
     close();
}
