/*
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 * 
 *  Gabber
 *  Copyright (C) 1999-2001 Dave Smith & Julian Missig
 */



#include "ConfigManager.hh"
#include "MessageManager.hh"
#include "GabberConfig.hh"
#include "GabberUtility.hh"
using namespace GabberUtil;

ConfigManager::ConfigManager(const char* name)
     : _CfgName(name),
       _CfgProfile("default")
{
     gnome_config_push_prefix(_CfgName.c_str());
     _nick = getStrValue("User/Nickname=guest");
     _show = indexShow(getIntValue("Status/Show=2"));
     _status = getStrValue("Status/Status=Online");
     _priority = getIntValue("Status/Priority=0");
}

void ConfigManager::setProfile(const string& profile)
{
     _CfgProfile = profile;
}

ConfigManager::~ConfigManager()
{
     gnome_config_pop_prefix();
}

const char* ConfigManager::get_PACKAGE()
{
     return GABBER_PACKAGE;
}

const char* ConfigManager::get_VERSION()
{
     return GABBER_VERSION;
}

const char* ConfigManager::get_GLADEDIR()
{
     return GABBER_GLADEDIR;
}

const char* ConfigManager::get_SHAREDIR()
{
     return GABBER_SHAREDIR;
}

const char* ConfigManager::get_CONFIG()
{
     return GABBER_CONFIG;
}

const char* ConfigManager::get_PIXPATH() 
{
     return GABBER_PIXPATH;
}

const string ConfigManager::get_CSSFILE() 
{
     return GABBER_CSSFILE;
}

void ConfigManager::sync()
{
     putValue("User/Nickname", _nick);
     putValue("Status/Show", indexShow(_show));
     putValue("Status/Status", _status);
     putValue("Status/Priority", _priority);
     gnome_config_sync();
}

// Value setters
void ConfigManager::putValue(const char* name, bool value, bool useprofile)
{
//    if (useprofile)
//	  gnome_config_set_bool(string(_CfgProfile + name).c_str(), value);
//     else
	  gnome_config_set_bool(name, value);
}

void ConfigManager::putValue(const char* name, int value, bool useprofile)
{
//     if (useprofile)
//	  gnome_config_set_int(string(_CfgProfile + name).c_str(), value);
//     else
	  gnome_config_set_int(name, value);
}

void ConfigManager::putValue(const char* name, float value, bool useprofile)
{     
//     if (useprofile)
//	  gnome_config_set_float(string(_CfgProfile + name).c_str(), value);
//     else
	  gnome_config_set_float(name, value);
}

void ConfigManager::putValue(const char* name, const char* value, bool useprofile)
{
//     if (useprofile)
//	  gnome_config_set_string(string(_CfgProfile + name).c_str(), value);
//    else
	  gnome_config_set_string(name, value);
}

void ConfigManager::putPrivValue(const char* name, const char* value, bool useprofile)
{
//     if (useprofile)
//	  gnome_config_private_set_string(string(_CfgProfile + name).c_str(), value);
//     else
	  gnome_config_private_set_string(name, value);
}

void ConfigManager::saveEntryHistory(Gnome::Entry* gentry)
{
     gentry->save_history();
}

// Value retrievers
string ConfigManager::getStrValue(const char* name, bool useprofile)
{
//     if (useprofile)
//	  return string(gnome_config_get_string(string(_CfgProfile + name).c_str()));
//     else
     char* cval = gnome_config_get_string(name);
     string value(cval);
     g_free(cval);

     return value;
}

string ConfigManager::getPrivStrValue(const char* name, bool useprofile)
{
//     if (useprofile)
//	  return string(gnome_config_private_get_string(string(_CfgProfile + name).c_str()));
//     else
     char* cval = gnome_config_private_get_string(name);
     string value(cval);
     g_free(cval);

     return value;
}

bool ConfigManager::getBoolValue(const char* name, bool useprofile)
{
//     if (useprofile)
//	  return gnome_config_get_bool(string(_CfgProfile + name).c_str());
//     else
	  return gnome_config_get_bool(name);
}

int ConfigManager::getIntValue(const char* name, bool useprofile)
{
//     if (useprofile)
//	  return gnome_config_get_int(string(_CfgProfile + name).c_str());
//     else
	  return gnome_config_get_int(name);
}

float ConfigManager::getFloatValue(const char* name, bool useprofile)
{
//     if (useprofile)
//	  return gnome_config_get_float(string(_CfgProfile + name).c_str());
//     else
	  return gnome_config_get_float(name);
}

void ConfigManager::loadEntryHistory(Gnome::Entry* gentry)
{
     gentry->load_history();
}

// Presence pixmap/color cache
ConfigManager::PresenceInfo::PresenceInfo(const string& status, const Gdk_Drawable& window, ConfigManager& cfgm, const string& defaultcolor)
     : _color(cfgm.getStrValue("Colors/" + status + "=" + defaultcolor))
{
     // Only attempt to load the pixmap if status != unavailable
     if (status != "offline")
     {
	  // Build a table of possible image locations
	  const char* pixpath_tbl[4] = { ConfigManager::get_PIXPATH(), "./", "./pixmaps/", "../pixmaps/" };
	  // Look for and load the pixmap for the specified status...if it exists
	  for (int i = 0; i < 4; i++)
	  {
	       string filename = pixpath_tbl[i] + status + ".xpm";
	       // If we find a pixmap by this name, load it and return
	       if (g_file_exists(filename.c_str()))
	       {
		    _pixmap.create_from_xpm(window, _bitmap, Gdk_Color("white"), filename);
		    return;
	       }
	  }
	  // If this point is reached, no suitable file was found
	  cerr << "AHAGH! Missing pixmap: " << status << endl;
     }
}

void ConfigManager::initPresenceCache(const Gdk_Drawable& window)
{
     // Step 1: Make sure the PI cache is clear
     _PICache.clear();

     // Step 2: Create a table of valid presence status & initial colors (HACK)
     const int   ptable_sz = 7;
     const char* ptable[ptable_sz] = { "online"   , "chat"   , "away"   , "xa"     , "dnd"    , "offline", "stalker" };
     const char* ctable[ptable_sz] = { "#0066CC"  , "#0099CC", "#004D80", "#004D4D", "#666633", "#666666", "#663333" };

     // Step 3: Load the cache by walking the presence status table
     for (int i = 0; i < ptable_sz; i++)
	  _PICache.insert(make_pair(ptable[i], PresenceInfo(ptable[i], window, *this, ctable[i])));
}

GdkPixmap* ConfigManager::getPresencePixmap(const string& status)
{
     map<string, PresenceInfo>::iterator it = _PICache.find(status);
     if (it != _PICache.end())
	  return it->second._pixmap.gdkobj();
     else
	  return NULL;
}

GdkBitmap* ConfigManager::getPresenceBitmap(const string& status)
{
     map<string, PresenceInfo>::iterator it = _PICache.find(status);
     if (it != _PICache.end())
	  return it->second._bitmap;
     else
	  return NULL;
}

GdkColor* ConfigManager::getPresenceColor(const string& status)
{
     map<string, PresenceInfo>::iterator it = _PICache.find(status);
     if (it != _PICache.end())
	  return it->second._color.gdkobj();
     else
	  return NULL;
}

// Event pixmap cache
ConfigManager::EventInfo::EventInfo(const Gdk_Drawable& window, ConfigManager& cfgm, const string& event, const string& pixmap, const string& defaultcolor) : _color(cfgm.getStrValue("Colors/" +  event + "=" + defaultcolor))
{
     // Build a table of possible image locations
     const char* pixpath_tbl[5] = { ConfigManager::get_PIXPATH(), ConfigManager::get_SHAREDIR(), "./", "./pixmaps/", "../pixmaps/" };
     // Look for and load the pixmap for the specified status...if it exists
     for (int i = 0; i < 5; i++)
     {
          string filename = pixpath_tbl[i] + pixmap;
          // If we find a pixmap by this name, load it and return
          if (g_file_exists(filename.c_str()))
          {
               _pixmap.create_from_xpm(window, _bitmap, Gdk_Color("white"), filename);
               return;
          }
     }
     // If this point is reached, no suitable file was found
     cerr << "AHAGH! Missing pixmap: " << pixmap << endl;
}

void ConfigManager::initEventCache(const Gdk_Drawable& window)
{
     // Step 1: Make sure the EI cache is clear
     _EICache.clear();

     // Step 2: Create a table of valid event pixmaps
     const int   ptable_sz = 5;
     const MessageManager::MessageType ctable[ptable_sz] = { MessageManager::mtNormal, MessageManager::mtChat, MessageManager::mtRoster, MessageManager::mtGCI, MessageManager::mtNone };
     const char* ntable[ptable_sz] = { "evMessage", "evChat", "evRoster", "evGCI", ""};
     const char* ptable[ptable_sz] = { "glade-message.xpm" , "glade-ooochat.xpm", "glade-contact.xpm", "glade-gci.xpm", "glade-blank.xpm" };
     const char* ztable[ptable_sz] = { "#669999", "#996699", "#999966", "#999966", "#000000" };

     // Step 3: Load the cache
     for (int i = 0; i < ptable_sz; i++)
     {
          _EICache.insert(make_pair(ctable[i], EventInfo(window, *this, ntable[i], ptable[i], ztable[i])));
     }

}

GdkPixmap* ConfigManager::getEventPixmap(const MessageManager::MessageType& event)
{
     map<MessageManager::MessageType, EventInfo>::iterator it = _EICache.find(event);
     if (it != _EICache.end())
          return it->second._pixmap.gdkobj();
     else
          return NULL;
}

GdkBitmap* ConfigManager::getEventBitmap(const MessageManager::MessageType& event)
{
     map<MessageManager::MessageType, EventInfo>::iterator it = _EICache.find(event);
     if (it != _EICache.end())
          return it->second._bitmap;
     else
          return NULL;
}

GdkColor* ConfigManager::getFlashEventColor(const MessageManager::MessageType& event)
{
     map<MessageManager::MessageType, EventInfo>::iterator it = _EICache.find(event);
     if (it != _EICache.end())
          return it->second._color.gdkobj();
     else
          return NULL;
}

