/*
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 * 
 *  Gabber
 *  Copyright (C) 1999-2001 Dave Smith & Julian Missig
 */


#include "Gabber.hh"
#include "GabberApp.hh"
#include "GabberWin.hh"
#include "AutoAway.hh"

AutoAway::AutoAway()
     : _autoaway(false), _have_xss(false)
{
#ifdef HAVE_XSS
     int event_base, error_base;

     if (XScreenSaverQueryExtension(GDK_DISPLAY(), &event_base, &error_base))
     {
          _scrnsaver_info = XScreenSaverAllocInfo();
	  _have_xss = true;
     }
     else
#endif /* HAVE_XSS */
	  cerr << "X Screen Saver Extension not found, auto-away disabled" << endl;

     G_App->getSession().evtConnected.connect(slot(this, &AutoAway::on_session_connected));
     G_App->getSession().evtDisconnected.connect(slot(this, &AutoAway::on_session_disconnected));
     G_App->getSession().evtOnLast.connect(slot(this, &AutoAway::on_session_last));
}

AutoAway::~AutoAway()
{
}

void AutoAway::on_session_connected(const Tag& t)
{
     _autoaway = false;
     _timer = Gtk::Main::timeout.connect(slot(this, &AutoAway::auto_away_timer), 1000);
}

void AutoAway::on_session_disconnected()
{
     _timer.disconnect();
}

gint AutoAway::auto_away_timer()
{
     unsigned long idle_time = 0;
     ConfigManager& cf = G_App->getCfg();

     if (!cf.getBoolValue("AutoAway/Enabled=true"))
	  return TRUE;

     idle_time = get_idle_time();
   
     Presence::Show show = (Presence::Show)cf.get_show();

     unsigned int aatime = cf.getIntValue("AutoAway/AwayAfter=5");
     unsigned int natime = cf.getIntValue("AutoAway/XAAfter=15");
     if (!aatime && !natime)
          return TRUE;

     // idle_time is in milisecond, assume that natime will be longer than away time
     if (natime && (idle_time > natime * 60000))
     {
	  if (_autoaway || (show == Presence::stOnline))
	       set_away(Presence::stXA);
     }
     else if (aatime && (idle_time > aatime * 60000))
     {
	  if (show == Presence::stOnline)
	       set_away(Presence::stAway);
     }
     else if (_autoaway)
          set_back();

     return TRUE;
}

void AutoAway::set_away(Presence::Show show)
{
     ConfigManager& cf = G_App->getCfg();

     // Only send presence if it's a change
     if (cf.get_show() == show)
	  return;

     // Only set old values if we are changing from online to autoaway
     if (!_autoaway)
     {
	  _oldStatus = cf.get_status();
	  _oldPriority = cf.get_priority();
     }

     string status = cf.getStrValue("AutoAway/Status=Automatically away due to being idle");
     cf.set_show(show);
     cf.set_status(status);
     cf.set_priority(0);
//     cf.sync();

     G_Win->display_status(show, status, 0);

     _autoaway = true;
}

void AutoAway::set_back()
{
     ConfigManager& cf = G_App->getCfg();

     cf.set_show(Presence::stOnline);
     cf.set_status(_oldStatus);
     cf.set_priority(_oldPriority);
//     cf.sync();

     G_Win->display_status(Presence::stOnline, _oldStatus, _oldPriority);
     _autoaway = false;
}

unsigned long AutoAway::get_idle_time()
{
#ifdef HAVE_XSS
     if (_have_xss)
     {
          XScreenSaverQueryInfo(GDK_DISPLAY(), GDK_ROOT_WINDOW(), _scrnsaver_info);
          return _scrnsaver_info->idle;
     }
#endif /* HAVE_XSS */
     // Is there any other way we could try to get idle time?

     return 0;
}

void AutoAway::on_session_last(string& idletime)
{
     unsigned long idle;
     char idlestr[1024];

     // get_idle_time returns time in milliseconds so we have to convert to secs
     idle = get_idle_time() / 1000;
     g_snprintf(idlestr, 1024, "%ld", idle);
     cerr << "IIIII****III***III**II Idle Time is: " << idlestr << endl;
     idletime = idlestr;
}
