/*
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 * 
 *  Gabber
 *  Copyright (C) 1999-2001 Dave Smith & Julian Missig
 */

#include "AddContactDruid.hh"
#include "GladeHelper.hh"
#include "GabberApp.hh"
#include "ConfigManager.hh"
#include "GabberUtility.hh"
#include "AgentInterface.hh"

using namespace jabberoo;
using namespace GabberUtil;

AddContactDruid* AddContactDruid::_Dialog = NULL;

void AddContactDruid::display(Page first_page)
{
     _Dialog = manage(new AddContactDruid(first_page, ""));
}

void AddContactDruid::display(const string& jid)
{
     _Dialog = manage(new AddContactDruid(auNickname, jid));
}

void AddContactDruid::display(Agent& agent)
{
     _Dialog = manage(new AddContactDruid(auSearch, agent));
}

AddContactDruid::AddContactDruid(Page first_page, const string& jid)
     : BaseGabberWindow("AUDruid_win"),
       _agent(0)
{
     // Set JID if we have it already
     if (!jid.empty())
	  _JID = jid;

     init();
     _druid->set_page(*translate_page(first_page));
     _druid->set_buttons_sensitive(0,1,1);
     // Display
     show();
}

AddContactDruid::AddContactDruid(Page first_page, Agent& agent)
     : BaseGabberWindow("AUDruid_win"),
       _agent((Agent*) &agent)
{
     init();
     _druid->set_page(*translate_page(first_page));
     search_buttons_toggle();
     // Display
     show();
}

void AddContactDruid::init()
{
     // Setup pointers for all of the pages
     _druid = getWidget<Gnome::Druid>("AUDruid_druid");
     _druid->cancel.connect(slot(this, &AddContactDruid::on_cancel));
     // Intro page
     _AUDruid_intro     = getWidget<Gnome::DruidPage>("AUDruid_intro");
     // Choose which way to locate a user
     _AUDruid_choice    = getWidget<Gnome::DruidPage>("AUDruid_choice");
     _AUDruid_choice    ->prepare.connect(slot(this, &AddContactDruid::on_Choice_prepare)); 
     _AUDruid_choice    ->next.connect(slot(this, &AddContactDruid::on_Choice_next));
     // Select a specific user
     _AUDruid_normaladd = getWidget<Gnome::DruidPage>("AUDruid_normaladd");
     _AUDruid_normaladd ->prepare.connect(slot(this, &AddContactDruid::on_Normaladd_prepare));
     _AUDruid_normaladd ->next.connect(slot(this, &AddContactDruid::on_Normaladd_next));
     _optTrans          = getOptionMenu("AUDruid_normaladd_transports_opt");
     _barStatus         = getWidget<Gnome::AppBar>("AUDruid_Status_bar");
     _entUsername       = getEntry("AUDruid_normaladd_username_txt");
     _entUsername       ->changed.connect(slot(this, &AddContactDruid::on_Username_changed));
     _entServer         = getEntry("AUDruid_normaladd_server_txt");
     _entServer         ->changed.connect(slot(this, &AddContactDruid::on_Server_changed));
     _have_agents       = false;
     // Select an agent
     _AUDruid_agents    = getWidget<Gnome::DruidPage>("AUDruid_agents");
     _AUDruid_agents    ->prepare.connect(slot(this, &AddContactDruid::on_Agents_prepare));
     _AUDruid_agents    ->next.connect(slot(this, &AddContactDruid::on_Agents_next));
     // Search
     _AUDruid_search    = getWidget<Gnome::DruidPage>("AUDruid_search");
     _AUDruid_search    ->prepare.connect(slot(this, &AddContactDruid::on_Search_prepare));
     _AUDruid_search    ->next.connect(slot(this, &AddContactDruid::on_Search_next));
     _init_interface    = false;
     _have_key          = false;
     _results_clist     = 0;
     _vboxSearch        = getWidget<Gtk::VBox>("AUDruid_search_vbox");
     _lblLoading        = getLabel("AUDruid_search_loading_lbl");
     // Search Results
     _AUDruid_searchresults = getWidget<Gnome::DruidPage>("AUDruid_searchresults");
     _AUDruid_searchresults ->next.connect(slot(this, &AddContactDruid::on_SearchResults_next));
     _results_frame     = getWidget<Gtk::Frame>("AUDruid_searchresults_results_frame");
     _results_scroll    = getWidget<Gtk::ScrolledWindow>("AUDruid_searchresults_results_scroll");
     // Edit User Nickname
     _AUDruid_nickname  = getWidget<Gnome::DruidPage>("AUDruid_nickname");
     _AUDruid_nickname  ->prepare.connect(slot(this, &AddContactDruid::on_Nickname_prepare));
     _AUDruid_nickname  ->next.connect(slot(this, &AddContactDruid::on_Nickname_next));
     _AUDruid_nickname  ->back.connect(slot(this, &AddContactDruid::on_Nickname_prev));
     _entNickname       = getEntry("AUDruid_nickname_Nickname_txt");
     _entNickname       ->changed.connect(slot(this, &AddContactDruid::on_Nickname_changed));;
     // Confirm Information
     _AUDruid_confirm   = getWidget<Gnome::DruidPage>("AUDruid_confirm");
     _AUDruid_confirm   ->prepare.connect(slot(this, &AddContactDruid::on_Confirm_prepare));
     _AUDruid_confirm   ->next.connect(slot(this, &AddContactDruid::on_Confirm_next));
     _AUDruid_confirm   ->finish.connect(slot(this, &AddContactDruid::on_finish));
     _txtRequest        = getWidget<Gtk::Text>("AUDruid_s10nrequest_txt");
     getCheckButton("AUDruid_confirm_Again_chk")->toggled.connect(slot(this, &AddContactDruid::on_AddAgain_toggled));
     // Finish
     _AUDruid_finish    = getWidget<Gnome::DruidPage>("AUDruid_finish");
     _AUDruid_finish    ->finish.connect(slot(this, &AddContactDruid::on_finish));

     _lastPage          = 0;

     // Grab the history and put it in the gnome entries (a cooler combo box)
     G_App->getCfg().loadEntryHistory(getWidget<Gnome::Entry>("AUDruid_normaladd_server_gent"));
}

void AddContactDruid::on_cancel()
{
     close();
}
 
void AddContactDruid::on_Choice_prepare()
{
     _druid->set_buttons_sensitive(0,1,1);
}
    
bool AddContactDruid::on_Choice_next()
{
     // Depending upon which AUDruid_*_rdo they have selected, redirect
     if (getCheckButton("AUDruid_jabberid_rdo")->get_active())
	  _druid->set_page(*_AUDruid_normaladd);
     else if(getCheckButton("AUDruid_agent_rdo")->get_active())
     {
	  _druid->set_page(*_AUDruid_agents);
	  // They want to search agents, so close this window
	  // and run the agents browser
//	  AgentBrowserDlg::execute();
//	  close();
     }
     return true;
}

void AddContactDruid::on_Normaladd_prepare()
{
     // Setup the transports for the server they're using, and fill the list with that

     // Prepend the server in their config file
     ConfigManager& c = G_App->getCfg();
     string theirserver = c.getStrValue("Server/Server=jabber.com");
     getWidget<Gnome::Entry>("AUDruid_normaladd_server_gent")->prepend_history(0, theirserver);
     getEntry("AUDruid_normaladd_server_txt")->set_text("");
     _lastPage = _AUDruid_normaladd;

     if (!_have_agents)
     {
          // Setup the list of transports
	  int item = 0;
          _menuTrans.add_item("Jabber", item++);
          _jidlist.push_back(c.getStrValue("Server/Server=jabber.com"));

	  // query all transports on the roster to add to the transport select menu
	  Roster::iterator it = G_App->getSession().roster().begin();
	  for ( ; it != G_App->getSession().roster().end(); it++)
	  {
	       if (G_App->isTransport(it->getJID()))
	       {
	            string id = G_App->getSession().getNextID();
	            Tag iq("iq");
	            iq.putAttrib("id", id);
	            iq.putAttrib("to", JID::getHost(it->getJID()));
	            iq.putAttrib("type", "get");
	            Tag& query = iq.addTag("query");
	            query.putAttrib("xmlns", "jabber:iq:agent");

	            // Send the query
	            G_App->getSession() << iq.getXML().c_str();
	            G_App->getSession().registerIQ(id, slot(this, &AddContactDruid::on_agent_reply));

		    // Add the jid to the list in case the agent doesn't reply to the query
		    _menuTrans.add_item(JID::getHost(it->getJID()), item++);
		    _jidlist.push_back(JID::getHost(it->getJID()));
	       }
	  }
	  // Also send a jabber:iq:agents query to server to try to get some of the transports
	  // since very few of them actually reply to jabber:iq:agent queries
	  G_App->getSession().queryNamespace("jabber:iq:agents", slot(this, &AddContactDruid::on_agents_reply), G_App->getCfg().getStrValue("Server/Server=jabber.com"));

          _barStatus->pop(); 
          _barStatus->push(_("Loading list of agents..."));
          _barStatus->get_progress()->set_activity_mode(true);

	  _menuTrans.selected.connect(slot(this, &AddContactDruid::get_trans_JID));

	  _optTrans->set_menu(_menuTrans.get_menu());
     }

     // They can't click next until they at least enter a username
     _druid->set_buttons_sensitive(1,0,1);
}

bool AddContactDruid::on_Normaladd_next()
{
     string username = toUTF8(_entUsername, _entUsername->get_text());
     string server = toUTF8(_entServer, _entServer->get_text());

     // Remove spaces and @'s
     for (string::size_type i = 0; i < username.length(); i++)
     {
	  if (username[i] == ' ') // If character at i is a space
	       username.erase(i, 1); // Erase the character at i
	  else if (username[i] == '@') // If character at i is @
	       username.replace(i, 1, '%'); // Replace the character at i
     }

     _JID = username + "@" + server;

     // Save the server gnome entry history
     G_App->getCfg().saveEntryHistory(getWidget<Gnome::Entry>("AUDruid_normaladd_server_gent"));

     // Jump to Edit User Nickname page
     _druid->set_page(*_AUDruid_nickname);

     return true;
}

void AddContactDruid::on_Nickname_prepare()
{
     // Set the JID label
     Gtk::Label* l = getLabel("AUDruid_nickname_JID");
     l->set_text(fromUTF8(l, _JID));

     // - Grab vCard
     // Get next session ID
     string id = G_App->getSession().getNextID();

     // Construct vCard request
     Tag iq("iq");
     iq.putAttrib("id", id);
     iq.putAttrib("to", _JID);
     iq.putAttrib("type", "get");
     Tag& vCard = iq.addTag("vCard");
     vCard.putAttrib("xmlns", "vcard-temp");
     vCard.putAttrib("version", "3.0");
     vCard.putAttrib("prodid", "-//HandGen//NONSGML vGen v1.0//EN");

     // Send the vCard request
     G_App->getSession() << iq.getXML().c_str();
     G_App->getSession().registerIQ(id, slot(this, &AddContactDruid::get_vCard));

     on_Nickname_changed();
}

void AddContactDruid::get_vCard(const Tag& t)
{
     if (t.hasChildren())
     {
	  string curtext;
	  Tag* vCard = t.getTag("vCard");
	  if (vCard != NULL)
	  {
	       curtext = vCard->getTaggedCDATA("NICKNAME");
	       if (_entNickname->get_text().empty())
		    _entNickname->set_text(fromUTF8(_entNickname, curtext));
	       curtext = vCard->getTaggedCDATA("FN");
	       Gtk::Label* l = getLabel("AUDruid_nickname_FullName_lbl");
	       l->set_text(fromUTF8(l, curtext));
	       curtext = vCard->getTaggedCDATA("EMAIL");
	       l = getLabel("AUDruid_nickname_eMail_lbl");
	       l->set_text(fromUTF8(l, curtext));
	       curtext = vCard->getTaggedCDATA("URL");
	       Gnome::HRef* hrefWeb = getWidget<Gnome::HRef>("AUDruid_nickname_WebSite_href");
	       hrefWeb->set_url(fromUTF8(hrefWeb, curtext));
	       hrefWeb->set_label(fromUTF8(hrefWeb, curtext));
	       Tag* ADR = vCard->getTag("ADR");
	       if (ADR != NULL)
	       {
		    curtext = ADR->getTaggedCDATA("COUNTRY");
		    l = getLabel("AUDruid_nickname_Country_lbl");
		    l->set_text(fromUTF8(l, curtext));
	       }
	  }
     }
     else
	  // Set the nickname to the username, if they haven't already set the nickname
	  if (_entNickname->get_text().empty())
	       _entNickname->set_text(_entUsername->get_text());
}

bool AddContactDruid::on_Nickname_next()
{
     // Set the nickname
     _nickname = toUTF8(_entNickname, _entNickname->get_text());

     // Enable back, since they can go back here
     _druid->set_page(*_AUDruid_confirm);
     _druid->set_buttons_sensitive(1,1,1);

     return true;
}

bool AddContactDruid::on_Nickname_prev()
{
     // _lastPage should be set
     g_return_val_if_fail(_lastPage != 0, false);

     // go back to the proper page
     _druid->set_page(*_lastPage);
     return true;
}

void AddContactDruid::on_Confirm_prepare()
{
     getCheckButton("AUDruid_confirm_Again_chk")->set_active(false);
     on_AddAgain_toggled();
     Gtk::Label* l = getLabel("AUDruid_confirm_JID");
     l->set_text(fromUTF8(l, _JID));
     l = getLabel("AUDruid_confirm_Nickname");
     l->set_text(fromUTF8(l, _nickname));
}

bool AddContactDruid::on_Confirm_next()
{
     // Detect whether we should start again
     Gtk::CheckButton* check = getCheckButton("AUDruid_confirm_Again_chk");
     if (check->get_active())
     {
	  on_finish();
	  _entUsername->set_text("");
	  _entNickname->set_text("");
	  _entServer->set_text("");
	  getWidget<Gnome::Entry>("AUDruid_normaladd_server_gent")->set_sensitive(true);
	  _optTrans->set_history(0);
	  _init_interface = false;
	  _currently_searchresults = false;
	  _druid->set_page(*_AUDruid_choice);
     }
     else
	  _druid->set_page(*_AUDruid_finish);

     return true;
}

void AddContactDruid::on_finish()
{
     if (_nickname != "" && _JID != "")
     {
	  // Add the user to the roster
	  G_App->getSession().roster() << Roster::Item(_JID, _nickname);
	  if (_txtRequest->get_chars(0, -1) != "")
	       G_App->getSession() << Presence(_JID, Presence::ptSubRequest, Presence::stInvalid, toUTF8(_txtRequest, _txtRequest->get_chars(0, -1)));
	  else
	       G_App->getSession() << Presence(_JID, Presence::ptSubRequest);

	  // Display group management
	  if (getCheckButton("AUDruid_confirm_Groups_chk")->get_active())
	       G_App->manage_groups = true;

	  // Close the druid
	  if (!getCheckButton("AUDruid_confirm_Again_chk")->get_active())
	       close();
     }
}

Gnome::DruidPage* AddContactDruid::translate_page(Page given_page)
{
     switch(given_page)
     {
     case auIntro:
	  return _AUDruid_intro;
     case auChoice:
	  return _AUDruid_choice;
     case auNormaladd:
	  return _AUDruid_normaladd;
     case auAgents:
	  return _AUDruid_agents;
     case auSearch:
	  return _AUDruid_search;
     case auNickname:
	  return _AUDruid_nickname;
     }
     return NULL;
}

void AddContactDruid::on_Username_changed()
{
     // If server and username are filled in
     if (!_entServer->get_text().empty() && !_entUsername->get_text().empty())
	  _druid->set_buttons_sensitive(1,1,1);
     else
	  _druid->set_buttons_sensitive(1,0,1);
}

void AddContactDruid::on_Server_changed()
{
     // If server and username are filled in
     if (!_entServer->get_text().empty() && !_entUsername->get_text().empty())
	  _druid->set_buttons_sensitive(1,1,1);
     else
	  _druid->set_buttons_sensitive(1,0,1);
}

void AddContactDruid::on_Nickname_changed()
{
     // If nickname is filled in, allow them to go forward
     if (!_entNickname->get_text().empty())
     {
	  if (_currently_searchresults)
	       // They were just at search results, allow them to go back
	       _druid->set_buttons_sensitive(1,1,1);
	  else
	       // Disable back, since that wouldn't bring them back to the proper page
	       _druid->set_buttons_sensitive(0,1,1);
     }
     else
     {
	  if (_currently_searchresults)
	       // They were just at search results, allow them to go back
	       _druid->set_buttons_sensitive(1,0,1);
	  else
	       // Disable back, since that wouldn't bring them back to the proper page
	       _druid->set_buttons_sensitive(0,0,1);
     }
}

void AddContactDruid::on_AddAgain_toggled()
{
     if (getCheckButton("AUDruid_confirm_Again_chk")->get_active())
	  _druid->set_show_finish(false);
     else
	  _druid->set_show_finish(true);
}

void AddContactDruid::get_trans_JID(int selected)
{
     _entServer->set_text(fromUTF8(_entServer, _jidlist[selected]));

     // If they selected "Jabber"
     // then enable the server entry
     if (selected == 0)
          getWidget<Gnome::Entry>("AUDruid_normaladd_server_gent")->set_sensitive(true);
     else
          getWidget<Gnome::Entry>("AUDruid_normaladd_server_gent")->set_sensitive(false);
}


// ---------------------------------------------------------
// SEARCH
// ---------------------------------------------------------

void AddContactDruid::on_Agents_prepare()
{
     // Loading...
     _barStatus->pop();
     _barStatus->push(_("Loading the list of agents..."));
     _barStatus->get_progress()->set_activity_mode(true);

     // Toggle the buttons
     _druid->set_buttons_sensitive(false, false, true);

     // Send request to the server for a filter
     G_App->getSession().queryNamespace("jabber:iq:agents", slot(this, &AddContactDruid::handle_Agents_iq), G_App->getCfg().getStrValue("Server/Server=jabber.com"));
}

void AddContactDruid::on_agent_selected(bool selected, bool registerable, bool searchable, bool gccapable, bool agents)
{
     // Set the sensitivity of the appropriate buttons
     _druid->set_buttons_sensitive(false, searchable, true);
}

void AddContactDruid::handle_Agents_iq(const Tag& iq)
{
     // No longer loading
     _barStatus->pop();
     _barStatus->get_progress()->set_activity_mode(false);

     Gnome::Dialog* d;

     // If we get a good query back, create a new AgentList
     // from the packet and display it in the browser
     if (iq.cmpAttrib("type", "result"))
     {
	  // Extract the query 
	  Tag* query = iq.getTag("query");
	  if (query != NULL)
	  {
	       // Initialize the browser
	       _browserSearch = manage(new AgentBrowser(this, "AUDruid_agents", G_App->getCfg().getStrValue("Server/Server=jabber.com"), AgentList(*query, G_App->getSession())));
	       // Hook up the browser
	       _browserSearch->agent_selected.connect(slot(this, &AddContactDruid::on_agent_selected));
	  }
	  else
	  {
	       d = manage(Gnome::Dialogs::warning(_("Error receiving XML for Agent Browser, see standard output.")));
	       d->set_modal(true); 
	       cerr << "ERROR->Unable to extract query for AgentBrowser: " << iq.getXML() << endl;
	  }
     }
     else
     {
	  d = manage(Gnome::Dialogs::warning(_("Error, this server does not support agents.")));
	  d->set_modal(true);
     }
}

bool AddContactDruid::on_Agents_next()
{
     _agent = _browserSearch->get_current_agent();
     return false;
}

void AddContactDruid::on_Search_prepare()
{
     // Set the agent labels
     getLabel("AUDruid_search_agent_lbl")->set_text(_agent->JID());
     getLabel("AUDruid_searchresults_agent_lbl")->set_text(_agent->JID());

     _barStatus->pop();
     _barStatus->push(_("Loading information from agent..."));
     _barStatus->get_progress()->set_activity_mode(true);

     // Set the search page
     _currently_searchresults = false;

     // Toggle the buttons
     search_buttons_toggle();

     // Send query to the agent
     _agent->requestSearch(slot(this, &AddContactDruid::handle_search_request_reply));
}

void AddContactDruid::handle_search_request_reply(const Tag& iq)
{
     if (iq.cmpAttrib("type", "result"))
     {
          // Extract the query 
          Tag* query = iq.getTag("query");
	  if (_init_interface)
	  {
               _key = query->getTaggedCDATA("key");
               _have_key = true;
	  }
	  else 
	  {
               ElementList::const_iterator it = query->getChildren().begin();

               _field_names.push_back("JID");

	       // Create fields table
	       Gtk::Table* fld_tbl = manage(new Gtk::Table(query->getChildren().size()-1, 2));
	       fld_tbl->set_row_spacings(4);
	       fld_tbl->set_col_spacings(4);
	       fld_tbl->set_border_width(0);
	       // Add the table to the vbox
	       _vboxSearch->pack_start(*fld_tbl, true, true, 8);
	       int row = 0;

               for (; it != query->getChildren().end(); it++)
               {
                    // Cast the child element into a tag
                    Tag& t = *static_cast<Tag*>(*it);

                    if (t.getName() == "instructions")
	            {
                         _lblLoading->set_text(fromUTF8(_lblLoading, t.getData()));
                    }
	            else if (t.getName() == "key")
	            {
                         // Jabber docs say key is not necessary for searches but some servers complain
                         // without it.  
                         _key = t.getData();
			 _have_key = true;
                    }
	            else
	            {
			 // Create a label
			 Gtk::Label* lbl = manage(new Gtk::Label());
			 lbl->set_text(fromUTF8(lbl, t.getName() + ":"));
			 lbl->set_justify(GTK_JUSTIFY_RIGHT);
			 lbl->set_alignment(1.0, 0.0);
			 fld_tbl->attach(*lbl, 0, 1, row, row+1,GTK_FILL,GTK_FILL);

			 // Create entry
			 Gtk::Entry* entry = manage(new Gtk::Entry());
			 fld_tbl->attach(*entry, 1, 2, row, row+1,GTK_EXPAND|GTK_FILL,0);
		         _field_names.push_back(t.getName());
		         string* fieldname = &(_field_names.back());
                         entry->set_data("fieldname", fieldname);
                         _entrys.push_back(entry);

			 row++;
		    }
               }
	       _vboxSearch->show_all();
               _results_clist = manage(new Gtk::CList(Gtk::SArray(_field_names)));
	       _results_clist->select_row.connect(slot(this, &AddContactDruid::on_SearchResults_select_row));
	       for (unsigned int i = 0; i < _field_names.size(); i++)
                    _results_clist->set_column_auto_resize(i, true);
	       _results_clist->set_selection_mode(GTK_SELECTION_SINGLE);
	       _results_clist->show();
	       _results_scroll->add(*_results_clist);

	       _init_interface = true;
	  }
	  _barStatus->pop();
	  _barStatus->push(_("Information loaded."));
	  _barStatus->get_progress()->set_activity_mode(false);
	  search_buttons_toggle();
     }
     else
     {
          Gnome::Dialog* d = manage(Gnome::Dialogs::warning(_("Error attempting to search agent.")));
          d->set_modal(true);
          close();
     }
}

bool AddContactDruid::on_Search_next()
{
     EntryList::const_iterator eit = _entrys.begin();

     // Get next session ID
     string id = G_App->getSession().getNextID();

     Tag iq("iq");
     iq.putAttrib("id", id);
     iq.putAttrib("to", _agent->JID());
     iq.putAttrib("type", "set");

     Tag& query = iq.addTag("query");
     query.putAttrib("xmlns", "jabber:iq:search");
     if (!_key.empty())
	  query.addTaggedCDATA("key", _key);

     for (; eit != _entrys.end(); eit++)
     {
          Gtk::Entry* entry = static_cast<Gtk::Entry*>(*eit);
	  string& field = *static_cast<string *>(entry->get_data("fieldname"));

	  if (!entry->get_text().empty()) {
               query.addTaggedCDATA(field, toUTF8(entry, entry->get_text()));
	  }
     }
     G_App->getSession() << iq.getXML().c_str();
     G_App->getSession().registerIQ(id, slot(this, &AddContactDruid::on_search_reply));

     _have_key = false;

     // Don't request a new key - it'll grab one when they click back
     //_agent->requestSearch(slot(this, &AddContactDruid::handleSearchRequestReply));

     // clear clist so old search results are confused with new results
     _results_clist->clear();
     for (unsigned int i = 0; i < _results_clist->columns().size(); i++)
	  _results_clist->set_column_resizeable(i, false);
     _barStatus->pop();
     _barStatus->push(_("Searching..."));
     _barStatus->get_progress()->set_activity_mode(true);

     // We're on search results
     _currently_searchresults = true;
     search_buttons_toggle();
     return false;
}

void AddContactDruid::on_search_reply(const Tag& iq)
{
     _results_clist->freeze();
     _results_clist->clear();
     if (iq.cmpAttrib("type", "result"))
     {
	  // We can recv multiple query tags each of which can contain multiple item tags
          ElementList::const_iterator qit = iq.getChildren().begin();
	  for (; qit != iq.getChildren().end(); qit++)
	  {
               Tag& query = *static_cast<Tag*>(*qit);
	       ElementList::const_iterator ait = query.getChildren().begin();
	       
	       for (; ait != query.getChildren().end(); ait++)
	       {
	            Tag& item = *static_cast<Tag*>(*ait);
		    
		    if (item.getName() == "item")
		    {
                         StringList::iterator sit = _field_names.begin();
		         const char **rowdata;
		         int num_fields = _results_clist->columns().size(), i;

		         rowdata = new const char * [num_fields];
			 rowdata[0] = g_strdup(fromUTF8(_results_clist, item.getAttrib("jid")).c_str());
			 // First field_name is JID which we get from the item tag
		         for (i = 1, sit++; sit != _field_names.end(); sit++, i++)
		         {
                              string* field = &(*sit);
			      
			      rowdata[i] = g_strdup(fromUTF8(_results_clist, item.getTaggedCDATA(field->c_str())).c_str());
		         }
		         _results_clist->append_row(rowdata);
			 for (i = 0; i < num_fields; i++)
                              g_free((gpointer)rowdata[i]);
		         delete [] rowdata;
	            }
	       }
	  }
     }
     else if (iq.cmpAttrib("type", "error"))
     {
	  Tag* error = iq.getTag("error");
	  string errormessage = string(_("Error ")) 
	       + string(error->getAttrib("code")) 
	       + string(":\n") 
	       + string(iq.getTaggedCDATA("error"));
	  Gnome::Dialog* d = manage(Gnome::Dialogs::warning(errormessage.c_str()));
	  d->set_modal(true);
     }
     
     _results_clist->thaw();
     for (unsigned int i = 0; i < _results_clist->columns().size(); i++)
	  _results_clist->set_column_resizeable(i, true);
     _results_frame->show();
     _barStatus->pop();
     _barStatus->push(_("Search completed."));
     _barStatus->get_progress()->set_activity_mode(false);
     search_buttons_toggle();
     // Request a new key
     //_agent->requestSearch(slot(this, &AddContactDruid::handleSearchRequestReply));
}

void AddContactDruid::on_SearchResults_select_row(int row, int col, GdkEvent* e)
{
     search_buttons_toggle();
     // If they double-clicked, act like they clicked next
     if (e->type == GDK_2BUTTON_PRESS)
	  on_SearchResults_next();
}

bool AddContactDruid::on_SearchResults_next()
{
     if (_results_clist->selection().begin() != _results_clist->selection().end()) 
     {
          _results_clist->get_text(_results_clist->selection().begin()->get_row_num(), 0, _JID);
	  _lastPage = _AUDruid_searchresults;
	  _druid->set_page(*_AUDruid_nickname);
	  return true;
     }
     else
     {
	  _druid->set_page(*_AUDruid_searchresults);
	  return true;
     }
}

void AddContactDruid::search_buttons_toggle()
{
     // If we're on the search page
     if (!_currently_searchresults)
     {
	  // If we've received the field
	  if (_init_interface && (_have_key || _key.empty()))   /* either they have the key or _key is null - not both */
	       // Allow them to click next, but not back
	       _druid->set_buttons_sensitive(0,1,1);
	  else 
	       // Don't allow next to be clicked
	       _druid->set_buttons_sensitive(0,0,1);
     }
     // We're on the search results page
     else
     {
	  // If they have something selected
	  if (_results_clist->selection().begin() != _results_clist->selection().end())
	       // Allow them to click next and back
	       _druid->set_buttons_sensitive(1,1,1);
	  else
	       // Don't allow them to click next
	       _druid->set_buttons_sensitive(1,0,1);
     }
}

void AddContactDruid::on_agent_reply(const Tag& iq)
{
     Tag* query = iq.getTag("query");
     int item = 1;

     if (iq.cmpAttrib("type", "result")) 
     {
          string jid = iq.getAttrib("jid");
	  Tag* service = query->getTag("service");

	  // Find menuitem in option menu so the agent jid can be replaced with the service name
	  Gtk::MenuShell::MenuList::iterator it = _menuTrans.get_menu()->items().begin();
	  vector<string>::iterator lit = _jidlist.begin();
	  int num = 0;
	  // don't bother to look if there was no service tag since there isn't anything to replace it with
	  for ( ; (lit != _jidlist.end()) && service; lit++, it++, num++)
	  {
	       if (jid == *lit)
	       {
		    // create a new menu item with the service as the label
	            Gtk::MenuItem* item = manage(new Gtk::MenuItem (service->getData()));
		    item->activate.connect(SigC::bind(_menuTrans.selected.slot(), num));
		    item->show_all ();

		    // add the new menu item and remove the old
		    _menuTrans.get_menu()->items().insert(it, item);
		    _menuTrans.get_menu()->items().erase(it);
	       }
	  }
          Gtk::OptionMenu* optTrans = getWidget<Gtk::OptionMenu>("AUDruid_normaladd_transports_opt");
          optTrans->set_menu(*_menuTrans.get_menu());

          _barStatus->pop();
          _barStatus->get_progress()->set_activity_mode(false);

          _have_agents = true;
     }
     else
     {
          _barStatus->pop();
	  _barStatus->push(_("Error loading list of agents."));
	  _barStatus->get_progress()->set_activity_mode(false);
     }
}

void AddContactDruid::on_agents_reply(const Tag& iq)
{
     Tag* query = iq.getTag("query");
     ElementList::const_iterator it = query->getChildren().begin();
     int item = 1;

     if (iq.cmpAttrib("type", "result")) {
          for (; it != query->getChildren().end(); it++)
          {
               Tag& agent = *static_cast<Tag*>(*it);

               if (agent.getName() != "agent")
                    continue;

               string jid = agent.getAttrib("jid");
	       Tag* service = agent.getTag("service");

	       // Find the entry in the options menu corresponding to the agent
	       Gtk::MenuShell::MenuList::iterator it = _menuTrans.get_menu()->items().begin();
	       vector<string>::iterator lit = _jidlist.begin();
	       int num = 0;
	       // don't bother looking if there was no service tag since without it there's
	       // nothing to change in the options menu
	       for ( ; (lit != _jidlist.end()) && service; lit++, it++, num++)
	       {
	            if (jid == *lit)
	            {
			 // create a new menuitem with the service name as the label
	                 Gtk::MenuItem* item = manage(new Gtk::MenuItem (service->getData()));
	                 item->activate.connect(SigC::bind(_menuTrans.selected.slot(), num));
	                 item->show_all ();

			 // add the new menuitem to the option menu and remove the old one
                         _menuTrans.get_menu()->items().insert(it, item);
                         _menuTrans.get_menu()->items().erase(it);
	            }
	       }
          }
          Gtk::OptionMenu* optTrans = getWidget<Gtk::OptionMenu>("AUDruid_normaladd_transports_opt");
          optTrans->set_menu(_menuTrans.get_menu());

          _barStatus->pop();
          _barStatus->get_progress()->set_activity_mode(false);

          _have_agents = true;
     }
     else
     {
          _barStatus->pop();
          _barStatus->push(_("Error loading list of agents."));
          _barStatus->get_progress()->set_activity_mode(false);
     }
}
