/********************************************************************************
 *   Jabberoo/Judo -- C++ Jabber Library                                        *
 *                                                                              * 
 *   Copyright (C) 1999-2000 Dave Smith (dave@jabber.org)                       *
 *                                                                              *
 *   This library is free software; you can redistribute it and/or              *
 *   modify it under the terms of the GNU Lesser General Public                 *
 *   License as published by the Free Software Foundation; either               *
 *   version 2.1 of the License, or (at your option) any later version.         *
 *                                                                              *
 *   This library is distributed in the hope that it will be useful,            *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of             *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU          *
 *   Lesser General Public License for more details.                            *
 *                                                                              *
 *   You should have received a copy of the GNU Lesser General Public           *
 *   License along with this library; if not, write to the Free Software        *
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA  *
 ********************************************************************************
 */

#include "judo.hh"

using namespace judo;

// *************************************************************************
// INTERNAL
// 
// ElementPredicates
// *************************************************************************
class P_TypeEq : public ElementPredicate
{
    Element::Type _type;
public:
    explicit P_TypeEq(const Element::Type e) : _type(e) {}
    bool operator()(const Element* el) const {return *el == _type; }
};

class P_NameEq : public ElementPredicate
{
     string _name;
public:
     explicit P_NameEq(const char* name) : _name(name) {}
     bool operator()(const Element* el) const { return *el == _name; }
};

class P_NamespaceEq : public ElementPredicate
{
     string _name;
     string _xmlns;
public:
     explicit P_NamespaceEq(const char* name, const char* xmlns) : _name(name), _xmlns(xmlns) {}
     bool operator()(const Element* el) const { 
	  return ((*el == _name) && (static_cast<const Tag*>(el)->cmpAttrib("xmlns", _xmlns))); 
     }
};    

class AOP_ElementListCopy : public unary_function<Element*, void>
{
public:
     explicit AOP_ElementListCopy(ElementList& l) : _l(l) {}
     void operator()(Element* el) const { 
	  if (el->getType() == Element::etTag)
	       _l.push_back(new Tag(*static_cast<Tag*>(el)));
	  else if (el->getType() == Element::etCDATA)
	       _l.push_back(new CDATA(*static_cast<CDATA*>(el)));
     }
private:
     ElementList& _l;
};

void AOP_ElementDelete(Element* e)
{
    delete e;
}

class XML_accumulator
{
public:
    XML_accumulator(string& s) : _result(s) {}
     void operator()(Element* e) { _result += e->getXML(); }
     void operator()(pair<string,string> p) { _result += " " + p.first + "='" + escape(p.second) + "'";}
    string& getResult() { return _result; }
private:
    string& _result;
};

// *************************************************************************
//
// Jabberoo::Tag
//
// *************************************************************************

// ---------------------------------------------------------
// Addition ops
// ---------------------------------------------------------
Tag& Tag::addTag(const string& name, const char** attribs)
{
    // Allocate a new tag & store it in the list
    return *addChildElement(new Tag(name, attribs));
}

CDATA& Tag::addCDATA(const char* pCDATA, int iCDATASz)
{
     // If the last child element is CDATA, merge this section
     // with it...
     if ((_Children.size() > 0) && (*_Children.back() == etCDATA))
     {
	  CDATA& c = *static_cast<CDATA*>(_Children.back());
	  c.appendText(pCDATA, iCDATASz);
	  return c;
     }
     else
	  return *addChildElement(new CDATA(pCDATA, iCDATASz));
}

Tag& Tag::addTaggedCDATA(const string& name, const string& value)
{
     Tag& t = addTag(name);
     t.addCDATA(value.c_str(), value.length());
     return t;
}

// ---------------------------------------------------------
// Accessor ops
// ---------------------------------------------------------
const char* Tag::getTaggedCDATA(const char* name) const
{
     Tag* t = getTag(name);
     if (t != NULL)
	  return t->getData();
     else
	  return "";
}

Tag* Tag::getTag(const char* name, const char* xmlns) const
{
     if (xmlns == NULL)
	  return static_cast<Tag*>(findChild(P_NameEq(name)));
     else
	  return static_cast<Tag*>(findChild(P_NamespaceEq(name, xmlns)));
}

const char* Tag::getData() const
{
     // Quick check..
     if (_Children.empty()) 
	  return "";
     CDATA* c = static_cast<CDATA*>(findChild(P_TypeEq(etCDATA)));
     if (c != NULL)
     {
	  return c->getText().c_str();
     }
     return "";    
}

string Tag::getXML() const
{
     string result;
     // Build base name
     result << "<" << getName();
     // Setup a XML accumulator  
     XML_accumulator accumulator(result);
     // Load attributes into the accumulator
     for_each(_Attribs.begin(), _Attribs.end(), accumulator);
     // Load the child nodes into the accumulator
     if (!_Children.empty())
     {
	  result << ">";
	  for_each(_Children.begin(), _Children.end(), accumulator);
	  result << "</" << getName() << ">";	    
     }
     else
     {
	  // Close the tag
	  result << "/>";
     }
     return result;
}


// ---------------------------------------------------------
// Attrib ops
// ---------------------------------------------------------
void Tag::putAttrib(const string& name, const string& value)
{
     // Set value in the _Attribs
     _Attribs[name] = value;
}

const string Tag::getAttrib(const string& name) const
{
     map<string,string>::const_iterator it = _Attribs.find(name);
     if (it != _Attribs.end())
	  return it->second;
     else
	  return "";
}

void Tag::delAttrib(const string& name)
{
     map<string,string>::iterator it = _Attribs.find(name);
     if (it != _Attribs.end())
	  _Attribs.erase(it);
}

bool Tag::cmpAttrib(const string& name, const string& value) const
{
     return (getAttrib(name) == value);
}

// ---------------------------------------------------------
// Iterator ops
// ---------------------------------------------------------
const map<string,string>& Tag::getAttribs() const
{
     return _Attribs;
}

const ElementList& Tag::getChildren() const
{
     return _Children;
}

// ---------------------------------------------------------
// Simple query ops
// ---------------------------------------------------------
bool Tag::hasTags() const
{
    // Walk the child list (if applicable) and find the first
    // tag..
     if (findChild(P_TypeEq(etTag)) != NULL)
	  return true;
     else
	  return false;
}

bool Tag::hasCDATA() const
{
     if (findChild(P_TypeEq(etCDATA)) != NULL)
	  return true;
     else
	  return false;   
}

inline bool Tag::hasChildren() const
{
     return !_Children.empty();
}


// ---------------------------------------------------------
// Initializers
// ---------------------------------------------------------
Tag::Tag(const string& name, const char** attribs)
     : Element(name, etTag)
{
    // Parse out the attributes
    if (attribs != NULL)
    {
        int i = 0;    
        while (attribs[i] != '\0')
        {
	     _Attribs[attribs[i]] = attribs[i+1];
	     i += 2;            
        }
    }
}

Tag::~Tag()
{
     // Release child nodes
     for_each(_Children.begin(), _Children.end(), AOP_ElementDelete);
}

Tag::Tag(const Tag& t)
     : Element(t.getName(), t.getType()),
       _Attribs(t._Attribs)
{
     // Copy each individual element in _Children
     for_each(t._Children.begin(), t._Children.end(), AOP_ElementListCopy(_Children));
}

Tag& Tag::operator=(const Tag& t)
{
     // Check for assignment to self
     if (this == &t)
	  return *this;
     // Copy underlying members
     Element::operator=(t);

     // Release pre-existing children/attribs if necessary
     if (!_Children.empty()) {
	  // Release child nodes
	  for_each(_Children.begin(), _Children.end(), AOP_ElementDelete);
	  _Children.clear();
     }
     if (!_Attribs.empty()) 
	  _Attribs.clear();

     // Copy children/attribs from t
     if (!t._Children.empty())
	  // Copy each individual element in t._Children
	  for_each(t._Children.begin(), t._Children.end(), AOP_ElementListCopy(_Children)); 
     if (!t._Attribs.empty())
	  _Attribs = t._Attribs;

     return *this;
}

// ---------------------------------------------------------
// Internal/private methods
// ---------------------------------------------------------
template <class T>
inline T* Tag::addChildElement(T* t)
{
     _Children.push_back(t);
     return t;
}

template <class T>
inline Element* Tag::findChild(const T& e) const
{
     if (_Children.empty())
	  return NULL;
     ElementList::const_iterator it = find_if(_Children.begin(), _Children.end(), e);
     if (it != _Children.end())
	  return *it;
     else
	  return NULL;
}
