/**********************************************************************
Copyright (C) 1996, 1997, 1998 Christopher Lee
Copyright (C) 2000 Rob Browning
 
This program is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2, or (at your option)
any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.
 
You should have received a copy of the GNU General Public License
along with this software; see the file COPYING.  If not, write to
the Free Software Foundation, 675 Mass Ave, Cambridge, MA 02139, USA.
**********************************************************************/

#include <guile/gh.h>
#include <libguile.h>
#include "g-wrap-runtime-guile.h"

#ifdef HAVE_CONFIG_H
# include "../conf.h"
#endif

#include <string.h>

#ifndef SCM_SMOB_DATA
#define SCM_SMOB_DATA(x) SCM_CDR((x))
#endif

#ifndef SCM_NEWSMOB
#define GW_NEWSMOB(smob, id, data) \
  SCM_NEWCELL((smob)); \
  SCM_CAR((smob)) = (id); \
  SCM_CDR((smob)) = (data);
#else
#define GW_NEWSMOB SCM_NEWSMOB
#endif

#ifndef SCM_RETURN_NEWSMOB
#define GW_RETURN_NEWSMOB(id, data) \
  do { \
    SCM __SCM_smob_answer; \
    GW_NEWSMOB(__SCM_smob_answer, (id), (data)); \
       return __SCM_smob_answer; \
  } while (0)
#else
#define GW_RETURN_NEWSMOB SCM_RETURN_NEWSMOB
#endif

void
gwp_runtime_get_version_info(int *major, int *revision, int *age) {
  *major = GW_RUNTIME_GUILE_INTERFACE_MAJOR_VER;
  *revision = GW_RUNTIME_GUILE_INTERFACE_REVISION;
  *age = GW_RUNTIME_GUILE_INTERFACE_AGE;
}

void
gw_puts(const char* str, SCM port) {
#if HAVE_SCM_PUTS
  scm_puts(str,port);
#else
  scm_gen_puts(scm_mb_string,str,port);
#endif
}

/****************************************************************************/
/* Wrapped C type/pointer info */

typedef struct {
  SCM name;
  SCM (*equal_p)(SCM wcp_a, SCM wcp_b);
  int (*print)(SCM wcp, SCM port, char writing_p, int *use_default_printer);
  SCM (*mark)(SCM wcp);
  scm_sizet (*cleanup)(SCM wcp);
} wrapped_c_type_data;

typedef struct {
  SCM type;
  void *pointer;
  SCM scm_data;
} wrapped_c_pointer_data;

static int wct_system_initialized = 0;
static long wct_smob_id = 0;
static long wcp_smob_id = 0;

/* forward defs */

#ifndef SCM_SMOB_PREDICATE
# define SCM_SMOB_PREDICATE(tag, obj) \
  (SCM_NIMP(obj) && SCM_TYP16 (obj) == (tag))
#endif

#define GW_WCT_P(obj) (SCM_SMOB_PREDICATE((wct_smob_id), (obj)))
#define GW_WCP_P(obj) \
  ((obj == SCM_BOOL_F) || (SCM_SMOB_PREDICATE((wcp_smob_id), (obj))))

int
gw_wct_p(SCM obj) {
  return(GW_WCT_P(obj));
}

int
gw_wcp_p(SCM obj) {
  return(GW_WCP_P(obj));
}

/****************************************************************************/
/* Wrapped C pointer functions */

static scm_sizet
wcp_data_free(SCM wcp) {
  scm_sizet maybe_total_reclaimed = 0;
  wrapped_c_pointer_data *data;
  wrapped_c_type_data *type_data;
  
  data = (wrapped_c_pointer_data *) SCM_SMOB_DATA(wcp);
  type_data = (wrapped_c_type_data *) SCM_SMOB_DATA(data->type);

  if(type_data->cleanup) {
    maybe_total_reclaimed += type_data->cleanup(wcp);
    /* c pointer may be destrotyed at this point (probably should be) */
  }

  free(data);
  maybe_total_reclaimed += sizeof(wrapped_c_pointer_data); 
  return maybe_total_reclaimed;
}

static int
wcp_data_print(SCM wcp, SCM port, scm_print_state *pstate) {
  char endstr[512];
  int result;
  int use_default_p = 1;
  int writing_p = SCM_WRITINGP(pstate);
  wrapped_c_pointer_data *data;
  wrapped_c_type_data *type_data;
  
  data = (wrapped_c_pointer_data *) SCM_SMOB_DATA(wcp);
  if(!GW_WCT_P(data->type)) {
    scm_misc_error("wcp_data_print", "Unknown type object.", data->type);
  }
  type_data = (wrapped_c_type_data *) SCM_SMOB_DATA(data->type);

  if(type_data->print) {
    use_default_p = 0;
    result = type_data->print(wcp, port, writing_p, &use_default_p);
  }
  
  if(use_default_p) {
    snprintf(endstr, sizeof(endstr), " %p>", data->pointer);
    gw_puts("#<gw:wcp ", port);
    scm_display(type_data->name, port);
    gw_puts(endstr, port);
    result = 1;
  }
  return result;
}

static SCM
wcp_data_mark(SCM wcp) {
  wrapped_c_pointer_data *data;
  wrapped_c_type_data *type_data;
  
  data = (wrapped_c_pointer_data *) SCM_SMOB_DATA(wcp);
  type_data = (wrapped_c_type_data *) SCM_SMOB_DATA(data->type);

  if(type_data->mark) {
    scm_gc_mark(type_data->mark(wcp));
  }

  scm_gc_mark(data->type);
  return(data->scm_data);
}

static SCM
wcp_data_equal_p(SCM wcp_a, SCM wcp_b) {
  wrapped_c_pointer_data *data_a;
  wrapped_c_pointer_data *data_b;
  wrapped_c_type_data *type_data;
  
  data_a = (wrapped_c_pointer_data *) SCM_SMOB_DATA(wcp_a);
  data_b = (wrapped_c_pointer_data *) SCM_SMOB_DATA(wcp_b);

  if(data_a == data_b) return SCM_BOOL_T;

  if(!gh_eq_p(data_a->type, data_b->type)) return SCM_BOOL_F;
 
  if((data_a->pointer == data_b->pointer)) return SCM_BOOL_T;

  type_data = (wrapped_c_type_data *) SCM_SMOB_DATA(data_a->type);

  if(!type_data->equal_p) return SCM_BOOL_F;

  return type_data->equal_p(wcp_a, wcp_b);
}

SCM
gw_wcp_assimilate_ptr(void *ptr, SCM type) {
  /* create a wrapped C pointer of the given type, wrapping ptr */
  wrapped_c_type_data *type_data;
  wrapped_c_pointer_data *ptr_data; 

  if(!GW_WCT_P(type)) return SCM_BOOL_F;

  type_data = (wrapped_c_type_data *) SCM_SMOB_DATA(type);

  ptr_data = (wrapped_c_pointer_data *)
    scm_must_malloc(sizeof(wrapped_c_pointer_data), "gw:wcp");

  ptr_data->pointer = ptr;
  ptr_data->type = type;
  ptr_data->scm_data = SCM_BOOL_F;

  GW_RETURN_NEWSMOB(wcp_smob_id, ptr_data);
}

void *
gw_wcp_get_ptr(SCM obj) {
  wrapped_c_pointer_data *ptr_data;
  if(!SCM_SMOB_PREDICATE(wcp_smob_id, obj)) return NULL;  
  ptr_data = (wrapped_c_pointer_data *) SCM_SMOB_DATA(obj);
  return(ptr_data->pointer);
}

int
gw_wcp_is_of_type_p(SCM type, SCM obj) {
  /* return non-zero if wrapped C pointer obj is of the given type. */
  if(SCM_SMOB_PREDICATE(wcp_smob_id, obj)) {
    wrapped_c_pointer_data *ptr_data =
      (wrapped_c_pointer_data *) SCM_SMOB_DATA(obj);
    return(gh_eq_p(ptr_data->type, type));
  }
  return 0;
}

SCM
gw_wcp_coerce(SCM obj, SCM new_type) {
  /* return a new wrapped C pointer */
  wrapped_c_pointer_data *ptr_data;

  if(!SCM_SMOB_PREDICATE(wcp_smob_id, obj)) return SCM_BOOL_F;
  if(!GW_WCT_P(new_type)) return SCM_BOOL_F;
  
  return(gw_wcp_assimilate_ptr(gw_wcp_get_ptr(obj),
                               new_type));
}

/****************************************************************************/
/* Wrapped C type functions */

static scm_sizet 
wct_data_free(SCM smob) {
  wrapped_c_type_data *data = (wrapped_c_type_data *) SCM_SMOB_DATA(smob);
  free(data);
  return sizeof(wrapped_c_type_data);
}

static SCM
wct_data_mark(SCM smob) {
  wrapped_c_type_data *data = (wrapped_c_type_data *) SCM_SMOB_DATA(smob);
  return(data->name);
}

static int
wct_data_print(SCM wct, SCM port, scm_print_state *pstate) {
  int writing_p = SCM_WRITINGP(pstate);
  wrapped_c_type_data *data = (wrapped_c_type_data *) SCM_SMOB_DATA(wct);

  gw_puts("#<gw:wct ", port);
  scm_display(data->name, port);
  gw_puts(">", port);
  return 1;
}

static void
initialize_wct_type_system()
{
  if(!wct_system_initialized) {

#ifdef GWRAP_OLD_GUILE_SMOB
    {
      scm_smobfuns wct_smob_data = {
        wct_data_mark,
        wct_data_free,
        wct_data_print,
        /* don't need equalp because there should never be more than one
           of these and if we do, then they're *not* equal - only one
           place (module or whatever) can provide a given type. */
        NULL
      };
      scm_smobfuns wcp_smob_data = {
        wcp_data_mark,
        wcp_data_free,
        wcp_data_print,
        wcp_data_equal_p
      };

      wct_smob_id = scm_newsmob(&wct_smob_data);
      wcp_smob_id = scm_newsmob(&wcp_smob_data);
    }
#else 
    wct_smob_id = scm_make_smob_type("gw:wct", sizeof(wrapped_c_type_data));
    scm_set_smob_mark(wct_smob_id, wct_data_mark);
    scm_set_smob_free(wct_smob_id, wct_data_free);
    scm_set_smob_print(wct_smob_id, wct_data_print);
    /* don't need equalp because there should never be more than one
       of these and if we do, then they're *not* equal - only one
       place (module or whatever) can provide a given type. */

    wcp_smob_id = scm_make_smob_type("gw:wcp", sizeof(wrapped_c_type_data));
    scm_set_smob_free(wcp_smob_id, wcp_data_free);
    scm_set_smob_print(wcp_smob_id, wcp_data_print);
    scm_set_smob_mark(wcp_smob_id, wcp_data_mark);
    scm_set_smob_equalp(wcp_smob_id, wcp_data_equal_p);
#endif

    wct_system_initialized = 1;
  }
}

SCM
gw_wct_create(const char *type_name,
              SCM (*equal_p)(SCM wcp_a, SCM wcp_b),
              int (*print)(SCM wcp, SCM port,
                           char writing_p,
                           int *use_default_printer_p),
              SCM (*mark)(SCM wcp),
              scm_sizet (*cleanup)(SCM wcp))
{
  /* see header for docs */
  wrapped_c_type_data *type_data;

  if(!type_name) {
    scm_misc_error("gw_wct_create_and_register",
                   "null type_name argument",
                   SCM_EOL);
  }

  type_data = (wrapped_c_type_data *)
    scm_must_malloc(sizeof(wrapped_c_type_data),
                    "gw_wct_create_and_register: type_data");

  type_data->name = gh_str02scm(type_name);

  type_data->equal_p = equal_p;
  type_data->print = print;
  type_data->mark = mark;
  type_data->cleanup = cleanup;

  GW_RETURN_NEWSMOB(wct_smob_id, type_data);
}


/****************************************************************************/
/* Initialization */

static int gw_initialized = 0;
static SCM *scm_gw_descriptions;

void
gw_initialize(void) {
  if(!gw_initialized ) {
    scm_gw_descriptions =
      SCM_CDRLOC(scm_sysintern("*gw:descriptions*", SCM_EOL));
    initialize_wct_type_system();
    gw_initialized = 1;
  }
}
  

void
gw_add_description(SCM lst) {
  *scm_gw_descriptions = scm_cons(lst,*scm_gw_descriptions);
}

#if 0

/****************************************************************************/
/* Guile add-on C type helpers */

static int initialized_p = 0;
static SCM longlongoffset;
static SCM mult;
static SCM add;
static SCM quotient;
static SCM remainder;

static void
initialize_longlongs() {
  mult = gh_eval_str("*");
  add = gh_eval_str("+");
  quotient = gh_eval_str("quotient");
  remainder = gh_eval_str("remainder");
  longlongoffset = gh_eval_str("#x100000000");
  gh_define("gwp:___longlongoffset___anti_gc_binding___", longlongoffset);
  initialized_p = 42;
}

SCM
gh_ulonglong2scm(unsigned long long x) {
  const unsigned long upper = x >> 32;
  const unsigned long lower = (unsigned long) (x & 0xFFFFFFFF);
  SCM result;
  
  if(!initialized_p) initialize_longlongs();
  
  result = gh_call2(mult, gh_ulong2scm(upper), longlongoffset);
  result = gh_call2(add, result, gh_ulong2scm(lower));

  return(result);
}

unsigned long long
gh_scm2ulonglong(SCM x) {
  unsigned long long result;
  unsigned long upper;
  unsigned long lower;
  SCM upper_scm;
  SCM lower_scm;
  
  if(!initialized_p) initialize_longlongs();

  upper_scm = gh_call2(quotient, x, longlongoffset);
  lower_scm = gh_call2(remainder, x, longlongoffset);

  upper = gh_scm2ulong(upper_scm);
  lower = gh_scm2ulong(lower_scm);

  result = (((unsigned long long) upper) << 32) | lower;

  return(result);
}

SCM
gh_longlong2scm(long long x) {
  const long upper = x >> 32;
  const long lower = (unsigned long) (x & 0xFFFFFFFF);
  SCM result;
  
  if(!initialized_p) initialize_longlongs();
  
  result = gh_call2(mult, gh_long2scm(upper), longlongoffset);
  result = gh_call2(add, result, gh_long2scm(lower));

  return(result);
}

long long
gh_scm2longlong(SCM x) {
  long long result;
  long upper;
  long lower;
  SCM upper_scm;
  SCM lower_scm;
  
  if(!initialized_p) initialize_longlongs();

  upper_scm = gh_call2(quotient, x, longlongoffset);
  lower_scm = gh_call2(remainder, x, longlongoffset);

  upper = gh_scm2long(upper_scm);
  lower = gh_scm2long(lower_scm);

  result = (((long long) upper) << 32) | lower;

  return(result);
}

#endif
