/* FIGARO'S PASSWORD MANAGER (FPM)
 * Copyright (C) 2000 John Conneely
 * 
 * FPM is open source / free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * FPM is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA
 *
 * fpm.c - general routines
 */

#include "fpm.h"
#include "fpm_gpw.h"
#include "passfile.h"
#include "callbacks.h"
#include "interface.h"
#include "support.h"
#include "fpm_crypt.h"


/* GLOBAL VARIABLES */
GtkWidget* glb_win_app;   /* The main window for the application   */
GtkCList* glb_clist_main; /* Password CList on main window         */

GtkWidget* glb_win_misc;  /* Misc window, used for dialogs         */

GtkWidget* glb_win_edit;  /* The edit window for a password item   */
fpm_data* glb_edit_data;  /* The password item being edited now    */
gint glb_num_row ;        /* The total number of rows in the CList */
gint glb_cur_row;         /* The last item clicked on in the CList */
gchar* glb_filename;      /* The location of the pasword file.     */
gboolean glb_dirty;       /* Have changes been made?               */
GTimer* glb_timer_click;  /* Timer used to check for double clicks */
gint glb_click_count;	  /* Click count used for double clicks    */
gchar* new_salt;	  /* Salt to use on next save.             */
gchar* old_salt;	  /* Salt to used on last save.            */

void* old_context;
void* new_context;

char* vstring;
char* file_version;

GList *glb_pass_list;	 /* A GList of fpm_data containing all items */
GList *glb_cat_string_list;	 /* A GList of strings, used to populate combos */
GList *glb_launcher_list;
GList *glb_launcher_string_list;

GCompareFunc glb_cmp;


/* Current versions of FPM encrypt all fields.  Fields other than the
 * password are decrypted when the program reads the password file,
 * and the password is decrypted as it is needed (to try to prevent it
 * from showing up in coredumps, etc.)  The global variable  glb_need_decrypt
 * is set by the passfile loading routines to specify that the passfile
 * was created with a version of FPM that requires this decryption.  
 * This allows backward compatibility with previous versions of FPM which
 * only encrypted passwords.
 */
gboolean glb_need_decrypt;

/* glb_using_defaults is true if the user has made at least one
 * password item a default.  If the user has done this, the
 * default category should be the opening category.  Otherwise,
 * the opening category will be <ALL CATEGORIES>.
 */
gboolean glb_using_defaults;
gboolean glb_quitting=FALSE;
gchar* glb_last_category;


void
fpm_new_passitem(GtkWidget** win_edit_ptr, fpm_data** data_ptr)
{
    fpm_data* new_data;
    gchar plaintext[FPM_PASSWORD_LEN+1];

    new_data = g_malloc0(sizeof(fpm_data));

    strncpy(plaintext, "", FPM_PASSWORD_LEN);
    fpm_encrypt_field(	old_context, new_data->password,
			plaintext, FPM_PASSWORD_LEN);
    new_data->title=g_strdup("");
    new_data->arg=g_strdup("");
    new_data->user=g_strdup("");
    new_data->notes=g_strdup("");
    new_data->category=g_strdup("");
    new_data->launcher=g_strdup("");
    *data_ptr = new_data;


    fpm_edit_passitem(win_edit_ptr, new_data);
}




void
fpm_edit_passitem(GtkWidget** win_edit_ptr, fpm_data* data)
{

  GtkCombo* combo;
  gchar cleartext[FPM_PASSWORD_LEN+1];
  GtkText* notes;

  /* Create new dialog box */
  *win_edit_ptr = create_dialog_edit_passitem();

  /* Populate drop down boxes on edit screen */
  fpm_create_category_list(1);
  combo = GTK_COMBO(lookup_widget(*win_edit_ptr, "combo_item_category"));
  g_assert(glb_cat_string_list!=NULL);
  gtk_combo_set_popdown_strings(combo, glb_cat_string_list);

  
  fpm_create_launcher_string_list();
  g_assert(glb_launcher_string_list!=NULL);
  combo = GTK_COMBO(lookup_widget(*win_edit_ptr, "combo_edit_launcher"));
  if(glb_launcher_string_list!=NULL)
    gtk_combo_set_popdown_strings(combo, glb_launcher_string_list);

  /* Load the data into the dialog box */
  fpm_set_entry(*win_edit_ptr, "entry_title", data->title);
  fpm_set_entry(*win_edit_ptr, "entry_arg", data->arg);
  fpm_set_entry(*win_edit_ptr, "entry_user", data->user);
  fpm_set_entry(*win_edit_ptr, "entry_edit_category", data->category);
  fpm_set_entry(*win_edit_ptr, "entry_edit_launcher", data->launcher);
  gtk_toggle_button_set_active(
	GTK_TOGGLE_BUTTON(lookup_widget(*win_edit_ptr, "checkbutton_default")),
	data->default_list);

  fpm_decrypt_field(old_context, cleartext, data->password, FPM_PASSWORD_LEN);
  fpm_set_entry(*win_edit_ptr, "entry_password", cleartext);
  memset(cleartext, 0, FPM_PASSWORD_LEN);

  notes = GTK_TEXT(lookup_widget(*win_edit_ptr, "text_notes"));
  gtk_text_insert(notes, NULL, NULL, NULL, data->notes, -1);
  
  gtk_widget_show(*win_edit_ptr);
  gtk_widget_set_sensitive(glb_win_app, FALSE);

}


void
fpm_save_passitem(GtkWidget* win_edit, fpm_data* data)
{
  gint row;
  char* blank_row[FPM_NUM_COL];
  gint i;
  gchar* orig_plaintext;
  gchar plaintext[FPM_PASSWORD_LEN+1];
  GtkEntry* entry;
  GtkEditable* notes;

  /* Set dirty flag so we know we need to save later. */
  glb_dirty = TRUE;

  /* First update data structures */
  data->title = fpm_get_entry(win_edit, "entry_title");
  data->arg = fpm_get_entry(win_edit, "entry_arg");
  data->user = fpm_get_entry(win_edit, "entry_user");
  data->category = fpm_get_entry(win_edit, "entry_edit_category");
  data->launcher = fpm_get_entry(win_edit, "entry_edit_launcher");

  /* Update default check box */
  data->default_list = (GTK_TOGGLE_BUTTON(lookup_widget(win_edit, "checkbutton_default"))->active);

  /* Update password */
  entry = GTK_ENTRY(lookup_widget(win_edit, "entry_password"));
  orig_plaintext = gtk_entry_get_text(entry);
  strncpy(plaintext, orig_plaintext, FPM_PASSWORD_LEN);
  fpm_encrypt_field(old_context, data->password, plaintext, FPM_PASSWORD_LEN);
  memset(orig_plaintext, 0, strlen(orig_plaintext));
  memset(plaintext, 0, FPM_PASSWORD_LEN);

  /* Update notes */
  notes = GTK_EDITABLE(lookup_widget(win_edit, "text_notes"));
  g_free(data->notes);
  data->notes=gtk_editable_get_chars(notes, 0, -1);

  row = gtk_clist_find_row_from_data(glb_clist_main, data);

  /* Check to see if this is a new entry */ 
  if (row<0)
  {

    /* Construct blank row */
    for(i=0;i<FPM_NUM_COL; i++) blank_row[i]=NULL;

    /* Append row to end of CList */
    gtk_clist_append(glb_clist_main, blank_row);

    /* Keep track of the number of rows in the CList */
    glb_num_row++;

    /* Update the current row */
    row = glb_num_row-1;

    /* Associate this row with the data structure. */
    gtk_clist_set_row_data(glb_clist_main, row, data);

    /* Also, add this row to our main GList */
    glb_pass_list=g_list_append(glb_pass_list, data);

    g_print("inserting...\n");

  }
  else g_print("updating...\n");

  /* Update Visible CList Fields */
  gtk_clist_set_text(glb_clist_main, row, FPM_COL_TITLE, data->title);
  gtk_clist_set_text(glb_clist_main, row, FPM_COL_URL, data->arg);
  gtk_clist_set_text(glb_clist_main, row, FPM_COL_USER, data->user);

  /* Update Category drop down list on main screen */
  fpm_clist_populate_cat_list();

  gnome_question_dialog_modal("Do you want to save changes?",
    fpm_dialog_answer_save, NULL);
}


char*
fpm_get_entry(GtkWidget* win, gchar* entry_name)
{
  gchar* tmp;
  GtkEntry* entry_field;

  entry_field = GTK_ENTRY(lookup_widget (win, entry_name));
  tmp = gtk_entry_get_text(entry_field);
  return (g_strdup(tmp));
}


void
fpm_set_entry(GtkWidget* win, gchar* entry_name, char* text)
{

  GtkEntry* entry_field;

  entry_field = GTK_ENTRY(lookup_widget (win, entry_name));
  gtk_entry_set_text(entry_field, text);
}

void
fpm_init(void)
{
  glb_launcher_string_list=NULL;
  glb_win_app = create_app_safe();

  glb_filename=g_strdup_printf("%s/.fpm/fpm", g_get_home_dir());
  glb_timer_click = g_timer_new();
  g_timer_start(glb_timer_click);
  glb_click_count=-1;
  glb_need_decrypt=FALSE;
  fpm_gpw_set_options(8, TRUE, TRUE, TRUE, FALSE, TRUE);

  glb_clist_main = GTK_CLIST(lookup_widget (glb_win_app, "clist_main"));

 /* Now done in load passfile 
  * fpm_init_launchers();
  */

  if(passfile_load(glb_filename))
  {
    glb_win_misc = create_dialog_cpw();
    gtk_widget_show(glb_win_misc);
  }
  else
  {
    glb_win_misc = create_dialog_password();
    gtk_widget_show(glb_win_misc);
  }
}

void
fpm_quit(void)
{
  glb_quitting=TRUE;
  if (glb_dirty)
  {
    gnome_question_dialog_modal("Do you want to save changes?",
	 fpm_dialog_answer_save, NULL);
  }
  else
  {
    gtk_main_quit();

  }

}

void
fpm_dialog_answer_save(gint reply, gpointer data)
{
   if (reply==0)
   {
     passfile_save(glb_filename);
   }

   if (glb_quitting) gtk_main_quit();
}

void
fpm_dialog_answer_delete(gint reply, gpointer data)
{
  if (reply==0)
  {
    glb_pass_list=g_list_remove(glb_pass_list, glb_edit_data);
    gtk_clist_remove(glb_clist_main, glb_cur_row);
    glb_num_row--;
    if(glb_cur_row>=glb_num_row) glb_cur_row=glb_num_row;
    if(glb_cur_row<0)
    {
      glb_edit_data=NULL;
    }
    else
    {
      glb_edit_data= (fpm_data*)
        gtk_clist_get_row_data(glb_clist_main, glb_cur_row);
    }
    glb_dirty = TRUE;
  }
}


void
fpm_select(gchar* text, gboolean use_clipboard)
{
  GtkEntry* entry_hidden;
  
  g_assert(text != NULL);

  entry_hidden = GTK_ENTRY(lookup_widget(glb_win_app, "entry_hidden"));

  gtk_entry_set_text(entry_hidden, text);
  gtk_editable_select_region(GTK_EDITABLE(entry_hidden), 0, -1);
  if (use_clipboard)
    gtk_editable_copy_clipboard(GTK_EDITABLE(entry_hidden));
}

void
fpm_select_password(fpm_data* data, gboolean use_clipboard)
{
  gchar plaintext[FPM_PASSWORD_LEN+1];
  g_assert(data != NULL);

  fpm_decrypt_field(old_context, plaintext, data->password, FPM_PASSWORD_LEN);
  fpm_select(plaintext, use_clipboard);
  memset(plaintext, 0, FPM_PASSWORD_LEN);
}

void
fpm_double_click(fpm_data* data)
{
  if (data != NULL)
  {
    fpm_jump(data);
  }


}

void
fpm_jump(fpm_data* data)
{
  gchar* cmd;
  fpm_launcher *launcher;
  fpm_launcher *tmp;
  GList *list;

  launcher=NULL;
  list=g_list_first(glb_launcher_list);
  while (list!=NULL)
  {
    tmp=list->data;
    if (!strcmp(tmp->title, data->launcher)) launcher=tmp;
    list=g_list_next(list);
  }

  if (launcher)
  {

  /* Check for items that should be on clipboard or primary selection */
    if(launcher->copy_password == 2)
    {
      fpm_select_password(data, TRUE);
    }
    if(launcher->copy_user == 2) 
    {
      fpm_select(data->user, TRUE);
    }
    if(launcher->copy_user == 1)
    {
      fpm_select(data->user, FALSE);
    }
    if(launcher->copy_password == 1)
    {
      fpm_select_password(data, FALSE);
    }

    cmd=fpm_create_cmd_line(launcher->cmdline, data->arg,
	 data->user, data->password);

    gnome_execute_shell(NULL, cmd);
  }
  else
    gnome_warning_dialog("This password's launcher is undefined.\nPlease define it in the edit password item screen.");

}

void
fpm_check_password()
{
  GtkEntry* entry;
  GtkLabel* label;

  char vstring_plain[9];

  entry=GTK_ENTRY(lookup_widget(glb_win_misc, "entry_password"));

  fpm_crypt_init(gtk_entry_get_text(entry));

  fpm_decrypt_field(old_context, vstring_plain, vstring, 8);
  if (strcmp(vstring_plain, "FIGARO"))
  {
    gtk_editable_select_region(GTK_EDITABLE(entry), 0, -1);
    label=GTK_LABEL(lookup_widget(glb_win_misc, "label_password_message"));
    gtk_label_set_text(label, "Password incorrect.  Please try again.");
    return;
  }

  if (glb_need_decrypt)
  {
    fpm_decrypt_all();
    glb_need_decrypt=FALSE;
  }
/*
  fpm_clist_create_view(NULL);
 */

  if (glb_using_defaults)
  {
    glb_last_category=g_strdup(FPM_DEFAULT_CAT_MSG);
  }
  else
  {
    glb_last_category=g_strdup(FPM_ALL_CAT_MSG);
  }

  fpm_clist_populate_cat_list();



  gtk_widget_show(glb_win_app);
  gtk_widget_destroy(glb_win_misc);


}


void fpm_set_password()
{
  GtkEntry *entry;
  gchar *pw1, *pw2;
  gchar *cmd;
  byte *md5val, *prehash;

  entry=GTK_ENTRY(lookup_widget(glb_win_misc, "entry_cpw1"));
  pw1=gtk_entry_get_text(entry); 
  entry=GTK_ENTRY(lookup_widget(glb_win_misc, "entry_cpw2"));
  pw2=gtk_entry_get_text(entry); 
  
  if (strcmp(pw1, pw2))
  {
    gnome_error_dialog("Sorry, the passwords do not match.");
    return;
  }
  if (strlen(pw1)<4)
  {
    gnome_error_dialog("Sorry, your password is too short.");
    return;
  }

  cmd = g_strdup_printf("mkdir %s/.fpm", g_get_home_dir());
  gnome_execute_shell(NULL, cmd);
  g_free(cmd);
  cmd = g_strdup_printf("chmod 0700 %s/.fpm", g_get_home_dir());
  gnome_execute_shell(NULL, cmd);
  g_free(cmd);

  if(old_context == NULL)
  {
    /* This is the first time we are running the app.*/
    glb_last_category=g_strdup(FPM_ALL_CAT_MSG);
 /*
    fpm_init_launchers();
  */

    fpm_crypt_init(pw1);
    glb_dirty = TRUE;
    gtk_widget_destroy(glb_win_misc);
    gtk_widget_show(glb_win_app);

    fpm_clist_populate_cat_list();
  }
  else
  {
    /* We are chaning the password.*/
    new_salt=get_new_salt();
    prehash=g_strdup_printf("%s%s", new_salt, pw1);
    md5val=md5(prehash, strlen(prehash));
    fpm_setkey(new_context, md5val, 16);
    g_free(md5val);
    gtk_widget_destroy(glb_win_misc);
    passfile_save(glb_filename);
  }
}

void fpm_clear_list()
{
  GList * list;

  list=g_list_first(glb_pass_list);
  while (list!= NULL)
  {
    g_free(list->data);
    list=g_list_next(list);
  }
  g_list_free(glb_pass_list);
  glb_pass_list=NULL;
}

void fpm_init_launchers()
{
  GList * list;
  fpm_launcher* launcher;

  list = NULL;

  printf("Initiating default launchers...\n");

  launcher = g_malloc0(sizeof(fpm_launcher));
  launcher->title=g_strdup("Web");
  launcher->cmdline=g_strdup("netscape $a");
  launcher->copy_user = 2;
  launcher->copy_password = 1;
  list = g_list_append(list, launcher);

  launcher = g_malloc0(sizeof(fpm_launcher));
  launcher->title=g_strdup("ssh");
  launcher->cmdline=g_strdup("gnome-terminal -e 'ssh $a -l $u'");
  launcher->copy_user = 0;
  launcher->copy_password = 1;
  list = g_list_append(list, launcher);

  launcher = g_malloc0(sizeof(fpm_launcher));
  launcher->title=g_strdup("Generic Command");
  launcher->cmdline=g_strdup("$a");
  launcher->copy_user = 0;
  launcher->copy_password = 0;
  list = g_list_append(list, launcher);

  glb_launcher_list=list;

  fpm_create_launcher_string_list();
}

void fpm_create_launcher_string_list()
{
  GList * list;
  fpm_launcher* launcher;

  g_assert(glb_launcher_list!=NULL);
  /* FIXME:  I think you need the following free statement, but it seg faults
   * when I enable it.  Why?  If we need this statement, then the current
   * app will have a mem leak.  How big a deal is that?
   * g_free(glb_launcher_string_list);
   */
  glb_launcher_string_list=NULL;
  list=g_list_first(glb_launcher_list);
  while (list!= NULL)
  {
    launcher=list->data;
    glb_launcher_string_list=g_list_append(glb_launcher_string_list, g_strdup(launcher->title));
    list=g_list_next(list);
  }
}



gchar* fpm_create_cmd_line(gchar* cmd, gchar* arg, gchar* user, gchar* pass)
{
  gchar* result;
  gchar* tmp;
  gchar** cmd_arr;
  gint num, i;
  gchar* cleartext;

  cmd_arr = g_strsplit(cmd, "$", 0);
  num=sizeof(cmd_arr)/sizeof(char);
  result=g_strdup(cmd_arr[0]);
  i=1;
  while (cmd_arr[i]!=NULL)
  {
    tmp=result;
    if(cmd_arr[i][0]=='a')
      result=g_strconcat(tmp, arg, cmd_arr[i]+1, NULL);
    else if(cmd_arr[i][0]=='u')
      result=g_strconcat(tmp, user, cmd_arr[i]+1, NULL);
    else if(cmd_arr[i][0]=='p')
    {
      cleartext=fpm_decrypt_field_var(old_context, pass);
      result=g_strconcat(tmp, cleartext, cmd_arr[i]+1, NULL);
      memset(cleartext, 0, strlen(cleartext));
      g_free(cleartext);
    }
    else
      result=g_strconcat(tmp, "$", cmd_arr[i]);
    memset(tmp, 0, strlen(tmp));
    g_free(tmp);
    i++;
  }
  g_strfreev(cmd_arr);

  return(result);
}

void fpm_debug(gchar* msg)
{
  printf("Debug %s:%d:%d\n", msg, (glb_launcher_list==NULL), (glb_launcher_string_list==NULL));

}
