/*  -*- Mode: C; tab-width: 4; indent-tabs-mode: nil; c-basic-offset: 4 -*-
 * 
 * This file is part of the GNOME Debugging Framework.
 * 
 * Copyright (C) 1999-2000 Dave Camp <campd@oit.edu>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.  
 */

#include <config.h>
#include <gdf/gdf-widgets.h>
#include "dryad.h"
#include <liboaf/liboaf.h>

typedef GtkWidget *(*DryadWidgetCreateFunc)(DryadApp *app, void *data);
typedef void (*DryadWidgetDebuggerChangeFunc)(DryadApp *app, 
											   GtkWidget *w,
											   void *data);
typedef void (*DryadWidgetDestructionFunc)(DryadApp *app,
										   GtkWidget *w,
										   void *data);

typedef struct
{
    gchar *glade_window_id;
    gchar *menu_label;
} DryadWindow;

typedef struct 
{
	gchar *glade_parent_id;
	
	DryadWidgetCreateFunc create_func;
	void *create_data;
	DryadWidgetDebuggerChangeFunc dbg_change_func;
	void *dbg_change_data;
	DryadWidgetDestructionFunc destruction_func;
	void *destruction_data;
} DryadWidgetDesc;

typedef struct
{
	GtkWidget *widget;
	DryadWidgetDesc *desc;
} DryadWidget;

static void init_main_frame (DryadApp *app);
static void init_windows (DryadApp *app);
static void init_widgets (DryadApp *app);

static GtkWidget *bonobo_control_create_func (DryadApp *app, void *data);
static void bonobo_control_debugger_change_func (DryadApp *app, 
												  GtkWidget *w,
												  void *data);
static GtkWidget *output_terminal_create_func (DryadApp *app, void *data);
static void output_terminal_debugger_change_func (DryadApp *app, GtkWidget *w,
												   void *data);
static void output_terminal_destruction_func (DryadApp *app, GtkWidget *w,
											  void *data);
/** The generic stuff **/

/* takes a no parameter widget creation function for data */
static GtkWidget *widget_create_func (DryadApp *app, void *data);
/* takes a function which takes a widget and a dbg as data */
static void widget_dbg_func (DryadApp *app, GtkWidget *w, void *data);
/* calls gtk_widget_destroy on data */
static void widget_destruction_func (DryadApp *app, GtkWidget *w, void *data);

static void load_cb (GtkWidget *widget, DryadApp *app);
static void unload_cb (GtkWidget *widget, DryadApp *app);
static void run_cb (GtkWidget *widget, DryadApp *app);
static void attach_cb (GtkWidget *widget, DryadApp *app);
static void detach_cb (GtkWidget *widget, DryadApp *app);
static void core_cb (GtkWidget *widget, DryadApp *app);
static void stop_cb (GtkWidget *widget, DryadApp *app);
static void step_into_cb (GtkWidget *widget, DryadApp *app);
static void step_over_cb (GtkWidget *widget, DryadApp *app);
static void step_out_cb (GtkWidget *widget, DryadApp *app);
static void set_breakpoint_cb (GtkWidget *widget, DryadApp *app);
static void stack_up_cb (GtkWidget *widget, DryadApp *app);
static void stack_down_cb (GtkWidget *widget, DryadApp *app);
static void exit_cb (GtkWidget *widget, DryadApp *app);
static void file_selected_cb (GtkWidget *widget, GtkWidget *dlg);
static void corefile_selected_cb (GtkWidget *widget, GtkWidget *dlg);
static void parameters_cb (GtkWidget *widget, DryadApp *app);
static void about_cb (GtkWidget *widget, gpointer data);
static void hide_show_selected_cb (GtkWidget *widget, gpointer data);
static gint main_frame_delete_event_cb (GtkWidget *widget, GdkEvent *event,
                                        DryadApp *app);
static gint window_delete_event_cb (GtkWidget *widget, GdkEvent *event,
                                    gpointer data);


static DryadWindow windows [] = {
    { "breakpoint_window", N_("Breakpoint Manager") },
#if 0
    { "source_viewer_window", N_("Source Viewer") },
#endif
    { "output_window", N_("Output Terminal") },
    { "register_window", N_("Registers") },
    { "variable_window", N_("Variables") },
    { "stack_window", N_("Stack Browser") },
    { NULL, NULL }
};

static DryadWidgetDesc widgets_bonobo [] = {
    { 
		"breakpoint_frame", 
		bonobo_control_create_func, "OAFIID:control:gdf-breakpoint-manager:43548f6f-cb5a-451c-bd2c-1837a85b5e11",
		bonobo_control_debugger_change_func, NULL,
		widget_destruction_func, NULL 
	},
    { 
		"register_frame", 
		bonobo_control_create_func, "OAFIID:control:gdf-register-viewer:1ae95c26-5849-48c8-881d-36656795fa8c",
		bonobo_control_debugger_change_func, NULL,
		widget_destruction_func, NULL 
	},
    { 
		"source_viewer_frame", 
		bonobo_control_create_func, "OAFIID:control:gdf-default-source-viewer:bde33044-fad3-46b7-8b4a-32b4e923bb79",
		bonobo_control_debugger_change_func, NULL,
		widget_destruction_func, NULL 
	},
    { 
		"locals_frame", 
		bonobo_control_create_func, "OAFIID:control:gdf-locals-viewer:51323db4-0c64-4f8c-bdb9-1550506a9b66",
		bonobo_control_debugger_change_func, NULL,
		widget_destruction_func, NULL 
	},
    { 
		"variables_frame", 
		bonobo_control_create_func, "OAFIID:control:gdf-variable-viewer:e471e251-3cf7-4c6a-a6ce-bb6c1328f41a",
		bonobo_control_debugger_change_func, NULL,
		widget_destruction_func, NULL 
	},
    { 
		"output_frame", 
		output_terminal_create_func, NULL, 
		output_terminal_debugger_change_func, NULL, 
		output_terminal_destruction_func, NULL 
	},
    {
        "stack_frame",
        bonobo_control_create_func, "OAFIID:control:gdf-stack-browser:ed6030f5-6bb9-4694-814f-9d0e8a4af28c"
,
        bonobo_control_debugger_change_func, NULL,
        widget_destruction_func, NULL
    },
    { NULL, NULL, NULL, NULL, NULL, NULL }
    
};

static DryadWidgetDesc widgets[] = 
{
	{ 
		"breakpoint_frame", 
		widget_create_func, gdf_breakpoint_manager_new,
		widget_dbg_func, NULL,
		widget_destruction_func, NULL 
	},
    { 
		"register_frame", 
		widget_create_func, gdf_register_viewer_new,
		widget_dbg_func, NULL,
		widget_destruction_func, NULL 
	},
    { 
		"source_viewer_frame", 
		widget_create_func, gdf_source_viewer_manager_new,
		widget_dbg_func, NULL,
		widget_destruction_func, NULL 
	},
    { 
		"locals_frame", 
		widget_create_func, gdf_locals_viewer_new,
		widget_dbg_func, NULL,
		widget_destruction_func, NULL 
	},
    {
        "variables_frame",
        widget_create_func, gdf_variable_viewer_new,
        widget_dbg_func, NULL,
        widget_destruction_func, NULL
    },
    {
		"output_frame", 
		output_terminal_create_func, NULL, 
		output_terminal_debugger_change_func, NULL, 
		output_terminal_destruction_func, NULL 
	},
    {
        "stack_frame",
        widget_create_func, gdf_stack_browser_new,
        widget_dbg_func, NULL,
        widget_destruction_func, NULL
    },
    { NULL, NULL, NULL, NULL, NULL, NULL }
    
};

void 
dryad_ui_init (DryadApp *app)
{
    app->ui = glade_xml_new (DRYAD_GLADEDIR "/dryad.glade", NULL);
    app->widgets = NULL;

    init_main_frame (app);
    init_windows (app);
    init_widgets (app);
}

void
dryad_ui_destroy (DryadApp *app)
{
	GSList *cur = app->widgets;
	DryadWidget *w;
	
	for (; cur; cur = g_slist_next (cur)) {
		w = (DryadWidget*)cur->data;
		if (w->desc->destruction_func)
			w->desc->destruction_func (app, w->widget,
									   w->desc->destruction_data);
	}
}

void
dryad_ui_set_debuggers (DryadApp *app)
{
	GSList *cur = app->widgets;
	DryadWidget *w;
	
	for (; cur; cur = g_slist_next (cur)) {
		w = (DryadWidget*)cur->data;
		if (w->desc->dbg_change_func)
			w->desc->dbg_change_func (app, w->widget,
									   w->desc->dbg_change_data);
	}
}

void
init_main_frame (DryadApp *app)
{
	GtkWidget *widget;
	widget = glade_xml_get_widget (app->ui, "mnu_load");
	gtk_signal_connect (GTK_OBJECT (widget), "activate", 
						GTK_SIGNAL_FUNC (load_cb), (gpointer) app);
	widget = glade_xml_get_widget (app->ui, "load_btn");
	gtk_signal_connect (GTK_OBJECT (widget), "clicked", 
						GTK_SIGNAL_FUNC (load_cb), (gpointer) app);
	widget = glade_xml_get_widget (app->ui, "mnu_unload");
	gtk_signal_connect (GTK_OBJECT (widget), "activate", 
						GTK_SIGNAL_FUNC (unload_cb), (gpointer) app);
	widget = glade_xml_get_widget (app->ui, "unload_btn");
	gtk_signal_connect (GTK_OBJECT (widget), "clicked", 
						GTK_SIGNAL_FUNC (unload_cb), (gpointer) app);
	widget = glade_xml_get_widget (app->ui, "mnu_run");
	gtk_signal_connect (GTK_OBJECT (widget), "activate", 
                        GTK_SIGNAL_FUNC (run_cb), (gpointer) app);
	widget = glade_xml_get_widget (app->ui, "run_btn");
	gtk_signal_connect (GTK_OBJECT (widget), "clicked", 
						GTK_SIGNAL_FUNC (run_cb), (gpointer) app);
	widget = glade_xml_get_widget (app->ui, "mnu_attach");
	gtk_signal_connect (GTK_OBJECT (widget), "activate", 
						GTK_SIGNAL_FUNC (attach_cb), (gpointer) app);
	widget = glade_xml_get_widget (app->ui, "attach_btn");
	gtk_signal_connect (GTK_OBJECT (widget), "clicked", 
						GTK_SIGNAL_FUNC (attach_cb), (gpointer) app);
	widget = glade_xml_get_widget (app->ui, "mnu_detach");
	gtk_signal_connect (GTK_OBJECT (widget), "activate", 
						GTK_SIGNAL_FUNC (detach_cb), (gpointer) app);
	widget = glade_xml_get_widget (app->ui, "detach_btn");
	gtk_signal_connect (GTK_OBJECT (widget), "clicked", 
						GTK_SIGNAL_FUNC (detach_cb), (gpointer) app);
	widget = glade_xml_get_widget (app->ui, "mnu_core");
	gtk_signal_connect (GTK_OBJECT (widget), "activate", 
						GTK_SIGNAL_FUNC (core_cb), (gpointer) app);
	widget = glade_xml_get_widget (app->ui, "core_btn");
	gtk_signal_connect (GTK_OBJECT (widget), "clicked", 
						GTK_SIGNAL_FUNC (core_cb), (gpointer) app);
	widget = glade_xml_get_widget (app->ui, "mnu_stop");
	gtk_signal_connect (GTK_OBJECT (widget), "activate", 
						GTK_SIGNAL_FUNC (stop_cb), (gpointer) app);
	widget = glade_xml_get_widget (app->ui, "stop_btn");
	gtk_signal_connect (GTK_OBJECT (widget), "clicked", 
						GTK_SIGNAL_FUNC (stop_cb), (gpointer) app);
	widget = glade_xml_get_widget (app->ui, "mnu_step_into");
	gtk_signal_connect (GTK_OBJECT (widget), "activate", 
						GTK_SIGNAL_FUNC (step_into_cb), (gpointer) app);
	widget = glade_xml_get_widget (app->ui, "step_into_btn");
	gtk_signal_connect (GTK_OBJECT (widget), "clicked", 
						GTK_SIGNAL_FUNC (step_into_cb), (gpointer) app);
	widget = glade_xml_get_widget (app->ui, "mnu_step_over");
	gtk_signal_connect (GTK_OBJECT (widget), "activate", 
						GTK_SIGNAL_FUNC (step_over_cb), (gpointer) app);
	widget = glade_xml_get_widget (app->ui, "step_over_btn");
	gtk_signal_connect (GTK_OBJECT (widget), "clicked", 
						GTK_SIGNAL_FUNC (step_over_cb), (gpointer) app);
	widget = glade_xml_get_widget (app->ui, "mnu_step_out");
	gtk_signal_connect (GTK_OBJECT (widget), "activate", 
						GTK_SIGNAL_FUNC (step_out_cb), (gpointer) app);
	widget = glade_xml_get_widget (app->ui, "step_out_btn");
	gtk_signal_connect (GTK_OBJECT (widget), "clicked", 
						GTK_SIGNAL_FUNC (step_out_cb), (gpointer) app);
	widget = glade_xml_get_widget (app->ui, "mnu_set_breakpoint");
	gtk_signal_connect (GTK_OBJECT (widget), "activate", 
						GTK_SIGNAL_FUNC (set_breakpoint_cb), (gpointer) app);
	widget = glade_xml_get_widget (app->ui, "mnu_stack_up");
	gtk_signal_connect (GTK_OBJECT (widget), "activate", 
						GTK_SIGNAL_FUNC (stack_up_cb), (gpointer) app);
	widget = glade_xml_get_widget (app->ui, "stack_up_btn");
	gtk_signal_connect (GTK_OBJECT (widget), "clicked", 
						GTK_SIGNAL_FUNC (stack_up_cb), (gpointer) app);
	widget = glade_xml_get_widget (app->ui, "mnu_stack_down");
	gtk_signal_connect (GTK_OBJECT (widget), "activate", 
						GTK_SIGNAL_FUNC (stack_down_cb), (gpointer) app);
	widget = glade_xml_get_widget (app->ui, "stack_down_btn");
	gtk_signal_connect (GTK_OBJECT (widget), "clicked", 
						GTK_SIGNAL_FUNC (stack_down_cb), (gpointer) app);
    widget = glade_xml_get_widget (app->ui, "mnu_parameters");
    gtk_signal_connect (GTK_OBJECT (widget), "activate",
                        GTK_SIGNAL_FUNC (parameters_cb), (gpointer) app);
    widget = glade_xml_get_widget (app->ui, "mnu_exit");
	gtk_signal_connect (GTK_OBJECT (widget), "activate", 
						GTK_SIGNAL_FUNC (exit_cb), (gpointer) app);
	widget = glade_xml_get_widget (app->ui, "mnu_about");
	gtk_signal_connect (GTK_OBJECT (widget), "activate", 
						GTK_SIGNAL_FUNC (about_cb), (gpointer) app);
    widget = glade_xml_get_widget (app->ui, "dryad_app");
    gtk_signal_connect (GTK_OBJECT (widget), "delete_event",
                        GTK_SIGNAL_FUNC (main_frame_delete_event_cb),
                        (gpointer) app);
}

static void
setup_menu_item (DryadApp *app, gchar *label, 
                 GtkWidget *window, GtkWidget *menu)
{
    GtkWidget *menu_item;
    
    menu_item = gtk_check_menu_item_new_with_label (label);
    gtk_check_menu_item_set_show_toggle (GTK_CHECK_MENU_ITEM (menu_item),
                                         TRUE);
    gtk_menu_append (GTK_MENU (menu), menu_item);
    gtk_signal_connect (GTK_OBJECT (menu_item), "toggled", 
                        GTK_SIGNAL_FUNC (hide_show_selected_cb), 
                        NULL);
    gtk_widget_show (menu_item);
    
    gtk_object_set_data (GTK_OBJECT (menu_item), "window", window);
    gtk_object_set_data (GTK_OBJECT (window), "menuitem", menu_item);
    
    gtk_check_menu_item_set_active (GTK_CHECK_MENU_ITEM (menu_item),
                                    GTK_WIDGET_VISIBLE (window));  
}

void 
init_windows (DryadApp *app)
{
    GtkWidget *window;
    GtkWidget *menu, *menu_item;
    DryadWindow *wnd;
    
    wnd = windows;
    menu = gtk_menu_new ();
    while (wnd->glade_window_id) {
        /* Setup the window */
        window = glade_xml_get_widget (app->ui, wnd->glade_window_id);
        gtk_signal_connect (GTK_OBJECT (window), "delete_event", 
                            GTK_SIGNAL_FUNC (window_delete_event_cb),
                            NULL);
		

        /* Setup the menu item */
        setup_menu_item (app, wnd->menu_label, window, menu);
		
        wnd++;
    }    
    menu_item = glade_xml_get_widget (app->ui, "mnu_windows");
    gtk_menu_item_set_submenu (GTK_MENU_ITEM (menu_item), menu);
}

void
init_widgets (DryadApp *app)
{
	DryadWidgetDesc *desc;
	
	if (g_getenv ("DRYAD_USE_BONOBO")) 
		desc = widgets_bonobo;
	else
		desc = widgets;
	
	while (desc->create_func) {
		DryadWidget *dw;
		GtkWidget *parent = 
			glade_xml_get_widget (app->ui, desc->glade_parent_id);
		GtkWidget *widget = desc->create_func (app, desc->create_data);
		gtk_container_add (GTK_CONTAINER (parent), widget);
		gtk_widget_show (widget);

		dw = g_new0 (DryadWidget, 1);
		dw->widget = widget;
		dw->desc = desc;
		app->widgets = g_slist_prepend (app->widgets, dw);

		desc++;
	}
}

GtkWidget *
bonobo_control_create_func (DryadApp *app, void *data)
{
	GtkWidget *ret;
	
	ret = bonobo_widget_new_control ((char*)data, NULL);
	return ret;
}

void
bonobo_control_debugger_change_func (DryadApp *app, 
									  GtkWidget *w,
									  void *data)
{   
	gchar *ior;
	CORBA_Environment ev;   
	BonoboControlFrame *cf;
	Bonobo_PropertyBag *pb;
	
	CORBA_exception_init (&ev);
	if (app->dbg) {
        g_assert (app->dbg->objref != CORBA_OBJECT_NIL);
		ior = CORBA_ORB_object_to_string (oaf_orb_get (), 
										  app->dbg->objref,
										  &ev);
	} else {
		ior = CORBA_string_dup ("");
	}
	
	/* FIXME: Check for exceptions */
	
	cf = bonobo_widget_get_control_frame (BONOBO_WIDGET (w));
    pb = bonobo_control_frame_get_control_property_bag (cf, &ev);
    bonobo_property_bag_client_set_value_string (*pb, "debugger-ior", ior, &ev);
	CORBA_free (ior);
	
	CORBA_exception_free (&ev);
}

GtkWidget *
output_terminal_create_func (DryadApp *app, void *data)
{
	GtkWidget *ret;
	
	ret = create_output_terminal (&app->tty_name);
	
	return ret;
}

void 
output_terminal_debugger_change_func (DryadApp *app, GtkWidget *w,
									   void *data)
{
	if (app->dbg) {
		gdf_debugger_client_set_output_tty (app->dbg, app->tty_name);
	}
}

void
output_terminal_destruction_func (DryadApp *app, GtkWidget *w, void *data)
{
	g_free (app->tty_name);
}

typedef GtkWidget *(*NoArgWidgetCreationFunc)(void);
typedef void *(*WidgetDbgFunc)(GtkWidget *w, GdfDebuggerClient *dbg);

/* Generic creation routine.  Treats data as a function pointer of a no-arg
 * widget creation function */
GtkWidget *
widget_create_func (DryadApp *app, void *data)
{
	GtkWidget *ret = ((NoArgWidgetCreationFunc)data)();
	return ret;
}

void
widget_dbg_func (DryadApp *app, GtkWidget *w, void *data)
{
    gtk_object_set (GTK_OBJECT (w), "debugger", 
                    (GtkObject *)app->dbg, NULL);
}

void 
widget_destruction_func (DryadApp *app, GtkWidget *w, void *data)
{
	gtk_widget_destroy (w);
}

void
load_cb (GtkWidget *widget, DryadApp *app)
{
	GtkWidget *dlg;
	
	dlg = gtk_file_selection_new (_("Select Program"));
	gtk_object_set_data (GTK_OBJECT (dlg), "dryad_app", (gpointer) app);
	gtk_signal_connect (GTK_OBJECT (GTK_FILE_SELECTION (dlg)->ok_button),
						"clicked", GTK_SIGNAL_FUNC (file_selected_cb),
						(gpointer)dlg);
	gtk_signal_connect (GTK_OBJECT (GTK_FILE_SELECTION (dlg)->cancel_button),
						"clicked", GTK_SIGNAL_FUNC (file_selected_cb),
						(gpointer)dlg);
	gtk_widget_show (dlg);
}

void
unload_cb (GtkWidget *widget, DryadApp *app)
{
	gdf_debugger_client_unload_binary (app->dbg);

	g_free (app->binary_name);
	app->binary_name = NULL;

    dryad_unload_debugger (app);
}

void 
run_cb (GtkWidget *widget, DryadApp *app)
{
	GDF_Debugger_StateFlag state;
	g_assert (app->binary_name != NULL);
	
	state = gdf_debugger_client_get_state (app->dbg);
	
	if ((state & GDF_Debugger_TARGET_EXISTS) != GDF_Debugger_TARGET_EXISTS) {
		gdf_debugger_client_execute (app->dbg, app->parameters);
	} else {
		gdf_debugger_client_cont (app->dbg);
	}
}

void
attach_cb (GtkWidget *widget, DryadApp *app)
{
    gdf_attach_dialog (app->dbg);
}

void 
detach_cb (GtkWidget *widget, DryadApp *app)
{
    gdf_debugger_client_detach (app->dbg);
}

void
core_cb (GtkWidget *widget, DryadApp *app)
{
	GtkWidget *dlg;
	
	dlg = gtk_file_selection_new (_("Select Corefile"));
	gtk_object_set_data (GTK_OBJECT (dlg), "dryad_app", (gpointer) app);
	gtk_signal_connect (GTK_OBJECT (GTK_FILE_SELECTION (dlg)->ok_button),
						"clicked", GTK_SIGNAL_FUNC (corefile_selected_cb),
						(gpointer)dlg);
	gtk_signal_connect (GTK_OBJECT (GTK_FILE_SELECTION (dlg)->cancel_button),
						"clicked", GTK_SIGNAL_FUNC (corefile_selected_cb),
						(gpointer)dlg);
	gtk_widget_show (dlg);
}

void
stop_cb (GtkWidget *widget, DryadApp *app)
{
    GDF_Debugger_StateFlag state = gdf_debugger_client_get_state (app->dbg);
    if (state & GDF_Debugger_TARGET_EXISTS) {
        gdf_debugger_client_stop (app->dbg);
    } else if (state & GDF_Debugger_COREFILE) {
        gdf_debugger_client_unload_corefile (app->dbg);
    }
}

void 
step_into_cb (GtkWidget *widget, DryadApp *app)
{
	gdf_debugger_client_step_into (app->dbg);
}

void 
step_over_cb (GtkWidget *widget, DryadApp *app)
{
	gdf_debugger_client_step_over (app->dbg);
}

void 
step_out_cb (GtkWidget *widget, DryadApp *app)
{
	gdf_debugger_client_step_out (app->dbg);
}

void
set_breakpoint_cb (GtkWidget *widget, DryadApp *app)
{
    gdf_add_breakpoint_dialog (app->dbg);
}

void
stack_up_cb (GtkWidget *widget, DryadApp *app)
{
    gdf_debugger_client_up_frame (app->dbg);
}

void
stack_down_cb (GtkWidget *widget, DryadApp *app)
{
    gdf_debugger_client_down_frame (app->dbg);
}

void
exit_cb (GtkWidget *widget, DryadApp *app)
{
	dryad_exit (app);
}

typedef struct
{
    DryadApp *app;
    gchar *filename;
} FileSelectedIdleData;

static gint
load_file (FileSelectedIdleData *data)
{
    if (data->app->binary_name != NULL) {
        gdf_debugger_client_unload_binary (data->app->dbg);
        g_free (data->app->binary_name);
        data->app->binary_name = NULL;
    }

    /* FIXME: You don't really need to destroy and create debuggers */
    if (data->app->dbg)
        dryad_unload_debugger (data->app);

    dryad_load_debugger (data->app, data->filename);

    g_free (data->filename);
    g_free (data);

    return FALSE;
}

void 
file_selected_cb (GtkWidget *widget, GtkWidget *dlg)
{
    FileSelectedIdleData *data;

	if (widget != GTK_FILE_SELECTION (dlg)->cancel_button) {
        data = g_new0(FileSelectedIdleData, 1);
		data->filename = 
            g_strdup (gtk_file_selection_get_filename (GTK_FILE_SELECTION (dlg)));
		data->app = (DryadApp*)gtk_object_get_data (GTK_OBJECT (dlg), 
                                                    "dryad_app");
        gtk_idle_add ((GtkFunction)load_file, data);
	}
	
	gtk_widget_destroy (dlg);
}

static gint
load_corefile (FileSelectedIdleData *data)
{
    dryad_load_corefile (data->app, data->filename);

    g_free (data->filename);
    g_free (data);

    return FALSE;
}

void
corefile_selected_cb (GtkWidget *widget, GtkWidget *dlg)
{
    FileSelectedIdleData *data;
    
    if (widget != GTK_FILE_SELECTION (dlg)->cancel_button) {
        data = g_new0 (FileSelectedIdleData, 1);
        data->filename = 
            g_strdup (gtk_file_selection_get_filename (GTK_FILE_SELECTION (dlg)));
        data->app = (DryadApp*)gtk_object_get_data (GTK_OBJECT (dlg), 
                                                    "dryad_app");
        gtk_idle_add ((GtkFunction)load_corefile, data);
    }
    
    gtk_widget_destroy (dlg);
    
}

void
parameters_cb (GtkWidget *widget, DryadApp *app) 
{
    char *new_args = gdf_parameters_dialog ();
    if (new_args) {
        g_free (app->parameters);
        app->parameters = g_strdup (new_args);
    }
}

void 
about_cb (GtkWidget *widget, gpointer data)
{
	GtkWidget *about;
	static gchar *authors[] = { 
		N_("Dave Camp <campd@oit.edu>"), 
		N_("Martin Baulig <martin@home-of-linux.org>"),
		NULL
	};

	about = 
		gnome_about_new (_("Dryad"), VERSION, 
						 "Copyright (c) 1999-2000 Dave Camp, Martin Baulig",
						 (const gchar **)authors,
						 "A graphical debugging tool for the GNOME environment.",
						 NULL);
	gtk_widget_show (about);
}							 

void
hide_show_selected_cb (GtkWidget *widget, gpointer data)
{
    GtkWidget *window;
    
    window = GTK_WIDGET (gtk_object_get_data (GTK_OBJECT (widget), "window"));
    
    if (GTK_CHECK_MENU_ITEM (widget)->active)
        gtk_widget_show (window);
    else
        gtk_widget_hide (window);
}

gint
main_frame_delete_event_cb (GtkWidget *widget, GdkEvent *event, DryadApp *app)
{
    dryad_exit (app);

    return FALSE;
} 

gint
window_delete_event_cb (GtkWidget *widget, GdkEvent *event, gpointer data)
{
    GtkWidget *menu_item;
    menu_item = GTK_WIDGET (gtk_object_get_data (GTK_OBJECT (widget), 
                                                 "menuitem"));
    gtk_check_menu_item_set_active (GTK_CHECK_MENU_ITEM (menu_item), FALSE);
    gtk_widget_hide (widget);

    return TRUE;
}


