/*  -*- Mode: C; tab-width: 4; indent-tabs-mode: nil; c-basic-offset: 4 -*-
 * 
 * This file is part of the GNOME Debugging Framework.
 * 
 * Copyright (C) 2000 Dave Camp <campd@oit.edu>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this p3rogram; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.  
 */

#include <gnome.h>
#include <unistd.h>
#include <fcntl.h>
#include <sys/stat.h>
#include <sys/types.h>
#include <config.h>
#include <signal.h>
#include <sys/wait.h>
#include <zvt/zvtterm.h>

#include "dryad.h"

static const char shell_script_fmt[] =
"tty > %s; \
trap \"\" INT QUIT TSTP; \
exec <&-;exec >&-;
while :; do sleep 3600; done";

GtkWidget *
create_output_terminal (gchar **tty_name_out)
{
	gchar tty_name[50];
	gchar *fifo_name;
	gchar *ptr_ch;
	gint pid;
	gint fd, num_read;
	GtkWidget *zvt_term;
	
	fifo_name = g_strdup_printf ("/tmp/dryad-ttywin%05d", getpid());
	unlink (fifo_name); /* make sure it does not exist */
	
#ifdef HAVE_MKFIFO
	if (mkfifo (fifo_name, SIRUSR | S_IWUSER) < 0) {
		g_warning (_("Unable to create fifo for output window!"));
		g_free (fifo_name);
		return NULL;
	}
#else
	if (mknod (fifo_name, S_IFIFO | S_IRUSR | S_IWUSR, 0) < 0) {
		g_warning (_("Unable to create fifo for output window!"));
		g_free (fifo_name);
		return NULL;
	}
#endif
	
	zvt_term = zvt_term_new ();
	
	pid = zvt_term_forkpty (ZVT_TERM (zvt_term), 0);
	
	if (pid < 0) {
		g_warning (_("Unable to fork tty process for output window!"));
		g_free (fifo_name);
		return NULL;
	}
	
	if (pid == 0) {  /* child */
		
		/* spawn a shell which runs a script which will pass its tty back
		 * through the fifo */
		gchar *shell_script = g_strdup_printf (shell_script_fmt,
											   fifo_name);
		execlp ("sh", "sh", "-c", shell_script, NULL);
		g_error (_("Unable to spawn output window!"));
	}
	
	/* parent */
	fd = open (fifo_name, O_RDONLY);
	if (fd < 0) {
		unlink (fifo_name);
		g_free (fifo_name);
		return NULL;
	}
	
	unlink (fifo_name);
	
	num_read = read (fd, tty_name, sizeof (tty_name) - sizeof (gchar));
	close (fd);
	
	/* remove the cr at the end */
	ptr_ch = strchr (tty_name, '\n');
	if (ptr_ch)
		*ptr_ch = '\0';
	
	*tty_name_out = g_strdup (tty_name);
	g_free (fifo_name);
	return zvt_term;
}

void
destroy_output_terminal (GtkWidget *terminal)
{
	zvt_term_closepty (ZVT_TERM (terminal));
}
	
	
		
		
