/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */
/*
 * Copyright (C) 2009 Canonical Services Ltd (www.canonical.com)
 *               2009 Mikkel Kamstrup Erlandsen
 *
 * Authors: Rodrigo Moya <rodrigo.moya@canonical.com>
 *          Mikkel Kamstrup Erlandsen <mikkel.kamstrup@gmail.com>
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of version 2 of the GNU Lesser General Public
 * License as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
 * Boston, MA 02110-1301, USA.
 */

#include <libsoup/soup-logger.h>
#include <libsoup/soup-session-async.h>
#include <libsoup/soup-gnome.h>
#include <json-glib/json-glib.h>
#include "couchdb-document.h"
#include "utils.h"

struct _CouchdbDocument {
	GObject		parent;

	CouchdbSession  *couchdb;
	char		*dbname;
	JsonNode	*root_node;
};

G_DEFINE_TYPE(CouchdbDocument, couchdb_document, G_TYPE_OBJECT);

static void
couchdb_document_finalize (GObject *object)
{
	CouchdbDocument *document = COUCHDB_DOCUMENT (object);

	g_free (document->dbname);
	json_node_free (document->root_node);

	G_OBJECT_CLASS (couchdb_document_parent_class)->finalize (object);
}

static void
couchdb_document_class_init (CouchdbDocumentClass *klass)
{
	GObjectClass *object_class = G_OBJECT_CLASS (klass);

	object_class->finalize = couchdb_document_finalize;
}

static void
couchdb_document_init (CouchdbDocument *document)
{
	document->couchdb = NULL;
	document->root_node = NULL;
	document->dbname = NULL;
}

CouchdbDocument *
couchdb_document_new (CouchdbSession *couchdb)
{
	CouchdbDocument *document;

	document = g_object_new (COUCHDB_TYPE_DOCUMENT, NULL);
	document->couchdb = couchdb;
	document->dbname = NULL;

	document->root_node = json_node_new (JSON_NODE_OBJECT);
	json_node_set_object (document->root_node, json_object_new ());

	return document;
}

CouchdbDocument *
couchdb_document_get (CouchdbSession *couchdb,
		      const char *dbname,
		      const char *docid,
		      GError **error)
{
	char *url, *encoded_docid;
	JsonParser *parser;
	CouchdbDocument *document = NULL;

	g_return_val_if_fail (COUCHDB_IS_SESSION (couchdb), NULL);
	g_return_val_if_fail (dbname != NULL, NULL);
	g_return_val_if_fail (docid != NULL, NULL);

	encoded_docid = soup_uri_encode (docid, NULL);
	url = g_strdup_printf ("%s/%s/%s", couchdb_session_get_uri (couchdb), dbname, encoded_docid);
	parser = json_parser_new ();
	if (couchdb_session_send_message (couchdb, SOUP_METHOD_GET, url, NULL, parser, error)) {
		document = g_object_new (COUCHDB_TYPE_DOCUMENT, NULL);
		document->couchdb = couchdb;
		document->dbname = g_strdup (dbname);

		document->root_node = json_node_copy (json_parser_get_root (parser));		
	}
	g_object_unref (G_OBJECT (parser));
	g_free (encoded_docid);
	g_free (url);

	return document;
}

gboolean
couchdb_document_put (CouchdbDocument *document,
		      const char *dbname,
		      GError **error)
{
	char *url, *body;
	const char *id;
	JsonParser *parser;
	gboolean result = FALSE;
	gboolean send_ok;;

	g_return_val_if_fail (COUCHDB_IS_DOCUMENT (document), FALSE);
	g_return_val_if_fail (dbname != NULL, FALSE);

	id = couchdb_document_get_id (document);
	body = couchdb_document_to_string (document);
	parser = json_parser_new ();
	if (id) {
		char *encoded_docid;

		encoded_docid = soup_uri_encode (id, NULL);
		url = g_strdup_printf ("%s/%s/%s", couchdb_session_get_uri (document->couchdb), dbname, encoded_docid);		
		send_ok = couchdb_session_send_message (document->couchdb, SOUP_METHOD_PUT, url, body, parser, error);

		g_free (encoded_docid);
	} else {
		url = g_strdup_printf ("%s/%s/", couchdb_session_get_uri (document->couchdb), dbname);
		send_ok = couchdb_session_send_message (document->couchdb, SOUP_METHOD_POST, url, body, parser, error);
	}

	if (send_ok) {
		JsonObject *object;

		object = json_node_get_object (json_parser_get_root (parser));
		couchdb_document_set_id (document, json_object_get_string_member (object, "id"));
		couchdb_document_set_revision (document, json_object_get_string_member (object, "rev"));

		if (document->dbname) {
			g_free (document->dbname);
			document->dbname = g_strdup (dbname);
		}

		if (id)
			g_signal_emit_by_name (document->couchdb, "document_updated", dbname, document);
		else
			g_signal_emit_by_name (document->couchdb, "document_created", dbname, document);

		result = TRUE;
	}

	/* free memory */
	g_free (url);
	g_free (body);
	g_object_unref (G_OBJECT (parser));

	return result;
}

gboolean
couchdb_document_delete (CouchdbDocument *document, GError **error)
{
	const char *id, *revision;
	char *url;
	JsonParser *parser;
	gboolean result = FALSE;

	g_return_val_if_fail (COUCHDB_IS_DOCUMENT (document), FALSE);

	id = couchdb_document_get_id (document);
	revision = couchdb_document_get_revision (document);
	if (!id || !revision) /* we can't remove a document without an ID and/or a REVISION */
		return FALSE;

	url = g_strdup_printf ("%s/%s/%s?rev=%s", couchdb_session_get_uri (document->couchdb), document->dbname, id, revision);
	
	/* We don't parse the http response, therefore the parser arg is NULL */
	if (couchdb_session_send_message (document->couchdb, SOUP_METHOD_DELETE, url, NULL, NULL, error)) {
		result = TRUE;		
		g_signal_emit_by_name (document->couchdb, "document_deleted", document->dbname, id);
	}

	g_free (url);

	return result;
}

const char *
couchdb_document_get_id (CouchdbDocument *document)
{
	g_return_val_if_fail (COUCHDB_IS_DOCUMENT (document), NULL);

	if (document->root_node &&
	    json_node_get_node_type (document->root_node) == JSON_NODE_OBJECT) {
		if (json_object_has_member (json_node_get_object (document->root_node),
					    "_id"))
			return json_object_get_string_member (
				json_node_get_object (document->root_node),
				"_id");
	}

	return NULL;
}

void
couchdb_document_set_id (CouchdbDocument *document, const char *id)
{
	g_return_if_fail (COUCHDB_IS_DOCUMENT (document));
	g_return_if_fail (id != NULL);

	json_object_set_string_member (json_node_get_object (document->root_node),
				       "_id",
				       id);
}

const char *
couchdb_document_get_revision (CouchdbDocument *document)
{
	g_return_val_if_fail (COUCHDB_IS_DOCUMENT (document), NULL);

	if (document->root_node &&
	    json_node_get_node_type (document->root_node) == JSON_NODE_OBJECT) {
		return json_object_get_string_member (
			json_node_get_object (document->root_node),
			"_rev");
	}

	return NULL;
}

void
couchdb_document_set_revision (CouchdbDocument *document, const char *revision)
{
	g_return_if_fail (COUCHDB_IS_DOCUMENT (document));
	g_return_if_fail (revision != NULL);

	json_object_set_string_member (json_node_get_object (document->root_node),
				       "_rev",
				       revision);
}

const char *
couchdb_document_get_record_type (CouchdbDocument *document)
{
	g_return_val_if_fail (COUCHDB_IS_DOCUMENT (document), NULL);

	if (document->root_node &&
	    json_node_get_node_type (document->root_node) == JSON_NODE_OBJECT) {
		return json_object_get_string_member (
			json_node_get_object (document->root_node),
			"record_type");
	}

	return NULL;
}

void
couchdb_document_set_record_type (CouchdbDocument *document, const char *record_type)
{
	g_return_if_fail (COUCHDB_IS_DOCUMENT (document));
	g_return_if_fail (record_type != NULL);

	json_object_set_string_member (json_node_get_object (document->root_node),
				       "record_type",
				       record_type);
}

gboolean
couchdb_document_has_field (CouchdbDocument *document, const char *field)
{
	JsonNode *node;

	g_return_val_if_fail (COUCHDB_IS_DOCUMENT (document), FALSE);
	g_return_val_if_fail (field != NULL, FALSE);

	return json_object_has_member (json_node_get_object (document->root_node), field);
}

void
couchdb_document_remove_field (CouchdbDocument *document, const char *field)
{
	g_return_if_fail (COUCHDB_IS_DOCUMENT (document));
	g_return_if_fail (field != NULL);

	json_object_remove_member (json_node_get_object (document->root_node), field);
}

gboolean
couchdb_document_get_boolean_field (CouchdbDocument *document, const char *field)
{
	g_return_val_if_fail (COUCHDB_IS_DOCUMENT (document), FALSE);
	g_return_val_if_fail (field != NULL, FALSE);

	if (!json_object_has_member (json_node_get_object (document->root_node), field))
		return FALSE;

	return json_object_get_boolean_member (json_node_get_object (document->root_node),
					       field);
}

void
couchdb_document_set_boolean_field (CouchdbDocument *document, const char *field, gboolean value)
{
	g_return_if_fail (COUCHDB_IS_DOCUMENT (document));
	g_return_if_fail (field != NULL);

	json_object_set_boolean_member (json_node_get_object (document->root_node),
					field,
					value);
}

gint
couchdb_document_get_int_field (CouchdbDocument *document, const char *field)
{
	g_return_val_if_fail (COUCHDB_IS_DOCUMENT (document), -1);
	g_return_val_if_fail (field != NULL, -1);

	if (!json_object_has_member (json_node_get_object (document->root_node), field))
		return 0;

	return json_object_get_int_member (json_node_get_object (document->root_node),
					   field);
}

void
couchdb_document_set_int_field (CouchdbDocument *document, const char *field, gint value)
{
	g_return_if_fail (COUCHDB_IS_DOCUMENT (document));
	g_return_if_fail (field != NULL);

	json_object_set_int_member (json_node_get_object (document->root_node),
				    field,
				    value);
}

gdouble
couchdb_document_get_double_field (CouchdbDocument *document, const char *field)
{
	g_return_val_if_fail (COUCHDB_IS_DOCUMENT (document), -1);
	g_return_val_if_fail (field != NULL, -1);

	if (!json_object_has_member (json_node_get_object (document->root_node), field))
		return 0.0;
	return json_object_get_double_member (json_node_get_object (document->root_node),
					      field);
}

void
couchdb_document_set_double_field (CouchdbDocument *document, const char *field, gdouble value)
{
	g_return_if_fail (COUCHDB_IS_DOCUMENT (document));
	g_return_if_fail (field != NULL);

	json_object_set_double_member (json_node_get_object (document->root_node),
				       field,
				       value);
}

const char *
couchdb_document_get_string_field (CouchdbDocument *document, const char *field)
{
	g_return_val_if_fail (COUCHDB_IS_DOCUMENT (document), NULL);
	g_return_val_if_fail (field != NULL, NULL);

	if (!json_object_has_member (json_node_get_object (document->root_node), field))
		return NULL;

	return json_object_get_string_member (json_node_get_object (document->root_node),
					      field);
}

void
couchdb_document_set_string_field (CouchdbDocument *document, const char *field, const char *value)
{
	g_return_if_fail (COUCHDB_IS_DOCUMENT (document));
	g_return_if_fail (field != NULL);

	if (value) {
		json_object_set_string_member (json_node_get_object (document->root_node),
					       field,
					       value);
	} else {
		/* Remove field if it's a NULL value */
		couchdb_document_remove_field (document, field);
	}
}

CouchdbStructField *
couchdb_document_get_struct_field (CouchdbDocument *document, const char *field)
{
	g_return_val_if_fail (COUCHDB_IS_DOCUMENT (document), NULL);
	g_return_val_if_fail (field != NULL, NULL);

	if (!json_object_has_member (json_node_get_object (document->root_node), field))
		return NULL;

	return couchdb_struct_field_new_from_json_object (
		json_object_ref (json_object_get_object_member (json_node_get_object (document->root_node),
								field)));
}

void
couchdb_document_set_struct_field (CouchdbDocument *document, const char *field, CouchdbStructField *value)
{
	g_return_if_fail (COUCHDB_IS_DOCUMENT (document));
	g_return_if_fail (field != NULL);
	g_return_if_fail (value != NULL);

	json_object_set_object_member (json_node_get_object (document->root_node),
				       field,
				       json_object_ref (couchdb_struct_field_get_json_object (value)));
}

CouchdbStructField *
couchdb_document_get_application_annotations (CouchdbDocument *document)
{
	g_return_val_if_fail (COUCHDB_IS_DOCUMENT (document), NULL);

	return couchdb_document_get_struct_field (document, "application_annotations");
}

void
couchdb_document_set_application_annotations (CouchdbDocument *document, CouchdbStructField *annotations)
{
	g_return_if_fail (COUCHDB_IS_DOCUMENT (document));

	couchdb_document_set_struct_field (document, "application_annotations", annotations);
}

char *
couchdb_document_to_string (CouchdbDocument *document)
{
	g_return_val_if_fail (COUCHDB_IS_DOCUMENT (document), NULL);

	if (document->root_node) {
		JsonGenerator *generator;
		char *str;
		gsize size;

		generator = json_generator_new ();
		json_generator_set_root (generator, document->root_node);

		str = json_generator_to_data (generator, &size);
		g_object_unref (G_OBJECT (generator));

		return str;
	}

	return NULL;
}

JsonObject *
couchdb_document_get_json_object (CouchdbDocument *document)
{
	g_return_val_if_fail (COUCHDB_IS_DOCUMENT (document), NULL);
	
	return json_node_get_object (document->root_node);
}
