// Author: 
//   Mikael Hallendal <micke@imendio.com>
//
// (C) 2004 Imendio AB
// 

using System.Collections;
using System;
using System.Net;
using System.Xml.Serialization;
using System.ServiceModel.Syndication;

namespace Imendio.Blam {

    public interface IChannel
    {
        int NrOfItems {get; }
        int NrOfUnreadItems {get; }
        string Name {get; set; }
        string Url {get; set; }
        ArrayList ItemList {get; set; }
        Gtk.TreeIter Iter {get; set; }
        bool MarkAsRead();
        Item GetItem(string id);
        void Setup();
        event ChannelEventHandler Updated;
        void RemoveItems();
    }

    public class Channel : IChannel {
		[XmlAttribute("Name")] public string Int_Name = "";
		[XmlAttribute("Url")] public string Int_Url = "";

		// Used when updating the feed
		[XmlAttribute] public string LastModified = "";
		[XmlAttribute] public string ETag = "";
		[XmlAttribute] public string Type = "";
		[XmlAttribute] public string Keywords = "";
		[XmlAttribute] public DateTime LastRefreshed = new DateTime (0);
		
		// HTTP authentication options
		[XmlAttribute] public string http_username = "";
		[XmlAttribute] public string http_password = "";

        ItemStore store = null;
        ArrayList item_list = null;
        Object obj = new Object();

        public event ChannelEventHandler Updated;

        private Gtk.TreeIter mIter;

        public string Name {
            get {
                return Int_Name;
            }
            set {
                Int_Name = value;
            }
        }

        public string Url {
            get {
                return Int_Url;
            }
            set {
                Int_Url = value;
            }
        }

		public int NrOfItems {
			get {
                lock(obj){
                    return item_list.Count;
                }
			}
		}

        public int NrOfUnreadItems {
            get {
                int unread = 0;

                Item item;
                lock(obj){
                    foreach (string id in item_list) {
                        if(id != null){
                            item = store.Get(id);
                            if(item.Unread){
                                ++unread;
                            }
                        }
                    }
                }

                return unread;
            }
        }

        public int NrOfNewItems {
            get {
                int new_items = 0;
                Item item;
                lock(obj){
                    foreach(string id in item_list){
                        if(id == null){ /* TODO: Figure out why this happens */
                            continue;
                        }
                        item = store.Get(id);
                        if(item != null && item.Unread && !item.Old){
                            ++new_items;
                        }
                    }
                }

                return new_items;
            }
        }

        public ArrayList ItemList {
            get {
                lock(obj){
                    return item_list;
                }
            }
            set {
                lock(obj){
                    item_list = value;
                    Setup();
                }
            }
        }

		[XmlIgnore]
        public Gtk.TreeIter Iter {
            get {
                return mIter;
            }
            set {
                mIter = value;
            }
        }

        public Channel ()
        {
            item_list = new ArrayList();
            mIter = new Gtk.TreeIter();
            store = ItemStore.GetInstance();
        }

        public Channel (string name, string url) : this()
        {
            Name = name;
            Url = url;
        }

        public Item GetItem (string id)
        {
            return store.Get(id);
        }

        public void Setup()
        {
            lock(obj){
                ArrayList nlist = new ArrayList();
                foreach(string id in item_list){
                    if(id == null){
                        continue;
                    }
                    Item item = store.Get(id);
                    if(item == null){
                    } else {
                        nlist.Add(id);
                        item.Updated += ItemUpdated;
                    }
                }
                item_list = nlist;
            }
        }

		public void RemoveItems()
		{
			ItemStore store = ItemStore.GetInstance();
			lock(obj){
				foreach(string id in item_list){
					store.Remove(id);
				}
			}
		}

		public bool MarkAsRead ()
		{
			bool updated = false;
            Item item;

            lock(obj){
                foreach(string id in item_list){
                    if(id == null){
                        System.Console.WriteLine("null id {0} on {1}", id, Url);
                        continue;
                    }
                    item = store.Get(id);
                    if (item.Unread) {
                        item.SetUnread (false);
                        updated = true;
                        EmitUpdated();
                    }
                }
            }

			return updated;
		}

        public void EmitUpdated()
        {
            if(Updated != null){
                Updated(this);
            }
        }

        /**
         * Update this channel with the given feed
         */
        public bool Update(SyndicationFeed feed)
        {
            lock(obj){
                /* If our name is sub-optimal */
                if((Name == "" || Name == Url) &&
                   feed.Title.Text != null) {
                    Name = HtmlUtils.StripHtml(feed.Title.Text.Trim());
                }

                LastRefreshed = DateTime.Now;
                /*
                 * The new item list is just the new list, so create it
                 * on the spot
                 */
                ArrayList nitems = new ArrayList();
                foreach(SyndicationItem sitem in feed.Items){
					nitems.Add(sitem.Id);
                    /* If didn't have it already, add it to the store */
                    if(!item_list.Contains(sitem.Id)) {
                        store.Add(new Item(sitem));
					} else {
						if (sitem.Id == null)
							continue;
						
						store.Get(sitem.Id).Update(sitem);
					}
                }

                /* Delete the items that are no longer in the feed */
                foreach(string id in item_list){
                    if(!nitems.Contains(id)){
                        store.Remove(id);
                    }
                }

                item_list = nitems;
            }

            return true;
        }

        public void ItemUpdated(Item item)
        {
            EmitUpdated();
        }

        /**
         * Adds or updates the entry
         */
        public void Add(Item item)
        {
            if(item_list.Contains(item.Id)){
                /* In this case we only need to update */
                store.Get(item.Id).Update(item);
                return;
            }

            store.Add(item);
            lock(obj){
                item_list.Add(item.Id);
            }
            store.Get(item.Id).Updated += ItemUpdated;
        }

        /* Used to cross-mark as read */
        public void MarkItemIdAsRead (string id)
        {
            Item item = store.Get(id);
            if (item.Unread) {
                item.Unread = false;
                EmitUpdated();
            }
        }

		public bool GetHasKeyword (string keyword)
		{
			if (Keywords.IndexOf (keyword) >= 0) {
				return true;
			}

			return false;
		}
	}


}
