//
// Author: 
//   Mikael Hallendal <micke@imendio.com>
//
// (C) 2004 Imendio HB
// 

using Gdk;
using Gtk;
using GtkSharp;
using System;
using System.Collections;
using System.Runtime.InteropServices;

namespace Imendio.Blam {
    
    public class ChannelList : Gtk.TreeView {

	// Events
	public event ChannelEventHandler ChannelSelectedEvent;

	// Emitted when using right-click popup
	public event ChannelEventHandler RemoveChannelEvent;
	public event ChannelEventHandler MarkChannelAsReadEvent;
	public event ChannelEventHandler EditChannelEvent;
	public event ChannelEventHandler UpdateChannelEvent;
	
	private EventHandler selectionChangedHandler;

	private ChannelMenu popupMenu;

	private TreeViewColumn nameColumn;

	private IEnumerator channelEnumerator;
	
	public ChannelList(IList channels)
	{ 
	    TreeViewColumn col;
	    CellRenderer   cell;
	    
	    /* Channel name column */
	    nameColumn = new TreeViewColumn();
	    cell = new CellRendererText();

	    nameColumn.PackStart(cell, true);
	    nameColumn.AddAttribute (cell, "text", 1);
	    nameColumn.AddAttribute (cell, "weight", 2);
/*
	    nameColumn.SetCellDataFunc(cell, 
				       new TreeCellDataFunc(NameCellDataFunc));
*/
	    AppendColumn(nameColumn);

	    /* Items column */
	    col = new TreeViewColumn();
	    cell = new CellRendererText();

	    col.PackStart(cell, true);
	    col.SetCellDataFunc(cell,
				new TreeCellDataFunc(ItemsCellDataFunc));

	    AppendColumn(col);
	    
	    this.RulesHint = true;

	    selectionChangedHandler = new EventHandler(SelectionChanged);
	    this.Selection.Changed += selectionChangedHandler;
	   
	    this.Model = new ListStore (typeof(Channel), 
					typeof(string), typeof (int));
	    this.HeadersVisible = false;

	    // Sort the list
	    Utils.TreeSetSortFunc (this.Model.Handle, 1, 
				   new SortFunc (ChannelSortFunc));
	    ((ListStore)this.Model).SetSortColumnId (1, SortType.Ascending);	

	    // Right click popup
	    this.popupMenu = new ChannelMenu();
	    this.popupMenu.EditSelected       += EditChannelCb;
	    this.popupMenu.MarkAsReadSelected += MarkAsReadCb;
	    this.popupMenu.RemoveSelected     += RemoveChannelCb;
	    this.popupMenu.UpdateSelected     += UpdateChannelCb;
	    
	    channelEnumerator = channels.GetEnumerator();

	    GLib.Idle.Add(new GLib.IdleHandler(IdleAdd));
	}

	private bool IdleAdd()
	{
	    while (channelEnumerator.MoveNext ()) {
		Channel channel = (Channel)channelEnumerator.Current;
		Add(channel);
	    }
	
	    return false;
	}
	
	public void Add (Channel channel)
	{
	    int weight = (int)Pango.Weight.Normal;

	    if (channel.NrOfUnreadItems > 0) {
		weight = (int)Pango.Weight.Bold;
	    }
	    
	    ((ListStore)this.Model).AppendValues(channel, channel.Name, weight);
	}

	public Channel GetSelectedChannel ()
	{
	    TreeModel model;
	    TreeIter  iter;
	    
	    if (!this.Selection.GetSelected(out model, out iter)) {
		return null;
	    }

	    return (Channel)model.GetValue(iter, 0);
	}

	public void Updated (Channel channel)
	{
	    TreeIter iter = FindChannel (channel);
	    
	    if (!iter.Equals(TreeIter.Zero)) {
		int weight = (int)Pango.Weight.Normal;

		if (channel.NrOfUnreadItems > 0) {
		    weight = (int)Pango.Weight.Bold;
		}

		string oldName = (string) this.Model.GetValue (iter, 1);
		if (oldName != channel.Name) {
		    this.Model.SetValue (iter, 1, channel.Name);
		}
		this.Model.SetValue (iter, 2, weight);

		this.Model.EmitRowChanged (this.Model.GetPath(iter), iter);
	    } 
	}

	public void Remove (Channel channel)
	{
	    TreeIter iter = FindChannel(channel);
	    bool valid;

	    if (!iter.Equals(TreeIter.Zero)) {
		this.Selection.Changed -= selectionChangedHandler;
		
		valid = ((ListStore)this.Model).Remove(ref iter);
		
		this.Selection.Changed += selectionChangedHandler;
		
		if (!valid) {
		    EmitChannelSelected(null);
		    return;
		}

		this.Selection.SelectIter(iter);
	    }
	}

	public bool NextUnread ()
	{
	    TreeModel model;
	    TreeIter  iter;
	   
	    if (this.Selection.GetSelected(out model, out iter)) {
		Channel channel = (Channel)model.GetValue(iter, 0);
		if (channel.NrOfUnreadItems > 0) {
		    return true;
		} 
		
		if (!this.Model.IterNext(ref iter)) {
		    if (!this.Model.GetIterFirst(out iter)) {
			return false;
		    }
		}
	    } else {
		if (!this.Model.GetIterFirst(out iter)) {
		    return false;
		}
	    }

	    TreeIter startIter = iter.Copy();
	    
	    do {
		Channel channel = (Channel)model.GetValue(iter, 0);
		if (channel.NrOfUnreadItems > 0) {
		    this.Selection.SelectIter(iter);
		    ScrollToCell(this.Model.GetPath(iter), nameColumn,
				 false, 0, 0);
		    return true;
		}
		
		if (!this.Model.IterNext(ref iter)) {
		    this.Model.GetIterFirst(out iter);
		}
	    } while (!iter.Equals(startIter));

	    return false;
	}

	private void EditChannelCb()
	{
	    EmitEditChannelEvent(GetSelectedChannel());
	}

	private void MarkAsReadCb ()
	{
	    if (MarkChannelAsReadEvent != null) {
		MarkChannelAsReadEvent (GetSelectedChannel ());
	    }
	}

	private void RemoveChannelCb ()
	{
	    if (RemoveChannelEvent != null) {
		RemoveChannelEvent (GetSelectedChannel ());
	    }
	}
	
	private void UpdateChannelCb ()
	{
	    if (UpdateChannelEvent != null) {
		UpdateChannelEvent (GetSelectedChannel ());
	    }
	}
	
	private int ChannelSortFunc (IntPtr a, IntPtr b)
	{
	    TreeIter iterA = TreeIter.New (a);
	    TreeIter iterB = TreeIter.New (b);

	    Channel channelA = (Channel)this.Model.GetValue (iterA, 0);
	    Channel channelB = (Channel)this.Model.GetValue (iterB, 0);

	    return channelA.Name.CompareTo (channelB.Name);
	}

	protected override bool OnButtonPressEvent (EventButton bEvent)
	{
	    switch (bEvent.Button) {
	    case 1:
		if (bEvent.Type == EventType.TwoButtonPress) {
		    EditChannelCb();
		} else {
		    return base.OnButtonPressEvent (bEvent);
		}
		break;
	    case 3:
		TreePath path;

		if (!GetPathAtPos ((int) bEvent.X, (int) bEvent.Y, out path)) {
		    return false;
		}

		this.Selection.SelectPath (path);

		popupMenu.Popup((uint)bEvent.XRoot, (uint)bEvent.YRoot, 
				bEvent.Time);
		return false;
	    }

	    return false;
	}
	
	private void SelectionChanged(object obj, EventArgs args)
	{
	    TreeSelection selection = (TreeSelection)obj;
	    TreeIter      iter;
	    TreeModel     model;
	    Channel       channel;

	    if (!selection.GetSelected(out model, out iter)) {
		EmitChannelSelected(null);
		return;
	    }

	    channel = (Channel)model.GetValue(iter, 0);
	    if (channel != null) {
		EmitChannelSelected(channel);
	    }
	}

	private void NameCellDataFunc(TreeViewColumn col,
				      CellRenderer   cell,
				      TreeModel      model,
				      TreeIter       iter)
	{
	    Channel channel = (Channel) model.GetValue (iter, 0);
	    int weight = (int) Pango.Weight.Normal;

	    ((CellRendererText)cell).Text = channel.Name;

	    if (channel.NrOfUnreadItems > 0) {
		weight = (int)Pango.Weight.Bold;
	    }

	    ((CellRendererText)cell).Weight = weight;
	}
	
	private void ItemsCellDataFunc(TreeViewColumn col,
				       CellRenderer   cell,
				       TreeModel      model,
				       TreeIter       iter)
	{
	    Channel channel = (Channel)model.GetValue(iter, 0);
	    int weight = (int)Pango.Weight.Normal;

	    ((CellRendererText)cell).Text = channel.NrOfUnreadItems + "/" + channel.NrOfItems;
	    if (channel.NrOfUnreadItems > 0) {
		weight = (int)Pango.Weight.Bold;
	    }

	    ((CellRendererText)cell).Weight = weight;
	}

	private void EmitChannelSelected(Channel channel)
	{
	    if (ChannelSelectedEvent != null) {
		ChannelSelectedEvent(channel);
	    }
	}

	// Used by ChannelUpdated
	private Channel findChannel;
	private TreeIter foundIter;
	private bool ForeachFindChannel(TreeModel model, 
					TreePath  path,
					TreeIter  iter)
	{
	    Channel channel = (Channel)model.GetValue(iter, 0);
	    if (channel == findChannel) {
		foundIter = iter;
		return true;
	    }

	    return false;
	}

	private TreeIter FindChannel(Channel channel) 
	{
	    findChannel = channel;
	    foundIter = TreeIter.Zero;
	
	    this.Model.Foreach(new TreeModelForeachFunc(ForeachFindChannel));
	
	    return foundIter.Copy();
	}

	private void EmitEditChannelEvent(Channel channel)
	{
	    if (EditChannelEvent != null) {
		EditChannelEvent(channel);
	    }
	}
  }

    class ChannelMenu {
	public delegate void MenuItemSelectedHandler();

	public event MenuItemSelectedHandler EditSelected;
	public event MenuItemSelectedHandler MarkAsReadSelected;
	public event MenuItemSelectedHandler RemoveSelected;
	public event MenuItemSelectedHandler UpdateSelected;

	IntPtr Handle;
	
	private delegate void CallbackHandler(uint action);

	[DllImport ("libblam.so")]
	    static extern IntPtr blam_channel_menu_new (CallbackHandler menu_cb);
	[DllImport ("libblam.so")]
	    static extern void   blam_channel_menu_popup   (IntPtr obj,
							    uint   x,
							    uint   y,
							    uint   time_);
	[DllImport ("libblam.so")]
	    static extern void   blam_channel_menu_destroy (IntPtr obj);

	public ChannelMenu()
	{
	    Handle = blam_channel_menu_new(new CallbackHandler(this.MenuCb));
	}

	~ChannelMenu()
	{
	    blam_channel_menu_destroy(Handle);
	}
	
	public void Popup(uint x, uint y, uint time_)
	{
	    blam_channel_menu_popup(Handle, x, y, time_);
	}

	private void MenuCb(uint action)
	{
	    switch (action) {
	    case 1:
		if (EditSelected != null) {
		    EditSelected();
		}
		break;
	    case 2:
		if (MarkAsReadSelected != null) {
		    MarkAsReadSelected ();
		}
		break;
	    case 3:
		if (RemoveSelected != null) {
		    RemoveSelected();
		}
		break;
	    case 4:
		if (UpdateSelected != null) {
		    UpdateSelected();
		}
		break;
	    }
	}
    }
}

