/*
 * Copyright 2000 Murray Cumming
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the Free
 * Software Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <bakery/App/App_WithDoc.h>
#include <bakery/App/Dialog_OfferSave.h>
#include <bakery/App/Dialog_OverwriteFile.h>
#include <gconfmm.h>
#include <libintl.h>
#include <fstream>
#include <algorithm>

//#include <gtk/gtkfilesel.h>

#define BAKERY_GCONF_DIRECTORY_RECENTFILES "recent_files"

namespace Bakery
{


//Initialize static member data:
App_WithDoc::type_vecStrings App_WithDoc::m_vecDocHistory;

App_WithDoc::App_WithDoc(const Glib::ustring& appname)
: App(appname)
{  
  m_pDocument = 0;
  m_bCloseAfterSave = false;

  m_vecDocHistory.resize(4); //TODO: Make the size a user-defined preference.
  
  //If this is the first instance, then load the configuration:
  //if(m_AppInstanceManager.get_app_count() == 0)
    //session_load_document_history();
}

App_WithDoc::~App_WithDoc()
{
  //Delete the document:
  if(m_pDocument)
  {
    delete m_pDocument;
    m_pDocument = 0;
  }
}

void App_WithDoc::on_menu_file_close()
{
  if(m_pDocument->get_modified())
  {
    //Offer to save changes:
    m_bCloseAfterSave = true; //Checked in FileSelection signal handler.
    offer_to_save_changes(); //If a File|Exit is in progress, this could cancel it.
  }

  if(!get_operation_cancelled())
    ui_hide(); //The AppInstanceManager will delete it.
}


bool App_WithDoc::open_document(const Glib::ustring& strFilepath)
{
  //Check whether it's already open:
  //It could even be open in this instance.
  bool bAlreadyOpen = false;
  App_WithDoc* pAppAlreadyOpen = 0;

  AppInstanceManager::type_listAppInstances apps =  m_AppInstanceManager.get_instances();
  for(AppInstanceManager::type_listAppInstances::iterator iter = apps.begin(); iter != apps.end(); iter++)
  {
    App_WithDoc* pApp = dynamic_cast<App_WithDoc*>(*iter);
    if(pApp)
    {
      Document* pDoc = pApp->get_document();
      if(pDoc->get_filepath() == strFilepath)
      {
        bAlreadyOpen = true;
        pAppAlreadyOpen = pApp;
      }
    }
  }

  if(bAlreadyOpen)
  {
    //Bring it to the front:
    if(pAppAlreadyOpen)
    {
      pAppAlreadyOpen->ui_bring_to_front();
    }

    //Tell user that it's already open:
    ui_warning(gettext("This document is already open."));

    return true; //success.
  }
  else
  {
    //Open it:
        
    //Load it into a new instance unless the current document is just a default new.
    App_WithDoc* pApp = 0;
    bool bUsingNewInstance = false;
    if(!(get_document()->get_is_new())) //if it's not new.
    {
      //New instance:
      pApp = dynamic_cast<App_WithDoc*>(new_instance());
      pApp->init(); //It's shown too.
      bUsingNewInstance = true;
    }
    else
    {
      pApp = this; //Replace the default new document in this instance.
    }

    //Open it.
    pApp->m_pDocument->set_filepath(strFilepath);
    bool bTest = pApp->m_pDocument->load();

    if(!bTest) //if open failed.
    {
      ui_warning(gettext("Open failed."));

      if(bUsingNewInstance)
      {
        //Remove new instance:
        pApp->get_document()->set_modified(false); //avoid 'do you want to save?' dialog.
        pApp->on_menu_file_close();
      }
      else
      {
        //re-initialize document.
        delete pApp->m_pDocument;
        pApp->m_pDocument = 0;
        pApp->init_create_document();
      }

      return false; //failed.
    }
    else
    {
      //if open succeeded:
      pApp->on_document_load();
      pApp->update_window_title();
      set_document_modified(false); //disables menu and toolbar Save items.

      //Update document history list:
      document_history_add(strFilepath);

      return true; //success.
    }
  } //if already open.

  return false; //failed.
}

void App_WithDoc::on_menu_file_open()
{
  //Display File Open dialog and respond to choice:

  //Bring document window to front, to make it clear which document is being changed:
  ui_bring_to_front();

  //Ask user to choose file to open:
  Glib::ustring strFilepath = ui_file_select_open();
  if(!strFilepath.empty())
    open_document(strFilepath);
}

void  App_WithDoc::on_menu_file_open_recent(guint index)
{
  if(index < m_vecDocHistory.size())
  {
    const Glib::ustring& strFilepath = m_vecDocHistory[index];
    bool bTest = open_document(strFilepath);

    //Remove it from the recent documents list if it couldn't be opened.
    if(!bTest)
      document_history_remove(strFilepath);
  }
}

//This is for calling directly, use the other override for the signal handler:
void App_WithDoc::offer_saveas()
{
  on_menu_file_saveas();
}

void App_WithDoc::on_menu_file_saveas()
{
  //Display File Save dialog and respond to choice:
   
  //Bring document window to front, to make it clear which document is being saved:
  //This doesn't work: TODO.
  ui_bring_to_front();

  //Show the save dialog:  
  const Glib::ustring& strFilepathOld = m_pDocument->get_filepath();
  Glib::ustring strFilepath = ui_file_select_save(strFilepathOld);
  if(!strFilepath.empty())
  {
    bool bUseThisFilePath = true;

    //Check whether file exists already:
    {
      std::ifstream fStream(strFilepath.c_str());
      if(fStream.is_open())
      {
        //Ask user to confirm overwrite:
        bool bOverwrite = ui_ask_overwrite(strFilepath);

        //Respond to button that was clicked:
        bUseThisFilePath = bOverwrite;
      }
    }

    //Save to this filepath:
    if(bUseThisFilePath)
    {
      m_pDocument->set_filepath(strFilepath, true); //true = enforce file path
      bool bTest  = m_pDocument->save();

      if(!bTest)
      {
        ui_warning(gettext("Save failed."));
      }
      else
      {
        //Disable Save and SaveAs menu items:
        after_successful_save();
      }

      update_window_title();


      //Close if this save was a result of a File|Close or File|Exit:.
      //if(bTest && m_bCloseAfterSave) //Don't close if the save failed.
      //{
      //  on_menu_file_close(); //This could be the second time, but now there are no unsaved changes.
      //}
    }
    else
    {
      //Let the user choose a different file path,
      //because he decided not to overwrite the 1st one.
      offer_saveas(); //recursive.
    }
  }
  else
  {
    cancel_close_or_exit();
  }
}

void App_WithDoc::on_menu_file_save()
{
  if(m_pDocument)
  {
    //If there is already a filepath, then save to that location:
    if(!(m_pDocument->get_filepath().empty()))
    {
      bool bTest = m_pDocument->save();

      if(bTest)
      {
        //Disable Save and SaveAs menu items:
        after_successful_save();    
   
        //Close the document if this save was in response to a 'Do you want to save before closing?':
        //if(m_bCloseAfterSave) // || m_bExiting
        //  close_mark_or_destroy();
      }
      else
      {
        //The save failed. Tell the user and don't do anything else:
        ui_warning(gettext("Save failed."));
        
        cancel_close_or_exit();
      }
    }
    else
    {
      //If there is no filepath, then ask for one and save to that location:
      offer_saveas();
    }
  }

  if(!m_bCloseAfterSave) //Don't try to do anything after closing - this instance would not exist anymore.
  {
    update_window_title();
  }

}

void App_WithDoc::init()
{
  init_create_document();

  //Call base method:
  App::init();

  on_document_load(); //Show default empty document in the View.

  set_document_modified(false); //Disable Save menu item.

  //Fill the Recent Documents sub menu:
  session_load_document_history();
}

void App_WithDoc::init_create_document()
{
  //Overrides should call this base method at the end.

  if(!m_pDocument)
  {
    m_pDocument = new Document();
  }

  m_pDocument->set_is_new(true); //Can't be modified if it's just been created.
  
  m_pDocument->signal_modified().connect(SigC::slot(*this, &App_WithDoc::on_document_modified));

  update_window_title();
}

Document* App_WithDoc::get_document()
{
  return m_pDocument;
}

const Document* App_WithDoc::get_document() const
{
  return m_pDocument;
}

void App_WithDoc::offer_to_save_changes()
{
  if(m_pDocument)
  {
    if(m_pDocument->get_modified())
    {
      set_operation_cancelled(false); //Initialize it again. It might be set later in this method by cancel_close_or_exit().
      
      //The document has unsaved changes,
      //so ask the user whether the document should be saved:
      enumSaveChanges buttonClicked = ui_offer_to_save_changes();

      //Respond to button that was clicked:
      switch(buttonClicked)
      {
        case(SAVECHANGES_Save):
        {
          on_menu_file_save(); //If File|Exit is in progress, this could cancel it.
          break;
        }
        
        case(SAVECHANGES_Discard):
        {
          //Close if this save offer was a result of a FileClose (It probably always is):
          //close_mark_or_destroy();
          //Do nothing - the caller will probably hide() the window to have it deleted.
          break;
        }
        
        case(SAVECHANGES_Cancel): //Cancel.
        {
          cancel_close_or_exit();
          break;
        }
        
        default:
        {
          break;
        }
      }   
    }
  }
}

void App_WithDoc::close_mark_or_destroy()
{
  ui_hide();
}

void App_WithDoc::cancel_close_or_exit()
{
  set_operation_cancelled();
  m_bCloseAfterSave = false;
  m_AppInstanceManager.cancel_close_all();

  //exit_destroy_marked_instances(); //Clean up after an exit.
}
  
void App_WithDoc::on_document_load()
{        
  //Show document contents:
  if(m_pDocument)
  {
    Bakery::ViewBase* pView = m_pDocument->get_view();
    if(pView)
      pView->load_from_document();

    //Set document as unmodified (this could have been wrongly set during the load):
    set_document_modified(false);
  }
  
  //If you are not using Views, then override this to fill your various windows with stuff according to the contents of the document.
}

void App_WithDoc::update_window_title()
{

}

void App_WithDoc::on_document_modified()
{
  //Change the displayed 'modified' status.
  //This method could be overridden to e.g. enable a Save icon or enable the Save menu item.
  //TODO: enable/disable the Save menu item.

  ui_show_modification_status();

  //Change title bar:
  update_window_title();
}

void App_WithDoc::set_document_modified(bool bModified /* = true */)
{
  m_pDocument->set_modified(bModified);

  //Enable/Disable Save menu item and toolbar item:
  ui_show_modification_status();
}

void App_WithDoc::on_menu_edit_copy()
{
  Bakery::ViewBase* pView = m_pDocument->get_view();
  if(pView)
    pView->clipboard_copy();
}

void App_WithDoc::on_menu_edit_paste()
{
  Bakery::ViewBase* pView = m_pDocument->get_view();
  if(pView)
    pView->clipboard_paste();
}

void App_WithDoc::on_menu_edit_clear()
{
  Bakery::ViewBase* pView = m_pDocument->get_view();
  if(pView)
    pView->clipboard_clear();
}

void App_WithDoc::after_successful_save()
{
  set_document_modified(false); //enables/disables menu and toolbar widgets.
    
  //Update document history list:
  document_history_add(m_pDocument->get_filepath());     
}

Glib::ustring App_WithDoc::get_conf_fullkey(const Glib::ustring& key)
{
  return "/apps/" + m_strAppName + "/" + key;
}


void App_WithDoc::session_save_document_history()
{
  //Save document history:

  Glib::RefPtr<Gnome::Conf::Client> refConfClient = Gnome::Conf::Client::get_default_client();
  
  try
  {
    Glib::ustring key = get_conf_fullkey(BAKERY_GCONF_DIRECTORY_RECENTFILES);
    refConfClient->set_string_list(key, m_vecDocHistory);
  }
  catch(const Gnome::Conf::Error& ex)
  {
    std::cerr << "Bakery: App_WithDoc::session_save_document_history(): " << ex.what() << std::endl;
  }
}

void App_WithDoc::session_load_document_history()
{
  //Load document history:

  Glib::RefPtr<Gnome::Conf::Client> refConfClient = Gnome::Conf::Client::get_default_client();

  try
  {
    type_vecStrings vecTemp = refConfClient->get_string_list(get_conf_fullkey( BAKERY_GCONF_DIRECTORY_RECENTFILES ));

    //Check that it has not become corrupted:
    if(vecTemp.size() == m_vecDocHistory.size())
      m_vecDocHistory = vecTemp;
  }
  catch(const Gnome::Conf::Error& ex)
  {
    //Maybe the directory doesn't exist yet. It's not a problem.
    //std::cerr << "Bakery: App_WithDoc::session_save_document_history(): " << ex.what() << std::endl;
  }

  show_document_history_in_all_instances();
}


void App_WithDoc::document_history_add(const Glib::ustring& strFilepath)
{
  //TODO synchronize this static data.

  //If it's not already there:
  type_vecStrings::iterator iterFind = std::find(m_vecDocHistory.begin(), m_vecDocHistory.end(), strFilepath);
  if(iterFind == m_vecDocHistory.end())
  {
    guint sizeHistory = m_vecDocHistory.size(); //Remember size so that we can keep it the same.
    
    //Add new item:
    m_vecDocHistory.insert(m_vecDocHistory.begin(), strFilepath); //Inserts at the start, before begin().
 
    //Enforce size, possibly forgetting old history at the end:
    m_vecDocHistory.resize(sizeHistory);
   
    session_save_document_history();

    show_document_history_in_all_instances();
  }
}

void App_WithDoc::document_history_remove(const Glib::ustring& strFilepath)
{
  //TODO synchronize this static data.

  guint sizeHistory = m_vecDocHistory.size(); //Remember size so that we can keep it the same.

  //Remove it if it's there:
  type_vecStrings::iterator iterFind = std::find(m_vecDocHistory.begin(), m_vecDocHistory.end(), strFilepath);
  if(iterFind != m_vecDocHistory.end())
  {
    m_vecDocHistory.erase(iterFind);

    //Enforce size:
    m_vecDocHistory.resize(sizeHistory);

    session_save_document_history();

    show_document_history_in_all_instances();
  }
}


void App_WithDoc::show_document_history()
{
  //Override this.
}


  
void App_WithDoc::show_document_history_in_all_instances() //static
{
  //Update File menu in all instances:
  AppInstanceManager::type_listAppInstances apps = m_AppInstanceManager.get_instances();
  for(AppInstanceManager::type_listAppInstances::iterator iter = apps.begin(); iter != apps.end(); iter++)
  {
    App_WithDoc* pApp = dynamic_cast<App_WithDoc*>(*iter);
    if(pApp)
      pApp->show_document_history();
  }
}

} //namespace
