/* Atomixed -- the level-/themeeditor for atomix.
 * Copyright (C) 1999-2000 Jens Finke
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */
#include "playfield_editor.h"
#include "canvas_helper.h"
#include "image_win_ctrl.h"
#include "level_ctrl.h"
#include "callbacks.h"
#include "support.h"
#include "tile.h"
#include "main.h"
#include "util.h"
#include <gnome.h>
#include <math.h>

typedef struct _CallbackData CallbackData;
struct _CallbackData
{
	PlayFieldEditor *pfe;  /* current playfield editor */
	GnomeCanvasItem *item; /* with right button selected item */
};

typedef struct _MoveItemData MoveItemData;
struct _MoveItemData
{
	GnomeCanvasItem *item;
	PlayField *pf_source;
	Tile *tile;
	gint src_row;
	gint src_col;
};

/*=================================================================
 
  Declaration of internal functions

  ---------------------------------------------------------------*/
void playfield_editor_render(PlayFieldEditor *pfe);

void playfield_editor_create_grid(PlayFieldEditor *pfe);

void playfield_editor_clear_grid(PlayFieldEditor *pfe);

GnomeCanvasItem* create_obstacle (PlayFieldEditor *pfe, 
				  double x, double y, Tile *tile);

GnomeCanvasItem* create_moveable (PlayFieldEditor *pfe, 
				  double x, double y, Tile *tile);

GnomeCanvasItem* create_item (GdkImlibImage* img, double x, double y,
			      GnomeCanvasGroup* group, PlayFieldEditor *pfe);

gint mouse_event (GnomeCanvasItem *item, GdkEvent *event, PlayFieldEditor *pfe);

void shift_mouse_event(GdkEvent *event, PlayFieldEditor *pfe, 
		       gint start_row, gint start_col, gint end_row, gint end_col);

void handle_item_event (GnomeCanvasItem *item, GdkEvent *event, PlayFieldEditor *pfe);

void handle_grid_item_event (GnomeCanvasItem *item, GdkEvent *event, PlayFieldEditor *pfe);

gint item_move_event(GnomeCanvasItem *item, GdkEvent *event, PlayFieldEditor *pfe);

void item_context_menu(PlayFieldEditor *pfe, GnomeCanvasItem *item, GdkEvent *event);

void grid_context_menu(PlayFieldEditor *pfe, GnomeCanvasItem *item, GdkEvent *event);

void set_tile(GnomeCanvasItem *grid_item, PlayFieldEditor *pfe, Tile *tile);

void replace_tile(GnomeCanvasItem *item, PlayFieldEditor *pfe, Tile *tile);

void
on_context_copy_activate               (GtkMenuItem     *menuitem,
                                        CallbackData *data);

void
on_context_cut_activate                (GtkMenuItem     *menuitem,
                                        CallbackData *data);

void
on_context_clear_activate              (GtkMenuItem     *menuitem,
                                        CallbackData *data);

void 
insert_playfield(PlayFieldEditor *pfe, gdouble mouse_x, gdouble mouse_y,
		 PlayField *pf, GnomeCanvasItem *item);

void 
insert_tile(PlayFieldEditor *pfe, gdouble mouse_x, gdouble mouse_y,
	    MoveItemData *data);

void 
initiate_move_item(GnomeCanvasItem *item, PlayFieldEditor *pfe);

/*=================================================================
 
  Creation, initialisation and clean up

  ---------------------------------------------------------------*/

PlayFieldEditor* playfield_editor_new(gchar *canvas_name, 
				      gboolean anm)
{
	PlayFieldEditor *pfe = NULL;

	pfe = g_malloc(sizeof(PlayFieldEditor));
	
	if(pfe)
	{
		GnomeCanvas *canvas;

		pfe->canvas_name = g_strdup(canvas_name);
		pfe->theme = NULL;
		pfe->allow_non_moveables = anm;
		pfe->show_grid = TRUE;
		pfe->pf = NULL;
		pfe->map = NULL;
		pfe->grid_map = NULL;
		pfe->last_clicked_row = 0;
		pfe->last_clicked_col = 0;
		pfe->mode = PFE_NORMAL;
		
		canvas = GNOME_CANVAS(lookup_widget(get_app(), canvas_name));
		pfe->grid = create_group_ref(canvas, NULL);
		pfe->obstacles = create_group_ref(canvas, NULL);
		pfe->moveables = create_group_ref(canvas, NULL);
		gnome_canvas_item_lower_to_bottom(GNOME_CANVAS_ITEM(pfe->grid));

	}
	
	return pfe;
}

void playfield_editor_init(PlayFieldEditor *pfe)
{
}


void playfield_editor_destroy(PlayFieldEditor *pfe)
{
	if(pfe)
	{
		if(pfe->map)
		{
			canvas_map_destroy(pfe->map);
		}
		if(pfe->grid_map)
		{
			canvas_map_destroy(pfe->grid_map);
		}
		if(pfe->moveables)
		{
			gtk_object_destroy(GTK_OBJECT(pfe->moveables));
		}
		if(pfe->obstacles)
		{
			gtk_object_destroy(GTK_OBJECT(pfe->obstacles));
		}
		if(pfe->grid)
		{
			gtk_object_destroy(GTK_OBJECT(pfe->grid));
		}
		g_free(pfe);
	}
}

/*=================================================================
  
  Editor functions

  ---------------------------------------------------------------*/

void playfield_editor_clear(PlayFieldEditor *pfe)
{
	if(pfe->map)
	{
		canvas_map_destroy(pfe->map);
	}
	if(pfe->moveables)
	{
		gtk_object_destroy(GTK_OBJECT(pfe->moveables));
	}
	if(pfe->obstacles)
	{
		gtk_object_destroy(GTK_OBJECT(pfe->obstacles));
	}

	pfe->obstacles = create_group(pfe->canvas_name, NULL);
	pfe->moveables = create_group(pfe->canvas_name, NULL);	
	pfe->map = canvas_map_new();
}

void playfield_editor_set_playfield(PlayFieldEditor *pfe, PlayField *pf)
{
	gint width, height;
	gint tile_width, tile_height;

	g_return_if_fail(pfe != NULL);
	g_return_if_fail(pf != NULL);

	pfe->pf = pf;		
	playfield_editor_clear(pfe);
	playfield_editor_clear_grid(pfe);
	playfield_editor_render(pfe);
	playfield_editor_create_grid(pfe);	       		
	
	theme_get_tile_size(pfe->theme, &tile_width, &tile_height);
	width = pf->n_cols * tile_width;
	height = pf->n_rows * tile_height;		
	set_canvas_dimensions(pfe->canvas_name, width, height);
	set_background_color(pfe->canvas_name, 
			     theme_get_background_color(pfe->theme));
}

void playfield_editor_resize(PlayFieldEditor *pfe, guint n_rows, guint n_cols)
{
	g_return_if_fail(pfe!=NULL);

	if((pfe->pf->n_rows != n_rows) || (pfe->pf->n_cols != n_cols))
	{
		g_print("Resize Playfield to %ix%i\n", n_rows, n_cols);	
		playfield_set_matrix_size(pfe->pf, n_rows, n_cols);
		playfield_editor_set_playfield(pfe, pfe->pf);

		/* mark level as modified */
		level_ctrl_set_actual_level_modified();
	}
}


void playfield_editor_set_theme(PlayFieldEditor *pfe, Theme *theme, gboolean update)
{
	g_return_if_fail(pfe!=NULL);
	g_return_if_fail(theme!=NULL);

	pfe->theme = theme;

	if(update)
	{
		set_background_color(pfe->canvas_name, 
				     theme_get_background_color(pfe->theme));
		playfield_editor_clear(pfe);
		playfield_editor_clear_grid(pfe);
		playfield_editor_render(pfe);
		playfield_editor_create_grid(pfe);
	}
}


void playfield_editor_show_grid(PlayFieldEditor *pfe, gboolean show)
{
	if(show)
	{
		gnome_canvas_item_show(GNOME_CANVAS_ITEM(pfe->grid));
	}
	else
	{
		gnome_canvas_item_hide(GNOME_CANVAS_ITEM(pfe->grid));
	}
	pfe->show_grid = show;
}

void playfield_editor_insert_pf(PlayFieldEditor *pfe, PlayField *pf)
{
	PlayFieldEditor *pfe_dummy;
	MoveItemData *data;
	gint row, col;
	GtkWidget *canvas;

	canvas = lookup_widget(get_app(), pfe->canvas_name);

	/* create data structure */
	data = g_malloc(sizeof(MoveItemData));
	data->pf_source = pf;

	/* create dummy structure */
	pfe_dummy = g_malloc(sizeof(PlayFieldEditor));
	pfe_dummy->canvas_name = pfe->canvas_name;
	pfe_dummy->theme = pfe->theme;
	pfe_dummy->pf = pf;

	data->item = GNOME_CANVAS_ITEM(create_group_ref(GNOME_CANVAS(canvas), NULL));
	pfe_dummy->obstacles = create_group_ref(GNOME_CANVAS(canvas), 
						GNOME_CANVAS_GROUP(data->item));
	pfe_dummy->moveables = create_group_ref(GNOME_CANVAS(canvas), 
						GNOME_CANVAS_GROUP(data->item));
	pfe_dummy->grid = NULL;

       	for(row = 0; row < pf->n_rows; row++)
	{
		for(col = 0; col < pf->n_cols; col++)
		{
			GnomeCanvasItem *item = NULL;
			Tile *tile = playfield_get_tile(pf, row, col);
			TileType type = tile_get_type(tile);
			gdouble x,y;
			
			switch(type)
			{
			case TILE_MOVEABLE:
				convert_to_canvas(pfe->theme, 
						  row, col, &x, &y);
				item = create_moveable(pfe_dummy, x, y, tile);
				gtk_signal_disconnect_by_func(GTK_OBJECT(item),
							      (GtkSignalFunc)mouse_event,
							      pfe_dummy);
				break;
				
			case TILE_OBSTACLE:
				if(pfe->allow_non_moveables)
				{
					convert_to_canvas(pfe->theme, 
							  row, 
							  col, 
							  &x, &y);
					item = create_obstacle(pfe_dummy, 
							       x, y, 
							       tile);
					gtk_signal_disconnect_by_func(GTK_OBJECT(item),
								      (GtkSignalFunc)mouse_event,
								      pfe_dummy);
				}
				break;
				
			case TILE_NONE:
				default:
			}				
			
		}		
	}

	/* destroy dummy */
	g_free(pfe_dummy);
	
	pfe->mode = PFE_INSERT;
	pfe->mode_data = data;

	gtk_signal_connect(GTK_OBJECT(canvas), "motion-notify-event",
			   (GtkSignalFunc) item_move_event, pfe);
	gtk_signal_connect_after(GTK_OBJECT(canvas), "button-press-event",
			   (GtkSignalFunc) item_move_event, pfe);
	gnome_canvas_item_raise_to_top(GNOME_CANVAS_ITEM(data->item));

	/* move to cursor position */
	{
		gint mouse_x, mouse_y;
		gdouble world_x, world_y;
		gdouble x1, y1, x2, y2;
		gdouble diff_x, diff_y;

		gdk_window_get_pointer(canvas->window, &mouse_x, &mouse_y, NULL);
		gnome_canvas_window_to_world(GNOME_CANVAS(canvas), mouse_x, mouse_y,
					     &world_x, &world_y);		
		gnome_canvas_item_get_bounds(GNOME_CANVAS_ITEM(data->item), &x1, &y1, &x2, &y2);
		diff_x = (x2 - x1) / 2;
		diff_y = (y2 - y1) / 2;
		gnome_canvas_item_move(GNOME_CANVAS_ITEM(data->item), 
				       world_x - diff_x, world_y - diff_y);
	}

	gdk_pointer_grab(canvas->window, 
			 TRUE,
			 GDK_POINTER_MOTION_MASK | GDK_BUTTON_PRESS_MASK,
			 canvas->window,
			 NULL,
			 GDK_CURRENT_TIME);
}

/*=================================================================
 
  Internal creation functions

  ---------------------------------------------------------------*/
GnomeCanvasItem*
create_obstacle (PlayFieldEditor *pfe, double x, double y, Tile *tile)
{
	GdkImlibImage *img = NULL;
	GnomeCanvasItem *item = NULL;
	
	img = theme_get_tile_image(pfe->theme, tile);
	
	if(img) {
		item = create_item (img, x, y, pfe->obstacles, pfe);
		gtk_signal_connect(GTK_OBJECT(item), "event",
				   (GtkSignalFunc) mouse_event, pfe);
	} 
#ifdef DEBUG
	else {
		g_print("Obstacle image not found!\n");
	}
#endif    
	
	return item;
}

GnomeCanvasItem*
create_moveable (PlayFieldEditor *pfe, double x, double y, Tile *tile)
{
	GdkImlibImage *img = NULL;
	GdkImlibImage  *conn_img = NULL;
	GnomeCanvasItem *item = NULL;
	GnomeCanvasGroup *moveable_group = NULL;
	GSList *conn_images;
	
	img = theme_get_tile_image(pfe->theme, tile);
	conn_images = theme_get_tile_connection_images(pfe->theme, tile);
		
	if(img) 
	{
		gint i;

		/* create new group at (0,0) */
		moveable_group = create_group(pfe->canvas_name, pfe->moveables);

		/* add connection and moveable images */
		if(conn_images)
		{
			for(i = 0; i < g_slist_length(conn_images); i++)
			{
				conn_img = (GdkImlibImage*) g_slist_nth_data(conn_images, i);
				item = create_item(conn_img, 0.0, 0.0, moveable_group,
					pfe);
			}
		}
		item = create_item (img, 0.0, 0.0, moveable_group, pfe);
		gtk_signal_connect(GTK_OBJECT(moveable_group), "event",
				   (GtkSignalFunc)  mouse_event, pfe);

		/* move to the right location */
		gnome_canvas_item_move(GNOME_CANVAS_ITEM(moveable_group), x, y);
	} 
#ifdef DEBUG
	else {
		g_print("Atom image not found!\n");
	}
#endif    
	g_slist_free(conn_images);

	return GNOME_CANVAS_ITEM(moveable_group);
}

GnomeCanvasItem*
create_item (GdkImlibImage* img, double x, double y,
	     GnomeCanvasGroup* group, PlayFieldEditor *pfe)
{
	GnomeCanvasItem *item;
	
	item = gnome_canvas_item_new(group,
				     gnome_canvas_image_get_type(),
				     "image", img,
				     "x", x,
				     "y", y,
				     "width", (double)img->rgb_width,
				     "height", (double)img->rgb_height,
				     "anchor", GTK_ANCHOR_NW,
				     NULL);                              
	return item;
}

/*=================================================================
 
  Mouse handling functions

  ---------------------------------------------------------------*/

gint mouse_event(GnomeCanvasItem *item, GdkEvent *event, PlayFieldEditor *pfe)
{
	gdouble x1, y1, x2, y2;	
	GnomeCanvasItem *map_item = NULL;
	gint row;
	gint col;
	
	g_return_val_if_fail(pfe!=NULL, FALSE);

	if(pfe->mode != PFE_NORMAL)
	{
		return FALSE;
	}
	
	/* get the clicked row/column */
	gnome_canvas_item_get_bounds(item, &x1, &y1, &x2, &y2);       
	convert_to_playfield(pfe->theme, x1, y1, &row, &col);
	
	if((event->button.state & GDK_SHIFT_MASK) == GDK_SHIFT_MASK)
	{
		shift_mouse_event(event, pfe, 
				  pfe->last_clicked_row,
				  pfe->last_clicked_col,
				  row, col);
	}
	else
	{
		gboolean is_grid_item = FALSE;

		/* check whether it is a grid item */
		map_item = canvas_map_get_item(pfe->map, row, col);
		is_grid_item = (map_item == NULL);
		
		if(is_grid_item)
		{
			handle_grid_item_event(item, event, pfe);
		}
		else
		{
			handle_item_event(map_item, event, pfe);
		}
	}

	if (event->button.button == 1) 
	{
		/* remember this position */
		pfe->last_clicked_row = row;
		pfe->last_clicked_col = col;
	}	

	return TRUE;
}

void shift_mouse_event(GdkEvent *event, PlayFieldEditor *pfe, 
		       gint start_row, gint start_col, gint end_row, gint end_col)
{
	if(((start_row == end_row) || (start_col == end_col)) &&
	   (event->button.button == 1) && 
	   (image_win_ctrl_get_mouse_function() != IMG_WIN_CTRL_MOVE))
	{
		gboolean work_on_row;
		gint curr_col;
		gint curr_row;
		gint start;
		gint end; 
	       
		/* check if we need to change the row variable or the column and
		   if we need to increase it or decrease */
		work_on_row = (start_col == end_col);
		if(work_on_row)
		{
			start = start_row;
			end = end_row;
		}
		else
		{
			start = start_col;
			end = end_col;
		}
		if(start > end) 
		{
			gint tmp = start;
			start = end;
			end = tmp;
			curr_row = end_row;
			curr_col = end_col;
		}
		else
		{
			curr_row = start_row;
			curr_col = start_col;			
		}
		

		/* select the previous item (depending on the above results),
		   so we can select the next item first in the while loop */
		start--;
		if(work_on_row) curr_row--; else curr_col--;
		
		/* finally, call for every item the appropriate handler */
		while(start != end)
		{
			gboolean is_grid_item;
			GnomeCanvasItem *map_item;

			start++;
			if(work_on_row) curr_row++; else curr_col++;

			map_item = canvas_map_get_item(pfe->map, curr_row, curr_col);
			is_grid_item = (map_item == NULL);
			
			if(is_grid_item)		
			{
				GnomeCanvasItem *grid_item = canvas_map_get_item(pfe->grid_map, 
										 curr_row, curr_col);
				handle_grid_item_event(grid_item, event, pfe);
			}
			else
			{
				handle_item_event(map_item, event, pfe);
			}
		}
	}		
}

void handle_item_event (GnomeCanvasItem *item, GdkEvent *event, PlayFieldEditor *pfe)
{
	switch (event->type) {
	case GDK_BUTTON_PRESS:		
		if (event->button.button == 1) 
		{		    
			MouseFunction func; 
			CallbackData *cb_data;
			
			func = image_win_ctrl_get_mouse_function();
			switch(func)
			{
			case IMG_WIN_CTRL_SET:
				replace_tile(item, pfe, NULL);
				break;
				
			case IMG_WIN_CTRL_CLEAR:
				cb_data = g_malloc(sizeof(CallbackData));
				cb_data->pfe = pfe;
				cb_data->item = item;
				on_context_clear_activate(NULL, (gpointer)cb_data);
				break;
			case IMG_WIN_CTRL_MOVE:
				initiate_move_item(item, pfe);
				break;
				
			default:
			}
		}
		else if (event->button.button == 3) 
		{
			item_context_menu(pfe, item, event);
		}
		break;
	default:
	}
}

void handle_grid_item_event (GnomeCanvasItem *item, GdkEvent *event, PlayFieldEditor *pfe)
{

	switch (event->type) {
	case GDK_BUTTON_PRESS:		
	    if (event->button.button == 1) 
	    {
		    MouseFunction func; 

		    func = image_win_ctrl_get_mouse_function();
		    switch(func)
		    {
		    case IMG_WIN_CTRL_SET:
			    set_tile(item, pfe, NULL);
			    break;

		    case IMG_WIN_CTRL_CLEAR:
			    set_appbar_temporary(_("No tile to clear here."));
			    break;
			    
		    case IMG_WIN_CTRL_MOVE:
			    set_appbar_temporary(_("No tile to move here."));
			    break;
			    
		    default:
		    }
	    }
	    else if (event->button.button == 3) 
	    {
		    grid_context_menu(pfe, item, event);
	    }
	    break;

	default:
	}
}

void item_context_menu(PlayFieldEditor *pfe, GnomeCanvasItem *item, GdkEvent *event)
{
	/* right mouse click on an item, so show context menu */
	CallbackData *data = g_malloc(sizeof(CallbackData));
	GtkWidget *atom_context;
	GnomeUIInfo atom_context_uiinfo[] =
	{
		GNOMEUIINFO_MENU_COPY_ITEM (on_context_copy_activate, (gpointer)data),
		GNOMEUIINFO_MENU_CUT_ITEM (on_context_cut_activate, (gpointer)data),
		GNOMEUIINFO_MENU_CLEAR_ITEM (on_context_clear_activate, (gpointer)data),
		GNOMEUIINFO_SEPARATOR,
		{GNOME_APP_UI_ITEM, N_("Copy Goal"),
		 N_("Copy the goal into the playfield."),
		 on_copy_goal_activate, NULL, 
		 NULL, 0, 0, 0, GDK_CONTROL_MASK, 0 },
		{GNOME_APP_UI_ITEM, N_("Clear Grid"),
		 N_("Clears the whole grid."),
		 on_level_clear_activate, NULL, 
		 NULL, 0, 0, 0, GDK_CONTROL_MASK, 0 },
		GNOMEUIINFO_END
	};
	
	data->pfe = pfe;
	data->item = item;
	
	atom_context = gtk_menu_new ();
	gnome_app_fill_menu (GTK_MENU_SHELL (atom_context), atom_context_uiinfo,
			     NULL, FALSE, 0);

	if(!pfe->allow_non_moveables /* indication for goal editor */)
	{
		GtkWidget *widget = atom_context_uiinfo[4].widget;
		gtk_widget_set_sensitive(GTK_WIDGET(widget), FALSE);
	}

	gtk_menu_popup(GTK_MENU(atom_context), NULL, NULL, NULL, NULL, 3,
		       event->button.time);
}

void grid_context_menu(PlayFieldEditor *pfe, GnomeCanvasItem *item, GdkEvent *event)
{
	/* right mouse click on an item, so show context menu */
	GtkWidget *grid_context;
	GnomeUIInfo grid_context_uiinfo[] =
	{
		{GNOME_APP_UI_ITEM, N_("Copy Goal"),
		 N_("Copy the goal into the playfield."),
		 on_copy_goal_activate, NULL, 
		 NULL, 0, 0, 0, GDK_CONTROL_MASK, 0 },
		{GNOME_APP_UI_ITEM, N_("Clear Grid"),
		 N_("Clears the whole grid."),
		 on_level_clear_activate, NULL, 
		 NULL, 0, 0, 0, GDK_CONTROL_MASK, 0 },
		GNOMEUIINFO_END
	};
	
	grid_context = gtk_menu_new ();
	gnome_app_fill_menu (GTK_MENU_SHELL (grid_context), grid_context_uiinfo,
			     NULL, FALSE, 0);

	if(!pfe->allow_non_moveables /* indication for goal editor */)
	{
		GtkWidget *widget = grid_context_uiinfo[0].widget;
		gtk_widget_set_sensitive(GTK_WIDGET(widget), FALSE);
	}

	gtk_menu_popup(GTK_MENU(grid_context), NULL, NULL, NULL, NULL, 3,
		       event->button.time);
}


gint
item_move_event(GnomeCanvasItem *item, GdkEvent *event, PlayFieldEditor *pfe)
{
	GnomeCanvas *canvas;
	MoveItemData *data = (MoveItemData*)pfe->mode_data;

	canvas = GNOME_CANVAS(lookup_widget(get_app(), pfe->canvas_name));

	if(pfe->mode == PFE_NORMAL) return FALSE;

       	if(event->type == GDK_MOTION_NOTIFY)
	{
		gdouble item_x, item_y, mouse_x, mouse_y;
		gdouble diff_x, diff_y;
		gdouble x1, y1, x2, y2;

		gnome_canvas_item_get_bounds(GNOME_CANVAS_ITEM(data->item), 
					     &item_x, &item_y, &mouse_x, &mouse_y);
		gnome_canvas_window_to_world(canvas, event->motion.x, event->motion.y,
					     &mouse_x, &mouse_y);
		
		gnome_canvas_item_w2i (GNOME_CANVAS_ITEM(data->item)->parent, 
				       &mouse_x, &mouse_y);
		gnome_canvas_item_get_bounds(GNOME_CANVAS_ITEM(data->item), &x1, &y1, &x2, &y2);
		diff_x = mouse_x - item_x - (x2 - x1)/2;
		diff_y =  mouse_y - item_y - (y2 - y1)/2;

		gnome_canvas_item_move(GNOME_CANVAS_ITEM(data->item), diff_x, diff_y);
	}
	else if(event->type == GDK_BUTTON_PRESS)
	{
		gtk_signal_disconnect_by_func(GTK_OBJECT(canvas),
					      (GtkSignalFunc) item_move_event, pfe);
		gtk_signal_emit_stop_by_name(GTK_OBJECT(canvas), "button-press-event");

		switch(pfe->mode)
		{
		case PFE_MOVE:
			if(event->button.button == 1 /* left mouse button pressed */)
			{
				insert_tile(pfe, event->motion.x, event->motion.y,
					    data);
			}
			else if(event->button.button == 3 /* right mouse button pressed */)
			{
				insert_tile(pfe, -10, -10, data);
			}
			break;

		case PFE_INSERT:
			if(event->button.button == 1 /* left mouse button pressed */)
			{
				insert_playfield(pfe, event->motion.x, event->motion.y,
						 data->pf_source, data->item);
				
			}
			break;
		default:
		}

		/* clean up */
		gtk_object_destroy(GTK_OBJECT(data->item));
		gdk_pointer_ungrab(GDK_CURRENT_TIME);
		if(pfe->mode == PFE_INSERT)
		{
			playfield_destroy(data->pf_source);
		}
		else if(pfe->mode == PFE_MOVE)
		{
			tile_destroy(data->tile);
		}
		pfe->mode = PFE_NORMAL;
	}
	return TRUE;
}

void
on_context_copy_activate(GtkMenuItem *menuitem, CallbackData *data)
{
	gdouble x1, y1, x2, y2;	
	Tile *tile;
	gint row, col;

	/* get position */
	gnome_canvas_item_get_bounds(data->item, &x1, &y1, &x2, &y2);       
	convert_to_playfield(data->pfe->theme, x1, y1, &row, &col);
	
	tile = playfield_get_tile(data->pfe->pf, row, col);
	image_win_ctrl_set_actual_tile(tile);

	g_free(data);
}

void 
on_context_cut_activate(GtkMenuItem *menuitem, CallbackData *data)
{
	/* copy callback data */
	CallbackData *cp_data = g_malloc(sizeof(CallbackData));
	cp_data->pfe = data->pfe;
	cp_data->item = data->item;

	on_context_copy_activate(menuitem, (gpointer)data);
	on_context_clear_activate(menuitem, (gpointer)cp_data);
}

void 
on_context_clear_activate(GtkMenuItem *menuitem, CallbackData *data)
{
	gdouble x1, y1, x2, y2;	
	Tile *tile;
	gint row, col;

	/* get position */
	gnome_canvas_item_get_bounds(data->item, &x1, &y1, &x2, &y2);       
	convert_to_playfield(data->pfe->theme, x1, y1, &row, &col);

	canvas_map_remove_item(data->pfe->map, row, col);
	tile = playfield_clear_tile(data->pfe->pf, row, col);
	tile_destroy(tile);
	gtk_object_destroy(GTK_OBJECT(data->item));

	g_free(data);

	level_ctrl_set_actual_level_modified();
}


/*=================================================================
 
  Functions for setting/removing tiles.

  ---------------------------------------------------------------*/
void replace_tile(GnomeCanvasItem *item,  PlayFieldEditor *pfe, Tile *tile)
{
	gdouble x1, y1, x2, y2;	
	gint row, col;
	Tile *tile_cp;
	GnomeCanvasItem *new_item = NULL;

	g_return_if_fail(pfe!=NULL);

	/* get position */
	gnome_canvas_item_get_bounds(item, &x1, &y1, &x2, &y2);       
	convert_to_playfield(pfe->theme, x1, y1, &row, &col);

	/* get tile and check whether its allowed to be set */
	if(tile == NULL)
	{
		tile = image_win_ctrl_get_actual_tile();
	}
	if((!pfe->allow_non_moveables) && (tile_get_type(tile) == TILE_OBSTACLE))
	{
		return;
	}

	/* put the current selected item into the playfield structure */
	tile_cp = tile_copy(tile);
	playfield_set_tile(pfe->pf, row, col, tile_cp);

	/* destroy old canvas item */
	canvas_map_remove_item(pfe->map, row, col);
	gtk_object_destroy(GTK_OBJECT(item));

	/* put it in the canvas */
	convert_to_canvas(pfe->theme, row, col, &x1, &y1);
	switch(tile_get_type(tile))
	{
	case TILE_MOVEABLE:		
		new_item = create_moveable(pfe, x1, y1, tile);
		break;
	case TILE_OBSTACLE:
		new_item = create_obstacle(pfe, x1, y1, tile);
		break;
	default:
	}	

	/* add it to the canvas map */
	if(new_item)
	{
		canvas_map_set_item(pfe->map, row, col, new_item);
	}

	/* mark level as modified */
	level_ctrl_set_actual_level_modified();
}

void set_tile(GnomeCanvasItem *grid_item,  PlayFieldEditor *pfe, Tile *tile)
{
	gdouble x1, y1, x2, y2;	
	gint row, col;
	Tile *tile_cp;
	GnomeCanvasItem *item = NULL;

	/* get position */
	gnome_canvas_item_get_bounds(grid_item, &x1, &y1, &x2, &y2);       
	convert_to_playfield(pfe->theme, x1, y1, &row, &col);

	/* get tile and check whether its allowed to be set */
	if(tile == NULL)
	{
		tile = image_win_ctrl_get_actual_tile();
	}
	if((!pfe->allow_non_moveables) && (tile_get_type(tile) == TILE_OBSTACLE))
	{
		return;
	}
	
	/* put the current assembled tile in the playfield structure */
	tile_cp = tile_copy(tile);
	playfield_set_tile(pfe->pf, row, col, tile_cp);

	/* put it in the canvas */
	convert_to_canvas(pfe->theme, row, col, &x1, &y1);
	switch(tile_get_type(tile))
	{
	case TILE_MOVEABLE:		
		item = create_moveable(pfe, x1, y1, tile);
		break;
	case TILE_OBSTACLE:
		item = create_obstacle(pfe, x1, y1, tile);
		break;
	default:
	}
	if(item != NULL)
	{
		canvas_map_set_item(pfe->map, row, col, item);
	}

	/* mark level as modified */
	level_ctrl_set_actual_level_modified();
}

/*=================================================================
 
  Internal helper functions

  ---------------------------------------------------------------*/
void playfield_editor_render(PlayFieldEditor *pfe)
{
	PlayField *pf = pfe->pf;
	gint row, col;
	
	g_return_if_fail(pfe != NULL);
	g_return_if_fail(pfe->pf != NULL);
	

	for(row = 0; row < pf->n_rows; row++)
	{
		for(col = 0; col < pf->n_cols; col++)
		{
			GnomeCanvasItem *item = NULL;
			Tile *tile = playfield_get_tile(pf, row, col);
			TileType type = tile_get_type(tile);
			gdouble x,y;

			switch(type)
			{
			case TILE_MOVEABLE:
				convert_to_canvas(pfe->theme, 
						  row, col, &x, &y);
				item = create_moveable(pfe, x, y, tile);
				canvas_map_set_item(pfe->map, row, col, item);
				break;
					
			case TILE_OBSTACLE:
				if(pfe->allow_non_moveables)
				{
					convert_to_canvas(pfe->theme, 
							  row, 
							  col, 
							  &x, &y);
					item = create_obstacle(pfe, 
							       x, y, 
							       tile);
					canvas_map_set_item(pfe->map, row, col, item);
				}
				break;
					
			case TILE_NONE:
			default:
			}				

		}
	}		
}

void playfield_editor_clear_grid(PlayFieldEditor *pfe)
{
	if(pfe->grid_map)
	{
		canvas_map_destroy(pfe->grid_map);
	}
	if(pfe->grid)
	{
		gtk_object_destroy(GTK_OBJECT(pfe->grid));
	}
	
	pfe->grid = create_group(pfe->canvas_name, NULL);
	gnome_canvas_item_lower_to_bottom(GNOME_CANVAS_ITEM(pfe->grid));
	if(!pfe->show_grid)
	{
		gnome_canvas_item_hide(GNOME_CANVAS_ITEM(pfe->grid));
	}
}

void playfield_editor_create_grid(PlayFieldEditor *pfe)
{
	gint tile_width, tile_height;
	gint row, col;
	gint max_row, max_col;
	GdkColor *bg_color;

	/* initialize grid */
	pfe->grid_map = canvas_map_new();

	theme_get_tile_size(pfe->theme, &tile_width, &tile_height);
	bg_color = theme_get_background_color(pfe->theme);
	if(!gdk_color_alloc(gdk_colormap_get_system(), bg_color)) return;
	max_row = pfe->pf->n_rows;
	max_col = pfe->pf->n_cols;
	for(row = 0; row < max_row; row++)
	{
		for(col = 0; col < max_col; col++)
		{

			GnomeCanvasItem *item;
			gdouble x1, y1, x2, y2;

			x1 = col * tile_width;
			y1 = row * tile_height;
			x2 = x1 + tile_width;
			y2 = y1 + tile_height;

			item = gnome_canvas_item_new(pfe->grid,
						     gnome_canvas_rect_get_type(),
						     "x1", x1,
						     "y1", y1,
						     "x2", x2,
						     "y2", y2,
						     "fill_color_gdk", bg_color,
						     "outline_color", "white",
						     "width_pixels", 1,
						     NULL);
			canvas_map_set_item(pfe->grid_map, row, col, item);

			gtk_signal_connect(GTK_OBJECT(item), "event",
					   (GtkSignalFunc) mouse_event, pfe);
		}
	}
}

void initiate_move_item(GnomeCanvasItem *item, PlayFieldEditor *pfe)
{
	GnomeCanvas *canvas;
	MoveItemData *move_data;
	gdouble x1, y1, x2, y2;
	gint row, col;

	canvas = GNOME_CANVAS(lookup_widget(get_app(), pfe->canvas_name));
	gnome_canvas_item_get_bounds(item, &x1, &y1, &x2, &y2);
	convert_to_playfield(pfe->theme, x1, y1, &row, &col);	
	
	/* create data structure */
	move_data = g_malloc(sizeof(MoveItemData));
	move_data->item = item;
	move_data->tile = playfield_clear_tile(pfe->pf, row, col);
	move_data->src_row = row;
	move_data->src_col = col;
       
	pfe->mode_data = move_data;
	pfe->mode = PFE_MOVE;
	canvas_map_remove_item(pfe->map, row, col);

	/* connect mouse signals */
	gtk_signal_connect(GTK_OBJECT(canvas), "motion-notify-event",
			   (GtkSignalFunc) item_move_event, pfe);
	gtk_signal_connect(GTK_OBJECT(canvas), "button-press-event",
			   (GtkSignalFunc) item_move_event, pfe);

	gnome_canvas_item_reparent(GNOME_CANVAS_ITEM(move_data->item),
				   gnome_canvas_root(GNOME_CANVAS(canvas)));
	gnome_canvas_item_raise_to_top(GNOME_CANVAS_ITEM(move_data->item));
				   
	gdk_pointer_grab(GTK_WIDGET(canvas)->window, 
			 TRUE,
			 GDK_POINTER_MOTION_MASK | GDK_BUTTON_PRESS_MASK,
			 GTK_WIDGET(canvas)->window,
			 NULL,
			 GDK_CURRENT_TIME);			   
}

void insert_playfield(PlayFieldEditor *pfe, gdouble mouse_x, gdouble mouse_y,
		      PlayField *pf, GnomeCanvasItem *item)
{
	GnomeCanvas *canvas;
	gdouble world_x, world_y;
	gint pf_row, pf_col;
	gint row, col;
	gdouble x1, y1, x2, y2, mouse_x_offset, mouse_y_offset;
	gint row_offset, col_offset;

	canvas = GNOME_CANVAS(lookup_widget(get_app(), pfe->canvas_name));

	/* get clicked row/column */
	gnome_canvas_window_to_world(canvas, mouse_x, mouse_y,
				     &world_x, &world_y);
	convert_to_playfield(pfe->theme, world_x, world_y, &row, &col);
			
	if((row < 0) || (col < 0) || 
	   (col >= pfe->pf->n_cols) ||
	   (row >= pfe->pf->n_rows))
	{
		return;
	}
	
	/* get row/column offset */
	gnome_canvas_item_get_bounds(GNOME_CANVAS_ITEM(item), &x1, &y1, &x2, &y2);
	mouse_x_offset = (x2 - x1)/2;
	mouse_y_offset = (y2 - y1)/2; 
	convert_to_playfield(pfe->theme, mouse_x_offset, mouse_y_offset,
			     &row_offset, &col_offset);
	row -= row_offset;
	col -= col_offset;

	for(pf_row = 0; pf_row < pf->n_rows; pf_row++)
	{
		for(pf_col = 0; pf_col < pf->n_cols; pf_col++)
		{			
			Tile *tile;
			
			tile = playfield_get_tile(pf, pf_row, pf_col);
			if(tile_get_unique_id(tile) != 0)
			{
				GnomeCanvasItem *item; 
				item = canvas_map_get_item(pfe->map, row+pf_row, col+pf_col);
				if(item == NULL)
				{
					item = canvas_map_get_item(pfe->grid_map, 
								   row+pf_row, col+pf_col);
					if(item != NULL)
					{
						set_tile(item, pfe, tile);
					}
				}
				else
				{
					replace_tile(item, pfe, tile);
				}
			}
		}
	}
}

void insert_tile(PlayFieldEditor *pfe, gdouble mouse_x, gdouble mouse_y,
		 MoveItemData *data)
{
	GnomeCanvas *canvas;
	gdouble world_x, world_y;
	gint row, col;
	GnomeCanvasItem *pf_item; 

	canvas = GNOME_CANVAS(lookup_widget(get_app(), pfe->canvas_name));

	/* get clicked row/column */
	gnome_canvas_window_to_world(canvas, mouse_x, mouse_y,
				     &world_x, &world_y);
	convert_to_playfield(pfe->theme, world_x, world_y, &row, &col);
			
	if((row < 0) || (col < 0) || 
	   (col >= pfe->pf->n_cols) ||
	   (row >= pfe->pf->n_rows))
	{
		row = data->src_row;
		col = data->src_col;
	}

	/* insert tile */
	pf_item = canvas_map_get_item(pfe->map, row, col);
	if(pf_item == NULL)
	{
		pf_item = canvas_map_get_item(pfe->grid_map, 
					      row, col);
		if(pf_item != NULL)
		{
			set_tile(pf_item, pfe, data->tile);
		}
	}
	else
	{
		replace_tile(pf_item, pfe, data->tile);
	}	
}
