/* Atomixed -- the level-/themeeditor for atomix.
 * Copyright (C) 1999-2000 Jens Finke
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */
#include "image_win_ctrl.h"
#include "canvas_helper.h"
#include "support.h"
#include "tile.h"

/*=================================================================
 
  Global variables
  
  ---------------------------------------------------------------*/
static GtkWidget *window;
static Theme *actual_theme;
static Tile *actual_tile;

typedef struct {
	GnomeCanvasGroup *group[3];
	GHashTable *item[3];
	
	GnomeCanvasItem *selected_moveable;
	GnomeCanvasItem *selected_obstacle;	
} PreviewData;

static PreviewData *preview_data;
static MouseFunction function;

typedef	struct {
	GnomeIconList *gil;
	ThemeImageKind kind;
} AddImageData;


/*=================================================================
 
  Declaration of internal functions

  ---------------------------------------------------------------*/

void image_win_ctrl_clear_preview_data(void);

void image_win_ctrl_clear_preview_item_type(ThemeImageKind kind);

void image_win_ctrl_add_preview_item(GdkImlibImage *image, gint id, ThemeImageKind kind);

void image_win_ctrl_render_preview(Tile *tile);

void image_win_ctrl_add_image_to_clist(gpointer key, gpointer value, gpointer user_data);

void update_image_section(Theme *theme, const gchar *gil_name, ThemeImageKind kind);

void hide_canvas_item(gpointer key, gpointer value, gpointer user_data);
/*=================================================================
 
  Image Window Ctrl initialisation and clean up

  ---------------------------------------------------------------*/

void image_win_ctrl_init(GtkWidget *win)
{
	const gint row_spacing = 20;
	const gint col_spacing = 20;
	const gint text_spacing = 5;
	GnomeIconList *gil_obstacle;
	GnomeIconList *gil_moveable;
	GnomeIconList *gil_connection;

	window = win;
	function = IMG_WIN_CTRL_SET;

	/* init obastcle section */
	gil_obstacle = GNOME_ICON_LIST(lookup_widget(window, "obstacle_list"));
	gnome_icon_list_set_row_spacing (gil_obstacle,row_spacing);
        gnome_icon_list_set_col_spacing (gil_obstacle, col_spacing);
        gnome_icon_list_set_text_spacing (gil_obstacle, text_spacing);

	/* init moveable section */
	gil_moveable = GNOME_ICON_LIST(lookup_widget(window, "moveable_list"));
	gnome_icon_list_set_row_spacing (gil_moveable, row_spacing);
        gnome_icon_list_set_col_spacing (gil_moveable, col_spacing);
        gnome_icon_list_set_text_spacing (gil_moveable, text_spacing);

	/* fill connection section */	
	gil_connection = GNOME_ICON_LIST(lookup_widget(window, "connection_list"));
	gnome_icon_list_set_row_spacing (gil_connection, row_spacing);
        gnome_icon_list_set_col_spacing (gil_connection, col_spacing);
        gnome_icon_list_set_text_spacing (gil_connection, text_spacing);
	gnome_icon_list_set_selection_mode(gil_connection, GTK_SELECTION_MULTIPLE);

	actual_tile = tile_new();
}

void image_win_ctrl_destroy()
{
	tile_destroy(actual_tile);
	if(preview_data)
	{
		g_hash_table_destroy(preview_data->item[THEME_IMAGE_MOVEABLE]);
		g_hash_table_destroy(preview_data->item[THEME_IMAGE_CONNECTION]);
		g_hash_table_destroy(preview_data->item[THEME_IMAGE_OBSTACLE]);
		g_free(preview_data);
	}
}

/*=================================================================
  
  Image Window Ctrl functions

  ---------------------------------------------------------------*/

void image_win_ctrl_set_mouse_function(MouseFunction func)
{
	GtkWidget *bt_set;
	GtkWidget *bt_move;
	GtkWidget *bt_clear;

	bt_set = GTK_WIDGET(lookup_widget(window, "set_button"));
	bt_move = GTK_WIDGET(lookup_widget(window, "move_button"));
	bt_clear = GTK_WIDGET(lookup_widget(window, "clear_button"));
	
	// toggle previous selected button
	switch(function)
	{
	case IMG_WIN_CTRL_SET:
		gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(bt_set), FALSE);
		break;

	case IMG_WIN_CTRL_MOVE:
		gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(bt_move), FALSE);
		break;

	case IMG_WIN_CTRL_CLEAR:
		gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(bt_clear), FALSE);
		break;
	}
	function = func;
}

MouseFunction image_win_ctrl_get_mouse_function(void)
{
	return function;
}

void image_win_ctrl_set_theme(Theme *theme)
{
	GnomeCanvas *preview_canvas;
	GtkWidget *notebook;
	GnomeIconList *gil;
	gint tile_width, tile_height;
	
	g_return_if_fail(theme != NULL);

	actual_theme = theme;
	image_win_ctrl_clear_preview_data();	

	/* fill obstacle section */
	update_image_section(theme, "obstacle_list", THEME_IMAGE_OBSTACLE);
       
	/* fill moveable section */
	update_image_section(theme, "moveable_list", THEME_IMAGE_MOVEABLE);

	/* fill connection section */	
	update_image_section(theme, "connection_list", THEME_IMAGE_CONNECTION);
	
	/* init preview canvas*/
	preview_canvas = GNOME_CANVAS(lookup_widget(window, "atom_preview"));
	set_background_color_ref(GTK_WIDGET(preview_canvas), 
				 theme_get_background_color(theme));	
	theme_get_tile_size(theme, &tile_width, &tile_height);
	gnome_canvas_set_scroll_region(preview_canvas, 0.0, 0.0, tile_width, tile_height);

	/* select first item */
	gil = GNOME_ICON_LIST(lookup_widget(window, "obstacle_list"));
	gnome_icon_list_select_icon(gil, 0);
	notebook = GTK_WIDGET(lookup_widget(window, "theme_img_wnd_notebook"));
	gtk_notebook_set_page(GTK_NOTEBOOK(notebook), 1);
}

void image_win_ctrl_update_theme(Theme *theme, ThemeImageKind hint)
{
	g_return_if_fail(theme != NULL);

	if(g_strcasecmp(theme->name, actual_theme->name) != 0) return;

	switch(hint)
	{
	case THEME_IMAGE_MOVEABLE:
		image_win_ctrl_clear_preview_item_type(THEME_IMAGE_MOVEABLE);
		update_image_section(theme, "moveable_list", THEME_IMAGE_MOVEABLE);
		break;

	case THEME_IMAGE_CONNECTION:
		image_win_ctrl_clear_preview_item_type(THEME_IMAGE_CONNECTION);
		update_image_section(theme, "connection_list", THEME_IMAGE_CONNECTION);		
		break;

	case THEME_IMAGE_OBSTACLE:
		image_win_ctrl_clear_preview_item_type(THEME_IMAGE_OBSTACLE);
		update_image_section(theme, "obstacle_list", THEME_IMAGE_OBSTACLE);
		break;

	default:
		image_win_ctrl_clear_preview_data();
		update_image_section(theme, "obstacle_list", THEME_IMAGE_OBSTACLE);
		update_image_section(theme, "moveable_list", THEME_IMAGE_MOVEABLE);
		update_image_section(theme, "connection_list", THEME_IMAGE_CONNECTION);		
		break;
	};	

	actual_theme = theme;
}

void image_win_ctrl_select_obstacle(gint icon)
{
	GnomeIconList *gil_obstacle;
	gint image_id;

	gil_obstacle = GNOME_ICON_LIST(lookup_widget(window, "obstacle_list"));	

	if(tile_get_type(actual_tile) != TILE_OBSTACLE)
 	{
		GnomeIconList *gil_moveable;
		GnomeIconList *gil_connection;
		gil_moveable = GNOME_ICON_LIST(lookup_widget(window, "moveable_list"));
		gil_connection = GNOME_ICON_LIST(lookup_widget(window, "connection_list"));

		gnome_icon_list_unselect_all(gil_moveable, NULL, NULL);
		gnome_icon_list_unselect_all(gil_connection, NULL, NULL);

		tile_set_type(actual_tile, TILE_OBSTACLE);
	}

	image_id = GPOINTER_TO_INT(gnome_icon_list_get_icon_data(gil_obstacle,icon));
	tile_set_image_id(actual_tile, image_id);	
	image_win_ctrl_render_preview(actual_tile);
}


void image_win_ctrl_select_moveable(gint icon)
{
	GnomeIconList *gil_moveable;
	gint image_id;

	gil_moveable = GNOME_ICON_LIST(lookup_widget(window, "moveable_list"));	
	g_return_if_fail(gil_moveable != NULL);

	if(tile_get_type(actual_tile) != TILE_MOVEABLE)
	{
		GnomeIconList *gil_obstacle;
		gil_obstacle = GNOME_ICON_LIST(lookup_widget(window, "obstacle_list"));
		gnome_icon_list_unselect_all(gil_obstacle, NULL, NULL);

		tile_set_type(actual_tile, TILE_MOVEABLE);
	}

	image_id = GPOINTER_TO_INT(gnome_icon_list_get_icon_data(gil_moveable,icon));
	tile_set_image_id(actual_tile, image_id);	
	image_win_ctrl_render_preview(actual_tile);	
}

void image_win_ctrl_select_connection(gint icon)
{
	GnomeIconList *gil_connection;
	gint image_id;

	gil_connection = GNOME_ICON_LIST(lookup_widget(window, "connection_list"));	
	g_return_if_fail(gil_connection!= NULL);

	/* add image id to tile */
	image_id = GPOINTER_TO_INT(gnome_icon_list_get_icon_data(gil_connection,
								 icon));
	tile_add_connection_id(actual_tile, image_id);
	image_win_ctrl_render_preview(actual_tile);
}

void image_win_ctrl_unselect_connection(gint icon)
{
	GnomeIconList *gil_connection;
	gint image_id;
	
	gil_connection = GNOME_ICON_LIST(lookup_widget(window, "connection_list"));	
	g_return_if_fail(gil_connection!= NULL);

	/* remove image id from tile list*/
	image_id = GPOINTER_TO_INT(gnome_icon_list_get_icon_data(gil_connection,
								 icon));
	tile_remove_connection_id(actual_tile, image_id);
	image_win_ctrl_render_preview(actual_tile);
}

Tile* image_win_ctrl_get_actual_tile()
{
	return actual_tile;
}

void image_win_ctrl_set_actual_tile(Tile *tile)
{
	GtkWidget *list;
	GtkWidget *notebook;
	gint id;
	gint icon_index;

	if(tile_get_type(tile) == TILE_MOVEABLE)
	{
		GtkWidget *conn_list;
		GSList *conn_id;

		/* select moveable notebook page */
		notebook = GTK_WIDGET(lookup_widget(window, "theme_img_wnd_notebook"));
		gtk_notebook_set_page(GTK_NOTEBOOK(notebook), 0 /* the moveable page */);

		/* get list references and clear */
		list = GTK_WIDGET(lookup_widget(window, "moveable_list"));
		conn_list = GTK_WIDGET(lookup_widget(window, "connection_list"));
		gnome_icon_list_unselect_all(GNOME_ICON_LIST(list), NULL, NULL);
		gnome_icon_list_unselect_all(GNOME_ICON_LIST(conn_list), NULL, NULL);
		
		/* select appropriate image */
		id = tile_get_image_id(tile);
		icon_index = gnome_icon_list_find_icon_from_data(GNOME_ICON_LIST(list),
								 GINT_TO_POINTER(id));
		if(icon_index != -1)
		{
			gnome_icon_list_select_icon(GNOME_ICON_LIST(list), icon_index);
		}

		/* select connection images */
		conn_id = tile_get_connection_ids(tile);
		while(conn_id != NULL)
		{
			icon_index = gnome_icon_list_find_icon_from_data(GNOME_ICON_LIST(conn_list),
									 conn_id->data);
			if(icon_index != -1)
			{
				gnome_icon_list_select_icon(GNOME_ICON_LIST(conn_list), 
							    icon_index);
			}
			conn_id = conn_id->next;
		}
	}
	else if(tile_get_type(tile) == TILE_OBSTACLE)
	{
		
		/* select obstacle notebook page */
		notebook = GTK_WIDGET(lookup_widget(window, "theme_img_wnd_notebook"));
		gtk_notebook_set_page(GTK_NOTEBOOK(notebook), 1 /* the obstacle page */);

		/* clear all selected images */
		list = GTK_WIDGET(lookup_widget(window, "obstacle_list"));
		gnome_icon_list_unselect_all(GNOME_ICON_LIST(list), NULL, NULL);

		/* select appropriate image */
		id = tile_get_image_id(tile);
		icon_index = gnome_icon_list_find_icon_from_data(GNOME_ICON_LIST(list),
								 GINT_TO_POINTER(id));
		if(icon_index != -1)
		{
			gnome_icon_list_select_icon(GNOME_ICON_LIST(list), icon_index);
		}
	}

	actual_tile = tile_copy(tile);
}


/*=================================================================
 
  Internal helper functions

  ---------------------------------------------------------------*/

void 	
update_image_section(Theme *theme, const gchar *gil_name, ThemeImageKind kind)
{
	GnomeIconList *gil;
	AddImageData *add_image_data;
	gint tile_width, tile_height;
	GtkStyle *style;
	GdkColor *color;

	/* initialise data */
	theme_get_tile_size(theme, &tile_width, &tile_height);
	add_image_data = g_malloc(sizeof(AddImageData));
	gil = GNOME_ICON_LIST(lookup_widget(window, gil_name));

	/* fill gnome icon list */
	gnome_icon_list_freeze(gil);
	gnome_icon_list_set_icon_width(gil, tile_width); 
	gnome_icon_list_clear(gil);
	gnome_icon_list_unselect_all(gil, NULL, NULL);
	add_image_data->gil = gil;
	add_image_data->kind = kind;
	g_hash_table_foreach(theme->image_list[kind], 
			     (GHFunc) image_win_ctrl_add_image_to_clist, 
			     (gpointer) add_image_data);

	g_free(add_image_data);

	/* FIXME: set background color */
/*	style = gtk_style_copy(gtk_widget_get_style(GTK_WIDGET(gil)));
	color = theme_get_background_color(theme);
	g_print("setting color: red %i  green %i  blue %i\n", color->red, color->green, color->blue);
	gdk_color_alloc(gdk_colormap_get_system(), color);
	style->base[GTK_STATE_NORMAL] = (*color);
	style->base[GTK_STATE_ACTIVE] = (*color);
	style->base[GTK_STATE_PRELIGHT] = (*color);
	style->base[GTK_STATE_SELECTED] = (*color);
	style->base[GTK_STATE_INSENSITIVE] = (*color);
	gtk_widget_set_style(GTK_WIDGET(gil), style);
*/
	gnome_icon_list_thaw(gil);
}

void
image_win_ctrl_add_image_to_clist(gpointer key, gpointer value, gpointer user_data)
{
	ThemeElement *element = (ThemeElement*) value;
	AddImageData *data = (AddImageData*) user_data;
	GnomeIconList *gil = data->gil;
	gint id = GPOINTER_TO_INT(key);
	gint pos; 
	
	GdkImlibImage *image = theme_get_element_image(actual_theme, element);

 	pos = gnome_icon_list_append_imlib(gil, image, element->name);
	gnome_icon_list_set_icon_data(gil, pos, GINT_TO_POINTER(id));
	
	image_win_ctrl_add_preview_item(image, id, data->kind);
}

void image_win_ctrl_clear_preview_data()
{
	GnomeCanvas *canvas;
	gint i;

	canvas = GNOME_CANVAS(lookup_widget(window, "atom_preview"));

	if(preview_data != NULL)
	{
		for(i = 0; i < 3; i++)
		{
			if(preview_data->group[i])
			{
				gtk_object_destroy(GTK_OBJECT(preview_data->group[i]));
			}
			if(preview_data->item[i])
			{
				g_hash_table_destroy(preview_data->item[i]);
			}
		}
	}
	else
	{
		preview_data = g_malloc(sizeof(PreviewData));
	}
	
	for(i = 0; i < 3; i++)
	{
		preview_data->group[i] = create_group_ref(canvas, NULL);
		gnome_canvas_item_hide(GNOME_CANVAS_ITEM(preview_data->group[i]));
		preview_data->item[i] = g_hash_table_new((GHashFunc) g_direct_hash, 
							 (GCompareFunc) g_direct_equal);
	}

	gnome_canvas_item_lower_to_bottom(GNOME_CANVAS_ITEM(preview_data->group[THEME_IMAGE_CONNECTION]));
	gnome_canvas_item_raise_to_top(GNOME_CANVAS_ITEM(preview_data->group[THEME_IMAGE_OBSTACLE]));
       
       	preview_data->selected_moveable = NULL;
	preview_data->selected_obstacle = NULL;
}

void image_win_ctrl_clear_preview_item_type(ThemeImageKind kind)
{
	GnomeCanvas *canvas;

	g_return_if_fail(preview_data != NULL);

	canvas = GNOME_CANVAS(lookup_widget(window, "atom_preview"));

	if(preview_data->group[kind])
	{
		gtk_object_destroy(GTK_OBJECT(preview_data->group[kind]));
	}
	if(preview_data->item[kind])
	{
		g_hash_table_destroy(preview_data->item[kind]);
	}
		
	preview_data->group[kind] = create_group_ref(canvas,  NULL);
	gnome_canvas_item_hide(GNOME_CANVAS_ITEM(preview_data->group[kind]));
	preview_data->item[kind] = g_hash_table_new((GHashFunc) g_direct_hash, 
						    (GCompareFunc) g_direct_equal);

	if(kind == THEME_IMAGE_OBSTACLE)
	{
		preview_data->selected_obstacle = NULL;
	}
	if(kind == THEME_IMAGE_MOVEABLE)
	{
		preview_data->selected_moveable = NULL;
	}
}

void image_win_ctrl_add_preview_item(GdkImlibImage *image, gint id, ThemeImageKind kind)
{
	GnomeCanvasItem *item;

 	item = gnome_canvas_item_new(preview_data->group[kind],
				     gnome_canvas_image_get_type(),
				     "image", image,
				     "x", 0.0,
				     "y", 0.0,
				     "width", (double)image->rgb_width,
				     "height", (double)image->rgb_height,
				     "anchor", GTK_ANCHOR_NW,
				     NULL);                              
	gnome_canvas_item_hide(item);

	g_hash_table_insert(preview_data->item[kind],
			    GINT_TO_POINTER(id),
			    (gpointer)item);
}

void image_win_ctrl_render_preview(Tile *tile)
{
	if(preview_data)
	{
		GnomeCanvasItem *item;
		TileType type = tile_get_type(tile);

		if(type == TILE_MOVEABLE)
		{
			GSList *tile_conn_ids = NULL;

			/* hide all obstacles */
			gnome_canvas_item_hide(GNOME_CANVAS_ITEM(preview_data->group[THEME_IMAGE_OBSTACLE]));

			/* show atom image */
			if(preview_data->selected_moveable)
			{
				gnome_canvas_item_hide(preview_data->selected_moveable);
			}
			
			item = GNOME_CANVAS_ITEM(
				g_hash_table_lookup(preview_data->item[THEME_IMAGE_MOVEABLE],
						    GINT_TO_POINTER(tile_get_image_id(tile))));
			gnome_canvas_item_show(item);
			preview_data->selected_moveable = item;
			gnome_canvas_item_show(GNOME_CANVAS_ITEM(preview_data->group[THEME_IMAGE_MOVEABLE]));

			/* first hide all connections ... */
			g_hash_table_foreach(preview_data->item[THEME_IMAGE_CONNECTION],
					     (GHFunc) hide_canvas_item, NULL);
			
			/* ... then show the selected connection images */
			tile_conn_ids = tile_get_connection_ids(tile);
			while(tile_conn_ids != NULL)
			{
				gint id = GPOINTER_TO_INT(tile_conn_ids->data);
				item = GNOME_CANVAS_ITEM(
					g_hash_table_lookup(preview_data->item[THEME_IMAGE_CONNECTION],
							 GINT_TO_POINTER(id)));
				gnome_canvas_item_show(item);

				tile_conn_ids = tile_conn_ids->next;
			}
			gnome_canvas_item_show(GNOME_CANVAS_ITEM(preview_data->group[THEME_IMAGE_CONNECTION]));
		}
		else if(type == TILE_OBSTACLE)
		{
			/* hide all moveables and connections */
			gnome_canvas_item_hide(GNOME_CANVAS_ITEM(preview_data->group[THEME_IMAGE_MOVEABLE]));
			gnome_canvas_item_hide(GNOME_CANVAS_ITEM(preview_data->group[THEME_IMAGE_CONNECTION]));

			/* hide previous shown obstacle */
			if(preview_data->selected_obstacle)
			{
				gnome_canvas_item_hide(preview_data->selected_obstacle);
			}
			
			/* show actual selected obstacle image*/
			item = GNOME_CANVAS_ITEM(
				g_hash_table_lookup(preview_data->item[THEME_IMAGE_OBSTACLE], 
						    GINT_TO_POINTER(tile_get_image_id(tile))));
			gnome_canvas_item_show(item);
			preview_data->selected_obstacle = item;
			gnome_canvas_item_show(GNOME_CANVAS_ITEM(preview_data->group[THEME_IMAGE_OBSTACLE]));
		}
	}
}

void
hide_canvas_item(gpointer key, gpointer value, gpointer user_data)
{
	gnome_canvas_item_hide(GNOME_CANVAS_ITEM(value));
}
