#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <gnome.h>

#include "callbacks.h"
#include "interface.h"
#include "support.h"
#include "main.h"
#include "playfield_editor.h"
#include "level_ctrl.h"
#include "image_win_ctrl.h"
#include "level_properties.h"
#include "theme_properties.h"
#include "theme_ctrl.h"

extern Level *level;
extern Theme *theme;

typedef enum {
	EDIT_UNDEFINED,
	EDIT_LEVEL,
	EDIT_THEME
} GlobalEditMode;

typedef enum {	
	EDIT_LEVEL_UNDEFINED,
	EDIT_LEVEL_PLAYFIELD,
	EDIT_LEVEL_GOAL,
	EDIT_LEVEL_PROPERTIES
} LevelEditMode;

typedef enum {
	EDIT_THEME_UNDEFINED,
	EDIT_THEME_MOVEABLE,
	EDIT_THEME_OBSTACLE,
	EDIT_THEME_PROPERTIES
} ThemeEditMode;

static GlobalEditMode global_edit_mode = EDIT_LEVEL;
static LevelEditMode level_edit_mode = EDIT_LEVEL_PLAYFIELD;
static ThemeEditMode theme_edit_mode = EDIT_THEME_MOVEABLE;

static gchar *image_selector_path = NULL;

void on_fileselector_ok (GtkWidget *widget, GtkFileSelection *file_selector);
void on_imageselector_ok (GtkWidget *widget, GtkFileSelection *file_selector);
void on_new_theme(void);
void on_new_level(void);


void
on_mainframe_destroy_event             (GtkObject       *object,
                                        gpointer         user_data)
{
	GtkWidget *level_notebook;
	PlayFieldEditor *pfe;
	ThemeImgEditor *tie;

	/* avoid function call through destroy process */
	level_notebook = lookup_widget(get_app(), "level_notebook");
	gtk_signal_disconnect_by_func(GTK_OBJECT(level_notebook),
				      (GFunc) on_level_switch_page, NULL);

	/* clean up */
	pfe = get_level_editor();
	if(pfe)
	{
		playfield_editor_destroy(pfe);
	}
	pfe = get_goal_editor();
	if(pfe)
	{
		playfield_editor_destroy(pfe);
	}

	tie = get_obstacle_editor();
	if(tie)
	{
		theme_img_editor_destroy(tie);
	}
	tie = get_moveable_editor();
	if(tie)
	{
		theme_img_editor_destroy(tie);
	}
	tie = get_connection_editor();
	if(tie)
	{
		theme_img_editor_destroy(tie);
	}

	theme_image_win_destroy(); // gui window
	image_win_ctrl_destroy();  // data structure
	level_ctrl_destroy();
	theme_ctrl_destroy();
	
	gtk_main_quit();
}



void
on_exit_activate                      (GtkMenuItem     *menuitem,
                                        gpointer         user_data)
{
	GtkWidget *app;
	app = get_app();
	g_return_if_fail(app != NULL);

	/* save levels if neccessary */
	if(level_ctrl_check_for_modifications())
	{
		GtkWidget *dlg;
		gint button_nr;

		dlg = GTK_WIDGET(gnome_message_box_new("Some levels are unsaved. Save this ones?",
						       GNOME_MESSAGE_BOX_QUESTION,
						       GNOME_STOCK_BUTTON_YES,
						       GNOME_STOCK_BUTTON_NO,
						       NULL));
		button_nr = gnome_dialog_run(GNOME_DIALOG(dlg));
		if(button_nr == 0 /* ok */)
		{
			g_print("saving levels ...");
			level_ctrl_save_all_levels();
		}
	}

	/* save themes if neccessary */
	if(theme_ctrl_check_for_modifications())
	{
		GtkWidget *dlg;
		gint button_nr;

		dlg = GTK_WIDGET(gnome_message_box_new("Some themes are unsaved. Save this ones?",
						       GNOME_MESSAGE_BOX_QUESTION,
						       GNOME_STOCK_BUTTON_YES,
						       GNOME_STOCK_BUTTON_NO,
						       NULL));
		button_nr = gnome_dialog_run(GNOME_DIALOG(dlg));
		if(button_nr == 0 /* ok */)
		{
			g_print("saving themes ...");
			theme_ctrl_save_all_themes();
		}
	}

	gtk_widget_destroy(app);    	
}


void
on_level_clear_activate                (GtkWidget        *menuitem,
                                        gpointer         user_data)
{
	GtkWidget *dlg;
	gint button;
	
	dlg = GTK_WIDGET(gnome_message_box_new(_("Do you really want to clear the grid?"),
					       GNOME_MESSAGE_BOX_QUESTION,
					       GNOME_STOCK_BUTTON_YES, 
					       GNOME_STOCK_BUTTON_NO, NULL));
	button = gnome_dialog_run(GNOME_DIALOG(dlg));

	if(button == 0 /* yes */)
	{
		PlayFieldEditor *pfe = NULL;

		switch(level_edit_mode)
		{
		case EDIT_LEVEL_PLAYFIELD:
			pfe = get_level_editor();
			break;
		case EDIT_LEVEL_GOAL:
			pfe = get_goal_editor();
			break;
		default:
			break;
		}

		if(pfe != NULL)
		{
			playfield_editor_clear(pfe);
			playfield_clear(pfe->pf);
			level_ctrl_set_actual_level_modified();
		}
	}
}


void
on_preferences_activate               (GtkMenuItem     *menuitem,
                                        gpointer         user_data)
{

}


void
on_level_list_select_row               (GtkCList        *clist,
                                        gint             row,
                                        gint             column,
                                        GdkEvent        *event,
                                        gpointer         user_data)
{
	Level *level;
	Level *actual_level;

	level = level_ctrl_get_level(row);
	actual_level = level_ctrl_get_actual_level();

	if(level != actual_level)
	{
		PlayFieldEditor *level_editor;
		PlayFieldEditor *goal_editor;

		level_editor = get_level_editor();
		goal_editor = get_goal_editor();

		/* check if another theme is necessary */
		if(g_strcasecmp(level->theme_name, actual_level->theme_name) != 0)
		{
			Theme *theme = theme_ctrl_get_theme_by_name(level->theme_name);
			
			/* FIXME: handle theme == 0 */
			g_return_if_fail(theme != NULL);
			
			playfield_editor_set_theme(level_editor, theme, FALSE);
			playfield_editor_set_theme(goal_editor, theme, FALSE);
			image_win_ctrl_set_theme(theme);
		}

		/* update all views with new level informations */
		level_ctrl_select_row(row);
		playfield_editor_set_playfield(level_editor, level->playfield);
		playfield_editor_set_playfield(goal_editor, level->goal);
		level_properties_set_level(level);
	}
}


void
on_grid_activate                       (GtkMenuItem     *menuitem,
                                        gpointer         user_data)
{
	PlayFieldEditor *pfe;
	GtkWidget *button;
	GtkCheckMenuItem *item;
	gboolean state;

	item = GTK_CHECK_MENU_ITEM(menuitem);
	state = item->active;
	
	pfe = get_level_editor();	
	if(pfe != 0)
	{
		playfield_editor_show_grid(pfe, state);
		pfe =  get_goal_editor();	
		playfield_editor_show_grid(pfe, state);
		
		button = GTK_WIDGET(lookup_widget(get_app(), "grid_bt"));
		gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(button), state);
	}
}

void
on_grid_bt_toggled                     (GtkToggleButton *togglebutton,
                                        gpointer         user_data)
{
	GtkWidget *menu_item;
	PlayFieldEditor *pfe;
	gboolean state = gtk_toggle_button_get_active(togglebutton);
	
	pfe = get_level_editor();	
	playfield_editor_show_grid(pfe, state);
	pfe =  get_goal_editor();	
	playfield_editor_show_grid(pfe, state);

	menu_item = GTK_WIDGET(lookup_widget(get_app(), "item_grid"));
	gtk_check_menu_item_set_active(GTK_CHECK_MENU_ITEM(menu_item), state);
}


void
on_moveable_list_select_icon           (GnomeIconList   *gnomeiconlist,
                                        gint             icon,
                                        GdkEvent        *event,
                                        gpointer         user_data)
{
	image_win_ctrl_select_moveable(icon);
}


void
on_connection_list_select_icon         (GnomeIconList   *gnomeiconlist,
                                        gint             icon,
                                        GdkEvent        *event,
                                        gpointer         user_data)
{
	image_win_ctrl_select_connection(icon);
}


void
on_connection_list_unselect_icon       (GnomeIconList   *gnomeiconlist,
                                        gint             icon,
                                        GdkEvent        *event,
                                        gpointer         user_data)
{
	image_win_ctrl_unselect_connection(icon);	
}


void
on_obstacle_list_select_icon           (GnomeIconList   *gnomeiconlist,
                                        gint             icon,
                                        GdkEvent        *event,
                                        gpointer         user_data)
{
	image_win_ctrl_select_obstacle(icon);
}


void
on_new_file_activate                   (GtkWidget     *item,
                                        gpointer       user_data)
{
	if(global_edit_mode == EDIT_LEVEL)
	{
		on_new_level();
	}
	else if(global_edit_mode == EDIT_THEME)
	{
		on_new_theme();
	}
}

void on_new_level(void)
{		
	GtkWidget *new_level_win;
	GtkWidget *level_name;
	GtkWidget *level_file;
	GtkWidget *theme_combo;
	GtkWidget *level_row;
	GtkWidget *level_col;
	GtkWidget *goal_row;
	GtkWidget *goal_col;
		
	gchar *name = NULL;
	gchar *file = NULL;
	gchar *theme = NULL;
	guint pf_row = 10;
	guint pf_col = 10;
	guint gl_row = 10;
	guint gl_col = 10;
				
	gint button;
	gboolean accepted = FALSE;
	gboolean cancel = FALSE;
	GList *theme_name_list = NULL;

	new_level_win = create_new_level_win();
	name = g_strdup("noname");
		
	while(!accepted && !cancel)
	{
		/* init level name */
		level_name = lookup_widget(new_level_win, "level_name");
		gtk_entry_set_text(GTK_ENTRY(level_name), name);
		gtk_entry_select_region(GTK_ENTRY(level_name), 0, -1);
		gtk_widget_grab_focus(GTK_WIDGET(level_name));
		g_free(name);
						
		/* init file name */
		level_file = lookup_widget(new_level_win, "level_file");
		gtk_entry_set_text(GTK_ENTRY(level_file), "noname.atomix");
			
		/* init theme selector */
		theme_combo = lookup_widget(new_level_win, "theme_name_combo");
		theme_name_list = theme_ctrl_get_theme_name_list();
		if(theme_name_list != NULL)
		{
			gtk_combo_set_popdown_strings(GTK_COMBO(theme_combo),
						      theme_name_list);
			gtk_entry_set_text(GTK_ENTRY(GTK_COMBO(theme_combo)->entry), 
					   (gchar*)theme_name_list->data);
		}
			
		/* init level row/col */
		level_row = lookup_widget(new_level_win, "level_row");
		level_col = lookup_widget(new_level_win, "level_col");
		gtk_spin_button_set_value(GTK_SPIN_BUTTON(level_row), 10);
		gtk_spin_button_set_value(GTK_SPIN_BUTTON(level_col), 10);
			
		/* init goal row/col */
		goal_row = lookup_widget(new_level_win, "goal_row");
		goal_col = lookup_widget(new_level_win, "goal_col");
		gtk_spin_button_set_value(GTK_SPIN_BUTTON(goal_row), 10);
		gtk_spin_button_set_value(GTK_SPIN_BUTTON(goal_col), 10);
			
		button = gnome_dialog_run(GNOME_DIALOG(new_level_win));

		if(button == 0)
		{
				/* get values */
			name = gtk_editable_get_chars(GTK_EDITABLE(level_name),0,-1);
			file = gtk_editable_get_chars(GTK_EDITABLE(level_file),0,-1);
			theme = gtk_editable_get_chars(GTK_EDITABLE(GTK_COMBO(theme_combo)->entry),
						       0,-1);
			pf_row = gtk_spin_button_get_value_as_int(GTK_SPIN_BUTTON(level_row));
			pf_col = gtk_spin_button_get_value_as_int(GTK_SPIN_BUTTON(level_col));
			gl_row = gtk_spin_button_get_value_as_int(GTK_SPIN_BUTTON(goal_row));
			gl_col = gtk_spin_button_get_value_as_int(GTK_SPIN_BUTTON(goal_col));
				
		}
		else
		{
			cancel = TRUE;
		}

		if(level_ctrl_does_name_exists(name))
		{
			gtk_entry_select_region(GTK_ENTRY(level_name), 0, -1);
		}
		else
		{
			accepted = TRUE;
		}
	}		

	if(!cancel)
	{
		/* insert into level ctrl */
		level_ctrl_new_level(name, file, theme, pf_row, pf_col,
				     gl_row, gl_col);

		g_free(name);
		g_free(theme);
		g_free(file);
	}
		
	gtk_widget_destroy(GTK_WIDGET(new_level_win));
}

void on_new_theme()
{
	GtkWidget *theme_name_entry;
	GtkWidget *dir_name_entry;
	GtkWidget *new_theme_win;		

	gboolean accepted = FALSE;
	gboolean cancel = FALSE;
	gboolean name_exists = TRUE;
	gboolean dir_exists = TRUE;
	gint button;
	gchar *name = g_strdup("noname"); 
	gchar *dir_name = g_strdup("noname"); 

	new_theme_win = create_new_theme_win();

	theme_name_entry = lookup_widget(new_theme_win, "theme_name");
	dir_name_entry = lookup_widget(new_theme_win, "theme_dir_name");

	while(!accepted && !cancel)
	{
		/* init theme name entry */
		gtk_entry_set_text(GTK_ENTRY(theme_name_entry), name);
		if(name_exists)
		{
			gtk_entry_select_region(GTK_ENTRY(theme_name_entry), 0, -1);
			gtk_widget_grab_focus(GTK_WIDGET(theme_name_entry));
		}
		g_free(name);

		/* init dir name entry */
		gtk_entry_set_text(GTK_ENTRY(dir_name_entry), dir_name);
		if(dir_exists)
		{
			gtk_entry_select_region(GTK_ENTRY(dir_name_entry), 0, -1);
		}
		g_free(dir_name);
			
		button = gnome_dialog_run(GNOME_DIALOG(new_theme_win));

		if(button == 0)
		{
			GtkWidget *m_box;
			name = gtk_editable_get_chars(GTK_EDITABLE(theme_name_entry),0,-1);
			dir_name = gtk_editable_get_chars(GTK_EDITABLE(dir_name_entry),0,-1);

			theme_ctrl_does_theme_exist(name, dir_name, 
						    &name_exists, &dir_exists);
			if(name_exists && dir_exists)
			{
				m_box = gnome_message_box_new(_("There does already a theme with this name exist.\nSo does the dir name."), 
							      GNOME_MESSAGE_BOX_WARNING,
							      GNOME_STOCK_BUTTON_OK,
							      NULL);
				gnome_dialog_run(GNOME_DIALOG(m_box));
			}
			else if(name_exists)
			{
				m_box = gnome_message_box_new(_("There does already a theme with\nthis name exist."), 
							      GNOME_MESSAGE_BOX_WARNING,
							      GNOME_STOCK_BUTTON_OK,
							      NULL);
				gnome_dialog_run(GNOME_DIALOG(m_box));
			}
			else if(dir_exists)
			{
				m_box = gnome_message_box_new(_("There does already a directory\nwith this name exist."), 
							      GNOME_MESSAGE_BOX_WARNING,
							      GNOME_STOCK_BUTTON_OK,
							      NULL);
				gnome_dialog_run(GNOME_DIALOG(m_box));
				
			}
			else
			{
				accepted = TRUE;
			}
		}
		else
		{
			/* user pressed something else than ok */
			cancel = TRUE;
		}
	}
	
	if(!cancel)
	{
		/* insert into theme ctrl */
		theme_ctrl_new_theme(name, dir_name);

		g_free(name);
		g_free(dir_name);
	}
	
	gtk_widget_destroy(GTK_WIDGET(new_theme_win));
}



void
on_level_resize_activate                     (GtkMenuItem     *menuitem,
                                        gpointer         user_data)
{
	if(global_edit_mode == EDIT_LEVEL)
	{
		PlayFieldEditor *pfe = NULL;

		switch(level_edit_mode)
		{
		case EDIT_LEVEL_PLAYFIELD:
			pfe = get_level_editor();
			break;
		case EDIT_LEVEL_GOAL:
			pfe = get_goal_editor();
			break;
		default:
			break;
		}
		
		if(pfe)
		{
			GtkWidget *window;
			GtkWidget *old_row_entry;
			GtkWidget *old_col_entry;
			GtkWidget *new_row_entry;
			GtkWidget *new_col_entry;
			gint n_rows, n_cols;
			gchar* str_buffer;
			gint button;

			/* create window */
			window = create_resize_grid_win();

			/* get entry widgets */
			old_row_entry = lookup_widget(window, "old_row");
			old_col_entry = lookup_widget(window, "old_column");
			new_row_entry = lookup_widget(window, "new_row");
			new_col_entry = lookup_widget(window, "new_column");
			
			/* set values */
			n_rows = pfe->pf->n_rows;
			n_cols = pfe->pf->n_cols;
			str_buffer = g_malloc(5*sizeof(gchar));
			g_snprintf(str_buffer, 5, "%i", n_rows);
			gtk_entry_set_text(GTK_ENTRY(old_row_entry), str_buffer);
			g_snprintf(str_buffer, 5, "%i", n_cols);
			gtk_entry_set_text(GTK_ENTRY(old_col_entry), str_buffer);
			gtk_spin_button_set_value(GTK_SPIN_BUTTON(new_row_entry),
						 n_rows);
			gtk_spin_button_set_value(GTK_SPIN_BUTTON(new_col_entry),
						 n_cols);
			g_free(str_buffer);
			
			/* show dialog */
			button = gnome_dialog_run(GNOME_DIALOG(window));

			if(button == 0)
			{
				/* user clicked ok */
				n_rows = gtk_spin_button_get_value_as_int(GTK_SPIN_BUTTON(new_row_entry));
				n_cols = gtk_spin_button_get_value_as_int(GTK_SPIN_BUTTON(new_col_entry));
				playfield_editor_resize(pfe, n_rows, n_cols);
			}
			gtk_widget_destroy(window);
		}
	}
}

void
on_save_activate                       (GtkWidget     *item,
                                        gpointer         user_data)
{
	if(global_edit_mode == EDIT_LEVEL)
	{
		level_ctrl_save_all_levels();
	}
	else if(global_edit_mode == EDIT_THEME)
	{
		theme_ctrl_save_all_themes();
	}
}


void
on_level_delete_activate                     (GtkMenuItem     *menuitem,
                                        gpointer         user_data)
{
	if(global_edit_mode == EDIT_LEVEL)
	{
		GtkWidget *dlg;
		gint button;
		
		dlg = GTK_WIDGET(gnome_message_box_new("Do you really want to delete this level?",
						       GNOME_MESSAGE_BOX_QUESTION,
						       GNOME_STOCK_BUTTON_YES, 
						       GNOME_STOCK_BUTTON_NO, NULL));
		button = gnome_dialog_run(GNOME_DIALOG(dlg));
		if(button == 0 /* yes */)
		{
			level_properties_delete_level();
			level_ctrl_remove_actual_level();
		}
	}
}


void
on_level_up_clicked                    (GtkButton       *button,
                                        gpointer         user_data)
{
	level_ctrl_move_up();
}


void
on_level_down_clicked                  (GtkButton       *button,
                                        gpointer         user_data)
{
	level_ctrl_move_down();
}


void
on_level_switch_page                   (GtkNotebook     *notebook,
                                        GtkNotebookPage *page,
                                        gint             page_num,
                                        gpointer         user_data)
{
	switch(page_num)
	{
	case 0:
		level_edit_mode = EDIT_LEVEL_PLAYFIELD;
		break;
	case 1:
		level_edit_mode = EDIT_LEVEL_GOAL;
		break;
	case 2:
		level_edit_mode = EDIT_LEVEL_PROPERTIES;
		break;
	default:
		level_edit_mode = EDIT_LEVEL_UNDEFINED;
		break;
	}

	update_menu_sensibility();
}

void
on_about_activate                      (GtkMenuItem     *menuitem,
                                        gpointer         user_data)
{
	GtkWidget* dlg;
	dlg = create_about_dialog();
	gtk_widget_show(dlg);

}


gboolean
on_mainframe_delete_event              (GtkWidget       *widget,
                                        GdkEvent        *event,
                                        gpointer         user_data)
{
	on_exit_activate(0,0);
	
	return FALSE;
}


void
on_theme_window_item_activate          (GtkMenuItem     *menuitem,
                                        gpointer         user_data)
{
	GtkWidget *button;
	GtkCheckMenuItem *item = GTK_CHECK_MENU_ITEM(menuitem);
	gboolean state = item->active;
	
	if(state)
	{
		theme_image_win_show();
	}
	else
	{
		theme_image_win_hide();
	}

	button = GTK_WIDGET(lookup_widget(get_app(), "theme_wnd_bt"));
	gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(button), state);
	

}


void
on_theme_wnd_bt_toggled                (GtkToggleButton *togglebutton,
                                        gpointer         user_data)
{
	GtkWidget *menu_item;
	gboolean state = gtk_toggle_button_get_active(togglebutton);
	
	if(state)
	{
		theme_image_win_show();
	}
	else
	{
		theme_image_win_hide();
	}

	menu_item = GTK_WIDGET(lookup_widget(get_app(), "theme_window_item"));
	gtk_check_menu_item_set_active(GTK_CHECK_MENU_ITEM(menu_item), state);

}


gboolean
on_theme_image_win_delete_event        (GtkWidget       *widget,
                                        GdkEvent        *event,
                                        gpointer         user_data)
{
	GtkWidget *button;
	GtkWidget *menu_item;

	theme_image_win_hide();

	/* update button state */
	menu_item = GTK_WIDGET(lookup_widget(get_app(), "theme_window_item"));
	gtk_check_menu_item_set_active(GTK_CHECK_MENU_ITEM(menu_item), FALSE);

	button = GTK_WIDGET(lookup_widget(get_app(), "theme_wnd_bt"));
	gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(button), FALSE);
	
	return TRUE;
}

void on_fileselector_ok(GtkWidget *button, GtkFileSelection *file_selector) 
{
	gchar *selected_filename;

	selected_filename = gtk_file_selection_get_filename(file_selector);
	
	if(selected_filename)
	{
		level_ctrl_load_level(selected_filename);
	}
	
	gtk_widget_destroy(GTK_WIDGET(file_selector));
}

void
on_open_activate                       (GtkWidget     *item,
                                        gpointer         user_data)
{
	GtkFileSelection *file_selector;

	/* Create file selector */     
	file_selector = GTK_FILE_SELECTION(gtk_file_selection_new("Please select a file for editing."));
	gtk_window_set_modal(GTK_WINDOW(file_selector), TRUE);


	gtk_signal_connect (GTK_OBJECT (file_selector->ok_button), "clicked",
			    GTK_SIGNAL_FUNC (on_fileselector_ok), file_selector);
	gtk_signal_connect_object (GTK_OBJECT (file_selector->cancel_button), "clicked",
				   GTK_SIGNAL_FUNC (gtk_widget_destroy),
				   GTK_OBJECT(file_selector));

	gtk_window_set_position (GTK_WINDOW (file_selector), GTK_WIN_POS_MOUSE);

     
	/* add match pattern */
	gtk_file_selection_complete(GTK_FILE_SELECTION(file_selector),
				    "*.atomix");


	/* Display that dialog */	
	gtk_widget_show (GTK_WIDGET(file_selector));

}

void
on_img_win_ctrl_set_toggled            (GtkToggleButton *togglebutton,
                                        gpointer         user_data)
{
	if(gtk_toggle_button_get_active(togglebutton))
	{
		image_win_ctrl_set_mouse_function(IMG_WIN_CTRL_SET);
	}
}


void
on_img_win_ctrl_move_toggled           (GtkToggleButton *togglebutton,
                                        gpointer         user_data)
{
	if(gtk_toggle_button_get_active(togglebutton))
	{
		image_win_ctrl_set_mouse_function(IMG_WIN_CTRL_MOVE);
	}
}


void
on_img_win_ctrl_clear_toggled          (GtkToggleButton *togglebutton,
                                        gpointer         user_data)
{
	if(gtk_toggle_button_get_active(togglebutton))
	{
		image_win_ctrl_set_mouse_function(IMG_WIN_CTRL_CLEAR);
	}
}

void
on_copy_goal_activate(GtkMenuItem *item, gpointer user_data)
{
	PlayFieldEditor *pfe_goal;
	PlayFieldEditor *pfe_level; 
	PlayField *pf;
	
	g_return_if_fail(global_edit_mode == EDIT_LEVEL);
	g_return_if_fail(level_edit_mode == EDIT_LEVEL_PLAYFIELD);
	
	pfe_goal = get_goal_editor();
	pfe_level = get_level_editor();

	g_return_if_fail(pfe_goal != NULL);
	g_return_if_fail(pfe_level != NULL);       
	
	pf = playfield_strip(pfe_goal->pf);
	playfield_editor_insert_pf(pfe_level, pf);
}

void
on_theme_list_select_row               (GtkCList        *clist,
                                        gint             row,
                                        gint             column,
                                        GdkEvent        *event,
                                        gpointer         user_data)
{
	Theme *theme;
	
	theme = theme_ctrl_get_theme(row);
	
	if(theme != theme_ctrl_get_actual_theme())
	{
		ThemeImgEditor *tie;

		tie = get_moveable_editor();
		theme_img_editor_set_theme(tie, theme);

		tie = get_obstacle_editor();
		theme_img_editor_set_theme(tie, theme);

		tie = get_connection_editor();
		theme_img_editor_set_theme(tie, theme);
		
		theme_properties_set_theme(theme);

		theme_ctrl_select_row(row);
	}	
}

gboolean
on_image_list_text_changed             (GnomeIconList   *gnomeiconlist,
                                        gint             num,
                                        gchar*           text,
                                        gpointer         user_data)
{
	ThemeImgEditor *tie = get_image_editor_by_ref(gnomeiconlist);

	theme_img_editor_change_text(tie, num, text);

	return TRUE;
}


gboolean
on_image_list_button_release_event     (GtkWidget       *widget,
                                        GdkEventButton  *event,
                                        gpointer         user_data)
{

	if(event->button == 3)
	{
		ThemeImgEditor *tie = get_image_editor_by_ref(GNOME_ICON_LIST(widget));
	
		theme_img_editor_show_context_menu(tie);
	}

	return TRUE;
}


void
on_delete_theme_activate               (GtkMenuItem     *menuitem,
                                        gpointer         user_data)
{

}


void
on_level_theme_notebook_switch_page    (GtkNotebook     *notebook,
                                        GtkNotebookPage *page,
                                        gint             page_num,
                                        gpointer         user_data)
{
	switch(page_num)
	{
	case 0: 
		global_edit_mode = EDIT_LEVEL;
		break;
	case 1:
		theme_ctr_clear_update_flag();
		global_edit_mode = EDIT_THEME;
		break;
	default:
		global_edit_mode = EDIT_UNDEFINED;
		break;
	}	
	update_menu_sensibility();


	if(global_edit_mode == EDIT_LEVEL)
	{
		Theme *theme;
		Level *level = level_ctrl_get_actual_level();

		if(level == NULL) return;

		theme = theme_ctrl_get_theme_by_name(level->theme_name);
		if(theme->need_update)
		{
			PlayFieldEditor *pfe;
			pfe = get_level_editor();
			playfield_editor_set_theme(pfe, theme, TRUE);
			pfe = get_goal_editor();			
			playfield_editor_set_theme(pfe, theme, TRUE);
		}		
	}
}


void
on_theme_switch_page                   (GtkNotebook     *notebook,
                                        GtkNotebookPage *page,
                                        gint             page_num,
                                        gpointer         user_data)
{
	switch(page_num)
	{
	case 0: 
		theme_edit_mode = EDIT_THEME_MOVEABLE;
		break;
	case 1:
		theme_edit_mode = EDIT_THEME_OBSTACLE;
		break;
	case 2:
		theme_edit_mode = EDIT_THEME_PROPERTIES;
		break;
	default:
		theme_edit_mode = EDIT_THEME_UNDEFINED;
		break;
	}	
	update_menu_sensibility();
}

void
update_menu_sensibility()
{
	GtkWidget *widget;
	GSList *enable_item = NULL;
	GSList *disable_item = NULL;
	
	switch(global_edit_mode)
	{
	case EDIT_LEVEL:
		enable_item = g_slist_append(enable_item, "level_menu");
		enable_item = g_slist_append(enable_item, "view_menu");
		enable_item = g_slist_append(enable_item, "grid_bt");
		enable_item = g_slist_append(enable_item, "theme_wnd_bt");
		enable_item = g_slist_append(enable_item, "open_bt");

		disable_item = g_slist_append(disable_item, "theme_menu");
		break;

	case EDIT_THEME:
		enable_item = g_slist_append(enable_item, "theme_menu");
		enable_item = g_slist_append(enable_item, "theme_wnd_bt");

		disable_item = g_slist_append(disable_item, "level_menu");
		disable_item = g_slist_append(disable_item, "view_menu");
		disable_item = g_slist_append(disable_item, "open_bt");
		disable_item = g_slist_append(disable_item, "grid_bt");
		break;

	case EDIT_UNDEFINED:
		disable_item = g_slist_append(disable_item, "theme_menu");
		disable_item = g_slist_append(disable_item, "level_menu");
		disable_item = g_slist_append(disable_item, "view_menu");
		disable_item = g_slist_append(disable_item, "grid_bt");
		disable_item = g_slist_append(disable_item, "theme_wnd_bt");
		disable_item = g_slist_append(disable_item, "open_bt");
		break;
	}
	
	while(enable_item != NULL)
	{
		widget = lookup_widget(get_app(), (gchar*) enable_item->data);
		gtk_widget_set_sensitive(widget, TRUE);
		enable_item = enable_item->next;
	}

	while(disable_item != NULL)
	{
		widget = lookup_widget(get_app(), (gchar*) disable_item->data);
		gtk_widget_set_sensitive(widget, FALSE);
		disable_item = disable_item->next;
	}

	g_slist_free(enable_item);
	g_slist_free(disable_item);

}


/*===============================================================

  Callbacks for level properties

  --------------------------------------------------------------*/

void
on_level_name_changed                  (GtkEditable     *editable,
                                        gpointer         user_data)
{
	level_properties_update_level();
	level_ctrl_actual_level_renamed();
}


void
on_level_theme_name_entry_changed      (GtkEditable     *editable,
                                        gpointer         user_data)
{
	Level *level;
	Theme *theme;
	PlayFieldEditor *level_editor;
	PlayFieldEditor *goal_editor = get_goal_editor();
	
	level_properties_update_level();

	level = level_ctrl_get_actual_level();
	theme = theme_ctrl_get_theme_by_name(level->theme_name);      
	level_editor = get_level_editor();
	goal_editor = get_goal_editor();
		
	playfield_editor_set_theme(level_editor, theme, TRUE);
	playfield_editor_set_theme(goal_editor, theme, TRUE);
	image_win_ctrl_set_theme(theme); 				
	
	level_ctrl_set_actual_level_modified();
}


void
on_level_prop_changed                  (GtkWidget       *widget,
                                        gpointer         user_data)
{
	level_properties_update_level();
	level_ctrl_set_actual_level_modified();
}

/*===============================================================

  Callbacks for theme properties

  --------------------------------------------------------------*/
void
on_theme_name_changed                  (GtkEditable     *editable,
                                        gpointer         user_data)
{
	theme_properties_update_theme();
	theme_ctrl_actual_theme_renamed();
}


void
on_theme_dir_name_changed              (GtkEditable     *editable,
                                        gpointer         user_data)
{
	
}


void
on_theme_color_changed                 (GnomeColorPicker *gnomecolorpicker,
                                        guint            arg1,
                                        guint            arg2,
                                        guint            arg3,
                                        guint            arg4,
                                        gpointer         user_data)
{
	Theme *theme = theme_ctrl_get_actual_theme();
	if(!theme_ctrl_is_locale_theme(theme))
	{
		if(!theme_ctrl_make_theme_locale(theme))
		{
			return;
		}
	}
	theme_properties_update_theme();
	theme_ctrl_set_actual_theme_modified();
	image_win_ctrl_update_theme(theme, 1000);
}


void
on_theme_selector_file_changed         (GtkEditable     *editable,
                                        gpointer         user_data)
{

}


void
on_theme_selector_browse_clicked       (GtkButton       *button,
                                        gpointer         user_data)
{
	GtkFileSelection *image_selector_dlg;

	/* Create file selector */     
	image_selector_dlg = GTK_FILE_SELECTION(gtk_file_selection_new("Please select an image ..."));
	if(image_selector_path != NULL)
	{
		gtk_file_selection_set_filename(image_selector_dlg, image_selector_path);
	}
	gtk_window_set_modal(GTK_WINDOW(image_selector_dlg), TRUE);


	gtk_signal_connect (GTK_OBJECT (image_selector_dlg->ok_button), "clicked",
			    GTK_SIGNAL_FUNC (on_imageselector_ok), image_selector_dlg);
	gtk_signal_connect_object (GTK_OBJECT (image_selector_dlg->cancel_button),
				   "clicked",
				   GTK_SIGNAL_FUNC (gtk_widget_destroy),
				   GTK_OBJECT(image_selector_dlg));

	gtk_window_set_position (GTK_WINDOW (image_selector_dlg), GTK_WIN_POS_MOUSE);

     
	/* Display that dialog */	
	gtk_widget_show (GTK_WIDGET(image_selector_dlg));

}

void on_imageselector_ok (GtkWidget *widget, GtkFileSelection *image_selector_dlg)
{
	gchar *selected_filename;
	
	selected_filename = gtk_file_selection_get_filename(image_selector_dlg);
	
	
	if(selected_filename)
	{
		if(g_strncasecmp(gnome_mime_type_of_file(selected_filename), 
				 "image/", 6) == 0)
		{
			Theme *theme = theme_ctrl_get_actual_theme();
			gchar *dirname;

			if(!theme_ctrl_is_locale_theme(theme))
			{
				if(!theme_ctrl_make_theme_locale(theme))
				{
					return;
				}
			}
			
			/* ok, we have an image here */
			theme_ctrl_new_selector_image(selected_filename);
			
			theme_properties_update_view();

			/* change default path for next time */
			if(image_selector_path != NULL) g_free(image_selector_path);
			dirname = g_dirname(selected_filename);
			image_selector_path = g_strconcat(dirname, "/", NULL);
			g_free(dirname);
			g_print("%s\n", image_selector_path);
		}
		else
		{
		        GtkWidget *dlg;
			dlg = GTK_WIDGET(gnome_message_box_new("You didn't select an image file!",
							       GNOME_MESSAGE_BOX_ERROR,
							       GNOME_STOCK_BUTTON_OK,
							       NULL));
			gnome_dialog_run(GNOME_DIALOG(dlg));
		}

	}
	
	gtk_widget_destroy(GTK_WIDGET(image_selector_dlg));
}

void
on_theme_tile_size_changed             (GtkEditable     *editable,
                                        gpointer         user_data)
{

}

void
on_atomixed_manual_activate            (GtkMenuItem     *menuitem,
                                        gpointer         user_data)
{
	gchar *helpfile;

	helpfile = gnome_help_file_find_file("atomixed", "atomixed.html");
	
	if(helpfile != NULL)
	{
		gchar *url;
		
		url = g_strconcat("ghelp:", helpfile, NULL);
		
		gnome_help_goto(NULL, url);
		
		g_free(url);
		g_free(helpfile);
	}
	else
	{
		gnome_error_dialog(_("Couldn't find the Atomixed manual!"));
	}

}

