/* -*- Mode: C; tab-width: 8; indent-tabs-mode: nil; c-basic-offset: 8 -*-
 *
 * Copyright 2009-2010  Red Hat, Inc,
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 * Written by: Matthias Clasen <mclasen@redhat.com>
 */

#include "config.h"

#include <stdlib.h>

#include <glib.h>
#include <glib/gi18n.h>
#include <gtk/gtk.h>

#include "um-names-dialog.h"
#include "um-user-manager.h"
#include "um-utils.h"

struct _UmNamesDialog {
        GtkWidget *dialog;
        GtkWidget *user_icon;
        GtkWidget *user_name;
        GtkWidget *shortname_entry;
        GtkWidget *name_entry;
        GtkWidget *ok_button;

        UmUser *user;
};

static void
cancel_names_dialog (GtkButton     *button,
                     UmNamesDialog *um)
{
        gtk_widget_hide (um->dialog);
        um_names_dialog_set_user (um, NULL);
}

static void
accept_names_dialog (GtkButton     *button,
                     UmNamesDialog *um)
{
        const gchar *shortname;
        const gchar *name;

        shortname = gtk_entry_get_text (GTK_ENTRY (um->shortname_entry));
        name = gtk_entry_get_text (GTK_ENTRY (um->name_entry));

        um_user_set_user_name (um->user, shortname);
        um_user_set_real_name (um->user, name);

        gtk_widget_hide (um->dialog);
        um_names_dialog_set_user (um, NULL);
}

static void
update_sensitivity (UmNamesDialog *um)
{
        const gchar *name, *shortname;
        gboolean can_create;
        UmUserManager *manager;
        UmUser *user;
        gchar *tip;

        name = gtk_entry_get_text (GTK_ENTRY (um->name_entry));
        shortname = gtk_entry_get_text (GTK_ENTRY (um->shortname_entry));

        can_create = name[0] != 0 && shortname[0] != 0;

        if (can_create) {
                manager = um_user_manager_ref_default ();
                user = um_user_manager_get_user (manager, shortname);
                g_object_unref (manager);
                if (user != NULL && user != um->user) {
                        gtk_widget_error_bell (um->shortname_entry);
                        gtk_entry_set_icon_from_stock (GTK_ENTRY (um->shortname_entry),
                                                       GTK_ENTRY_ICON_SECONDARY,
                                                       GTK_STOCK_DIALOG_ERROR);
                        tip = g_strdup_printf (_("A user with the short name '%s' already exists."),
                                               shortname);
                        gtk_entry_set_icon_tooltip_text (GTK_ENTRY (um->shortname_entry),
                                                         GTK_ENTRY_ICON_SECONDARY,
                                                         tip);
                        g_free (tip);

                        can_create = FALSE;
                }
                else {
                        gtk_entry_set_icon_from_pixbuf (GTK_ENTRY (um->shortname_entry),
                                                        GTK_ENTRY_ICON_SECONDARY,
                                                        NULL);
                }
        }

        gtk_widget_set_sensitive (um->ok_button, can_create);
}

static void
shortname_changed (GtkWidget     *entry,
                   GParamSpec    *pspec,
                   UmNamesDialog *um)
{
        update_sensitivity (um);
}

static void
name_changed (GtkWidget     *entry,
              GParamSpec    *pspec,
              UmNamesDialog *um)
{
        update_sensitivity (um);
}

UmNamesDialog *
um_names_dialog_new (void)
{
        GtkBuilder *builder;
        GtkWidget *widget;
        UmNamesDialog *um;
        const gchar *filename;
        GError *error = NULL;

        builder = gtk_builder_new ();

        filename = UIDIR "/names-dialog.ui";
        if (!g_file_test (filename, G_FILE_TEST_EXISTS))
                filename = "../data/names-dialog.ui";
        if (!gtk_builder_add_from_file (builder, filename, &error)) {
                g_error ("%s", error->message);
                g_error_free (error);
                exit (1);
        }

        um = g_new0 (UmNamesDialog, 1);

        widget = (GtkWidget *) gtk_builder_get_object (builder, "dialog");
        g_signal_connect (widget, "delete-event",
                          G_CALLBACK (gtk_widget_hide_on_delete), NULL);
        um->dialog = widget;

        widget = (GtkWidget *) gtk_builder_get_object (builder, "cancel-button");
        g_signal_connect (widget, "clicked",
                          G_CALLBACK (cancel_names_dialog), um);

        widget = (GtkWidget *) gtk_builder_get_object (builder, "ok-button");
        g_signal_connect (widget, "clicked",
                          G_CALLBACK (accept_names_dialog), um);
        gtk_widget_grab_default (widget);

        widget = (GtkWidget *) gtk_builder_get_object (builder, "shortname-entry");
        g_signal_connect (widget, "notify::text",
                          G_CALLBACK (shortname_changed), um);
        um->shortname_entry = widget;

        widget = (GtkWidget *) gtk_builder_get_object (builder, "name-entry");
        g_signal_connect (widget, "notify::text",
                          G_CALLBACK (name_changed), um);
        um->name_entry = widget;

        um->ok_button = (GtkWidget *) gtk_builder_get_object (builder, "ok-button");
        um->user_icon = (GtkWidget *) gtk_builder_get_object (builder, "user-icon");
        um->user_name = (GtkWidget *) gtk_builder_get_object (builder, "user-name");

        return um;
}

void
um_names_dialog_free (UmNamesDialog *um)
{
        gtk_widget_destroy (um->dialog);

        if (um->user)
                g_object_unref (um->user);

        g_free (um);
}

void
um_names_dialog_set_user (UmNamesDialog *um,
                          UmUser        *user)
{
        GdkPixbuf *pixbuf;
        const gchar *name;

        if (um->user) {
                g_object_unref (um->user);
                um->user = NULL;
        }
        um->user = user;
        if (um->user) {
                g_object_ref (user);

                pixbuf = um_user_render_icon (user, FALSE, 48);
                gtk_image_set_from_pixbuf (GTK_IMAGE (um->user_icon), pixbuf);
                g_object_unref (pixbuf);

                name = um_user_get_real_name (user);
                gtk_label_set_label (GTK_LABEL (um->user_name), name);

                name = um_user_get_user_name (user);
                gtk_entry_set_text (GTK_ENTRY (um->shortname_entry), name);

                name = um_user_get_real_name (user);
                gtk_entry_set_text (GTK_ENTRY (um->name_entry), name);
        }
}

void
um_names_dialog_show (UmNamesDialog *um,
                      GtkWindow     *parent)
{
        gtk_window_set_transient_for (GTK_WINDOW (um->dialog), parent);
        gtk_window_present (GTK_WINDOW (um->dialog));
        gtk_widget_grab_focus (um->name_entry);
}


