"use strict";
/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *  http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.recreateQueryStringFromFormattedSyntaxTrees = void 0;
var types_1 = require("./types");
var utils_1 = require("./utils");
var recreateQueryStringFromFormattedSyntaxTree = function (syntaxTree) {
    if (syntaxTree.type === types_1.TokenType.NonGremlinCode) {
        return syntaxTree.code;
    }
    if (syntaxTree.type === types_1.TokenType.Traversal) {
        return syntaxTree.stepGroups
            .map(function (stepGroup) { return stepGroup.steps.map(recreateQueryStringFromFormattedSyntaxTree).join('.'); })
            .join('\n');
    }
    if (syntaxTree.type === types_1.TokenType.Method) {
        return ((syntaxTree.shouldStartWithDot ? '.' : '') +
            [
                recreateQueryStringFromFormattedSyntaxTree(syntaxTree.method) + '(',
                syntaxTree.argumentGroups
                    .map(function (args) { return args.map(recreateQueryStringFromFormattedSyntaxTree).join(', '); })
                    .join(',\n') +
                    ')' +
                    (syntaxTree.shouldEndWithDot ? '.' : ''),
            ].join(syntaxTree.argumentsShouldStartOnNewLine ? '\n' : ''));
    }
    if (syntaxTree.type === types_1.TokenType.Closure) {
        return ((syntaxTree.shouldStartWithDot ? '.' : '') +
            recreateQueryStringFromFormattedSyntaxTree(syntaxTree.method) +
            '{' +
            syntaxTree.closureCodeBlock
                .map(function (_a, i) {
                var lineOfCode = _a.lineOfCode, localIndentation = _a.localIndentation;
                return "".concat((0, utils_1.spaces)(localIndentation)).concat(lineOfCode);
            })
                .join('\n') +
            '}' +
            (syntaxTree.shouldEndWithDot ? '.' : ''));
    }
    if (syntaxTree.type === types_1.TokenType.String) {
        return (0, utils_1.spaces)(syntaxTree.localIndentation) + syntaxTree.string;
    }
    if (syntaxTree.type === types_1.TokenType.Word) {
        return ((0, utils_1.spaces)(syntaxTree.localIndentation) +
            (syntaxTree.shouldStartWithDot ? '.' : '') +
            syntaxTree.word +
            (syntaxTree.shouldEndWithDot ? '.' : ''));
    }
    // The following line is just here to convince TypeScript that the return type
    // is string and not string | undefined.
    return '';
};
var withIndentationIfNotEmpty = function (indentation) { return function (lineOfCode) {
    if (!lineOfCode)
        return lineOfCode;
    return (0, utils_1.spaces)(indentation) + lineOfCode;
}; };
var lineIsEmpty = function (lineOfCode) {
    return lineOfCode.split('').every((0, utils_1.eq)(' '));
};
var removeIndentationFromEmptyLines = function (lineOfCode) {
    if (lineIsEmpty(lineOfCode))
        return '';
    return lineOfCode;
};
var recreateQueryStringFromFormattedSyntaxTrees = function (_a) {
    var globalIndentation = _a.globalIndentation;
    return function (syntaxTrees) {
        return syntaxTrees
            .map(recreateQueryStringFromFormattedSyntaxTree)
            .join('')
            .split('\n')
            .map(withIndentationIfNotEmpty(globalIndentation))
            .map(removeIndentationFromEmptyLines)
            .join('\n');
    };
};
exports.recreateQueryStringFromFormattedSyntaxTrees = recreateQueryStringFromFormattedSyntaxTrees;
