/*
 * Decompiled with CFR 0.152.
 */
package org.apache.bookkeeper.bookie;

import com.google.common.base.Stopwatch;
import io.netty.util.internal.PlatformDependent;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Iterator;
import java.util.LinkedHashMap;
import java.util.LinkedList;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentLinkedQueue;
import java.util.concurrent.ConcurrentMap;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicInteger;
import org.apache.bookkeeper.bookie.Bookie;
import org.apache.bookkeeper.bookie.EntryKey;
import org.apache.bookkeeper.bookie.FileInfo;
import org.apache.bookkeeper.bookie.FileInfoBackingCache;
import org.apache.bookkeeper.bookie.IndexPersistenceMgr;
import org.apache.bookkeeper.bookie.LEPStateChangeCallback;
import org.apache.bookkeeper.bookie.LedgerCache;
import org.apache.bookkeeper.bookie.LedgerEntryPage;
import org.apache.bookkeeper.bookie.stats.IndexInMemPageMgrStats;
import org.apache.bookkeeper.conf.ServerConfiguration;
import org.apache.bookkeeper.stats.Counter;
import org.apache.bookkeeper.stats.Gauge;
import org.apache.bookkeeper.stats.OpStatsLogger;
import org.apache.bookkeeper.stats.StatsLogger;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

class IndexInMemPageMgr {
    private static final Logger LOG = LoggerFactory.getLogger(IndexInMemPageMgr.class);
    private static final ConcurrentHashMap<Long, LedgerEntryPage> EMPTY_PAGE_MAP = new ConcurrentHashMap();
    final int pageSize;
    final int entriesPerPage;
    final int pageLimit;
    final InMemPageCollection pageMapAndList;
    private final AtomicInteger pageCount = new AtomicInteger(0);
    private final IndexPersistenceMgr indexPersistenceManager;
    private final Counter ledgerCacheHitCounter;
    private final Counter ledgerCacheMissCounter;
    private final OpStatsLogger ledgerCacheReadPageStats;

    public IndexInMemPageMgr(int pageSize, int entriesPerPage, ServerConfiguration conf, IndexPersistenceMgr indexPersistenceManager, StatsLogger statsLogger) {
        this.pageSize = pageSize;
        this.entriesPerPage = entriesPerPage;
        this.indexPersistenceManager = indexPersistenceManager;
        this.pageMapAndList = new InMemPageCollection(statsLogger);
        long maxDirectMemory = PlatformDependent.estimateMaxDirectMemory();
        this.pageLimit = conf.getPageLimit() <= 0 ? (int)(maxDirectMemory / 3L / (long)this.pageSize) : conf.getPageLimit();
        LOG.info("maxDirectMemory = {}, pageSize = {}, pageLimit = {}", new Object[]{maxDirectMemory, pageSize, this.pageLimit});
        this.ledgerCacheHitCounter = statsLogger.getCounter("LEDGER_CACHE_HIT");
        this.ledgerCacheMissCounter = statsLogger.getCounter("LEDGER_CACHE_MISS");
        this.ledgerCacheReadPageStats = statsLogger.getOpStatsLogger("LEDGER_CACHE_READ_PAGE");
        statsLogger.registerGauge("NUM_INDEX_PAGES", (Gauge)new Gauge<Integer>(){

            public Integer getDefaultValue() {
                return 0;
            }

            public Integer getSample() {
                return IndexInMemPageMgr.this.getNumUsedPages();
            }
        });
    }

    public int getPageSize() {
        return this.pageSize;
    }

    private int getNumUsedPages() {
        return this.pageCount.get();
    }

    LedgerEntryPage getLedgerEntryPage(long ledger, long pageEntry) throws IOException {
        LedgerEntryPage lep = this.getLedgerEntryPageFromCache(ledger, pageEntry, false);
        if (lep == null) {
            this.ledgerCacheMissCounter.inc();
            lep = this.grabLedgerEntryPage(ledger, pageEntry);
        } else {
            this.ledgerCacheHitCounter.inc();
        }
        return lep;
    }

    LedgerEntryPage getLedgerEntryPageFromCache(long ledger, long firstEntry, boolean onlyDirty) {
        LedgerEntryPage lep = this.pageMapAndList.getPage(ledger, firstEntry);
        if (onlyDirty && null != lep && lep.isClean()) {
            return null;
        }
        if (null != lep) {
            lep.usePage();
        }
        return lep;
    }

    private LedgerEntryPage grabLedgerEntryPage(long ledger, long pageEntry) throws IOException {
        LedgerEntryPage lep = this.grabCleanPage(ledger, pageEntry);
        try {
            Stopwatch readPageStopwatch = Stopwatch.createStarted();
            boolean isNewPage = this.indexPersistenceManager.updatePage(lep);
            if (!isNewPage) {
                this.ledgerCacheReadPageStats.registerSuccessfulEvent(readPageStopwatch.elapsed(TimeUnit.MICROSECONDS), TimeUnit.MICROSECONDS);
            }
        }
        catch (IOException ie) {
            lep.releasePageNoCallback();
            this.pageMapAndList.addToListOfFreePages(lep);
            throw ie;
        }
        LedgerEntryPage oldLep = this.pageMapAndList.putPage(lep);
        if (lep != oldLep) {
            lep.releasePageNoCallback();
            this.pageMapAndList.addToListOfFreePages(lep);
            oldLep.usePage();
            lep = oldLep;
        }
        return lep;
    }

    void removePagesForLedger(long ledgerId) {
        this.pageMapAndList.removeEntriesForALedger(ledgerId);
    }

    long getLastEntryInMem(long ledgerId) {
        return this.pageMapAndList.getLastEntryInMem(ledgerId);
    }

    private LedgerEntryPage grabCleanPage(long ledger, long entry) throws IOException {
        if (entry % (long)this.entriesPerPage != 0L) {
            throw new IllegalArgumentException(entry + " is not a multiple of " + this.entriesPerPage);
        }
        while (true) {
            LedgerEntryPage lep;
            boolean canAllocate = false;
            if (this.pageCount.incrementAndGet() <= this.pageLimit) {
                canAllocate = true;
            } else {
                this.pageCount.decrementAndGet();
            }
            if (canAllocate) {
                lep = new LedgerEntryPage(this.pageSize, this.entriesPerPage, this.pageMapAndList);
                lep.setLedgerAndFirstEntry(ledger, entry);
                lep.usePage();
                return lep;
            }
            lep = this.pageMapAndList.grabCleanPage(ledger, entry);
            if (null != lep) {
                return lep;
            }
            LOG.info("Could not grab a clean page for ledger {}, entry {}, force flushing dirty ledgers.", (Object)ledger, (Object)entry);
            this.flushOneOrMoreLedgers(false);
        }
    }

    void flushOneOrMoreLedgers(boolean doAll) throws IOException {
        ArrayList<Long> ledgersToFlush = new ArrayList<Long>(this.pageMapAndList.getActiveLedgers());
        for (Long potentiallyDirtyLedger : ledgersToFlush) {
            try {
                this.flushSpecificLedger(potentiallyDirtyLedger);
            }
            catch (Bookie.NoLedgerException e) {
                continue;
            }
            if (doAll) continue;
            break;
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void flushSpecificLedger(long ledger) throws IOException {
        LinkedList firstEntryList = this.pageMapAndList.getFirstEntryListToBeFlushed(ledger);
        this.indexPersistenceManager.flushLedgerHeader(ledger);
        if (null == firstEntryList || firstEntryList.size() == 0) {
            if (LOG.isDebugEnabled()) {
                LOG.debug("Nothing to flush for ledger {}.", (Object)ledger);
            }
            return;
        }
        ArrayList<LedgerEntryPage> entries = new ArrayList<LedgerEntryPage>(firstEntryList.size());
        try {
            for (Long firstEntry : firstEntryList) {
                LedgerEntryPage lep = this.getLedgerEntryPageFromCache(ledger, firstEntry, true);
                if (lep == null) continue;
                entries.add(lep);
            }
            this.indexPersistenceManager.flushLedgerEntries(ledger, entries);
        }
        finally {
            for (LedgerEntryPage lep : entries) {
                lep.releasePage();
            }
        }
    }

    void putEntryOffset(long ledger, long entry, long offset) throws IOException {
        int offsetInPage = (int)(entry % (long)this.entriesPerPage);
        long pageEntry = entry - (long)offsetInPage;
        LedgerEntryPage lep = null;
        try {
            lep = this.getLedgerEntryPage(ledger, pageEntry);
            assert (lep != null);
            lep.setOffset(offset, offsetInPage * LedgerEntryPage.getIndexEntrySize());
        }
        catch (FileInfo.FileInfoDeletedException e) {
            throw new Bookie.NoLedgerException(ledger);
        }
        finally {
            if (null != lep) {
                lep.releasePage();
            }
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    long getEntryOffset(long ledger, long entry) throws IOException {
        int offsetInPage = (int)(entry % (long)this.entriesPerPage);
        long pageEntry = entry - (long)offsetInPage;
        LedgerEntryPage lep = null;
        try {
            lep = this.getLedgerEntryPage(ledger, pageEntry);
            long l = lep.getOffset(offsetInPage * LedgerEntryPage.getIndexEntrySize());
            return l;
        }
        finally {
            if (lep != null) {
                lep.releasePage();
            }
        }
    }

    public LedgerCache.PageEntriesIterable listEntries(long ledgerId) throws IOException {
        return new PageEntriesIterableImpl(ledgerId);
    }

    private class PageEntriesIterableImpl
    implements LedgerCache.PageEntriesIterable {
        final long ledgerId;
        final FileInfoBackingCache.CachedFileInfo fi;
        final long totalEntries;
        long curEntry = 0L;

        PageEntriesIterableImpl(long ledgerId) throws IOException {
            this.ledgerId = ledgerId;
            this.fi = IndexInMemPageMgr.this.indexPersistenceManager.getFileInfo(ledgerId, null);
            this.totalEntries = Long.max((long)IndexInMemPageMgr.this.entriesPerPage * (this.fi.size() / (long)IndexInMemPageMgr.this.pageSize), IndexInMemPageMgr.this.getLastEntryInMem(ledgerId));
        }

        @Override
        public Iterator<LedgerCache.PageEntries> iterator() {
            return new Iterator<LedgerCache.PageEntries>(){

                @Override
                public boolean hasNext() {
                    return PageEntriesIterableImpl.this.curEntry < PageEntriesIterableImpl.this.totalEntries;
                }

                @Override
                public LedgerCache.PageEntries next() {
                    PageEntriesImpl next = new PageEntriesImpl(PageEntriesIterableImpl.this.ledgerId, PageEntriesIterableImpl.this.curEntry);
                    PageEntriesIterableImpl.this.curEntry += (long)IndexInMemPageMgr.this.entriesPerPage;
                    return next;
                }
            };
        }

        @Override
        public void close() {
            this.fi.release();
        }
    }

    private class PageEntriesImpl
    implements LedgerCache.PageEntries {
        final long ledgerId;
        final long initEntry;

        PageEntriesImpl(long ledgerId, long initEntry) {
            this.ledgerId = ledgerId;
            this.initEntry = initEntry;
        }

        @Override
        public LedgerEntryPage getLEP() throws IOException {
            return IndexInMemPageMgr.this.getLedgerEntryPage(this.ledgerId, this.initEntry);
        }

        @Override
        public long getFirstEntry() {
            return this.initEntry;
        }

        @Override
        public long getLastEntry() {
            return this.initEntry + (long)IndexInMemPageMgr.this.entriesPerPage;
        }
    }

    private static class InMemPageCollection
    implements LEPStateChangeCallback {
        final ConcurrentMap<Long, ConcurrentMap<Long, LedgerEntryPage>> pages = new ConcurrentHashMap<Long, ConcurrentMap<Long, LedgerEntryPage>>();
        final Map<EntryKey, LedgerEntryPage> lruCleanPageMap = Collections.synchronizedMap(new LinkedHashMap(16, 0.75f, true));
        final ConcurrentLinkedQueue<LedgerEntryPage> listOfFreePages = new ConcurrentLinkedQueue();
        private final IndexInMemPageMgrStats inMemPageMgrStats;

        public InMemPageCollection(StatsLogger statsLogger) {
            this.inMemPageMgrStats = new IndexInMemPageMgrStats(statsLogger);
        }

        private LedgerEntryPage getPage(long ledgerId, long firstEntry) {
            ConcurrentMap map = (ConcurrentMap)this.pages.get(ledgerId);
            if (null != map) {
                return (LedgerEntryPage)map.get(firstEntry);
            }
            return null;
        }

        private LedgerEntryPage putPage(LedgerEntryPage lep) {
            LedgerEntryPage oldPage;
            ConcurrentMap<Long, LedgerEntryPage> map = (ConcurrentHashMap<Long, LedgerEntryPage>)this.pages.get(lep.getLedger());
            if (null == map) {
                ConcurrentHashMap<Long, LedgerEntryPage> mapToPut = new ConcurrentHashMap<Long, LedgerEntryPage>();
                map = this.pages.putIfAbsent(lep.getLedger(), mapToPut);
                if (null == map) {
                    map = mapToPut;
                }
            }
            if (null == (oldPage = map.putIfAbsent(lep.getFirstEntry(), lep))) {
                oldPage = lep;
                this.addToCleanPagesList(lep);
            }
            return oldPage;
        }

        private long getLastEntryInMem(long ledgerId) {
            long lastEntry = 0L;
            ConcurrentMap map = (ConcurrentMap)this.pages.get(ledgerId);
            if (map != null) {
                for (LedgerEntryPage lep : map.values()) {
                    if (lep.getMaxPossibleEntry() < lastEntry) continue;
                    lep.usePage();
                    long highest = lep.getLastEntry();
                    if (highest > lastEntry) {
                        lastEntry = highest;
                    }
                    lep.releasePage();
                }
            }
            return lastEntry;
        }

        private void removeEntriesForALedger(long ledgerId) {
            ConcurrentMap lPages = (ConcurrentMap)this.pages.remove(ledgerId);
            if (null != lPages) {
                for (Map.Entry pageEntry : lPages.entrySet()) {
                    LedgerEntryPage lep = (LedgerEntryPage)pageEntry.getValue();
                    lep.usePage();
                    lep.markDeleted();
                    lep.releasePage();
                }
            }
        }

        private LinkedList<Long> getFirstEntryListToBeFlushed(long ledgerId) {
            ConcurrentMap pageMap = (ConcurrentMap)this.pages.get(ledgerId);
            if (pageMap == null || pageMap.isEmpty()) {
                return null;
            }
            LinkedList<Long> firstEntryList = new LinkedList<Long>();
            for (Map.Entry entry : pageMap.entrySet()) {
                LedgerEntryPage lep = (LedgerEntryPage)entry.getValue();
                if (lep.isClean()) {
                    if (!lep.inUse()) {
                        this.addToCleanPagesList(lep);
                    }
                    if (!LOG.isTraceEnabled()) continue;
                    LOG.trace("Page is clean " + lep);
                    continue;
                }
                firstEntryList.add(lep.getFirstEntry());
            }
            return firstEntryList;
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        private void addToCleanPagesList(LedgerEntryPage lep) {
            Map<EntryKey, LedgerEntryPage> map = this.lruCleanPageMap;
            synchronized (map) {
                if (lep.isClean() && !lep.inUse()) {
                    this.lruCleanPageMap.put(lep.getEntryKey(), lep);
                }
            }
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        private void removeFromCleanPageList(LedgerEntryPage lep) {
            Map<EntryKey, LedgerEntryPage> map = this.lruCleanPageMap;
            synchronized (map) {
                if (!lep.isClean() || lep.inUse()) {
                    this.lruCleanPageMap.remove(lep.getEntryKey());
                }
            }
        }

        Set<Long> getActiveLedgers() {
            return this.pages.keySet();
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        LedgerEntryPage grabCleanPage(long ledgerId, long firstEntry) {
            LedgerEntryPage lep = this.listOfFreePages.poll();
            if (null != lep) {
                lep.resetPage();
                lep.setLedgerAndFirstEntry(ledgerId, firstEntry);
                lep.usePage();
                return lep;
            }
            while (this.lruCleanPageMap.size() > 0) {
                lep = null;
                Map<EntryKey, LedgerEntryPage> map = this.lruCleanPageMap;
                synchronized (map) {
                    Iterator<Map.Entry<EntryKey, LedgerEntryPage>> iterator = this.lruCleanPageMap.entrySet().iterator();
                    Map.Entry<EntryKey, LedgerEntryPage> entry = null;
                    while (iterator.hasNext()) {
                        entry = iterator.next();
                        iterator.remove();
                        if (!entry.getValue().isClean() || entry.getValue().inUse()) continue;
                        lep = entry.getValue();
                        break;
                    }
                    if (null == lep) {
                        if (LOG.isDebugEnabled()) {
                            LOG.debug("Did not find eligible page in the first pass");
                        }
                        return null;
                    }
                }
                ConcurrentMap pageMap = (ConcurrentMap)this.pages.get(lep.getLedger());
                if (null != pageMap && pageMap.remove(lep.getFirstEntry(), lep)) {
                    if (!lep.isClean()) {
                        pageMap.put(lep.getFirstEntry(), lep);
                        lep = null;
                        continue;
                    }
                    this.pages.remove(lep.getLedger(), EMPTY_PAGE_MAP);
                    lep.usePage();
                    lep.zeroPage();
                    lep.setLedgerAndFirstEntry(ledgerId, firstEntry);
                    return lep;
                }
                lep = null;
            }
            return lep;
        }

        public void addToListOfFreePages(LedgerEntryPage lep) {
            if (null == lep || lep.inUse()) {
                this.inMemPageMgrStats.getIllegalStateResetCounter().inc();
            }
            if (null != lep) {
                this.listOfFreePages.add(lep);
            }
        }

        @Override
        public void onSetInUse(LedgerEntryPage lep) {
            this.removeFromCleanPageList(lep);
        }

        @Override
        public void onResetInUse(LedgerEntryPage lep) {
            if (!lep.isDeleted()) {
                this.addToCleanPagesList(lep);
            } else {
                this.addToListOfFreePages(lep);
            }
        }

        @Override
        public void onSetClean(LedgerEntryPage lep) {
            this.addToCleanPagesList(lep);
        }

        @Override
        public void onSetDirty(LedgerEntryPage lep) {
            this.removeFromCleanPageList(lep);
        }
    }
}

