/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 * 
 *      http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.apache.portals.applications.webcontent2.rewriter.htmlcleaner;

import org.apache.commons.beanutils.ConstructorUtils;
import org.htmlcleaner.HtmlCleaner;
import org.htmlcleaner.Serializer;

/**
 * Default generic implementation of {@link SerializerFactory}.
 * <p>
 * A bean of this class should be injected with the physical HtmlCleaner
 * {@link Serializer} class and object argument array.
 * Then this will find and invoke the constructor of the specified class
 * with the specified object argument array to create a {@link Serializer} instance.
 * </p>
 */
public class DefaultSerializerFactory implements SerializerFactory
{
    private Class<? extends Serializer> serializerClass;
    private Object[] arguments;

    /**
     * Zero-argument default constructor.
     */
    public DefaultSerializerFactory()
    {
    }

    /**
     * Returns the class of the physical HtmlCleaner {@link Serializer}
     * such as {@link org.htmlcleaner.SimpleHtmlSerializer}.
     * @return
     */
    public Class<? extends Serializer> getSerializerClass()
    {
        return serializerClass;
    }

    /**
     * Sets the class of the physical HtmlCleaner {@link Serializer}
     * such as {@link org.htmlcleaner.SimpleHtmlSerializer}.
     * @param serializerClass
     */
    public void setSerializerClass(Class<? extends Serializer> serializerClass)
    {
        this.serializerClass = serializerClass;
    }

    /**
     * Returns the object argument array to be used when invoking the constructor
     * of the physical {@link Serializer} class.
     * @return
     */
    public Object[] getArguments()
    {
        return arguments;
    }

    /**
     * Sets the object argument array to be used when invoking the constructor
     * of the physical {@link Serializer} class.
     * @param arguments
     */
    public void setArguments(Object[] arguments)
    {
        this.arguments = arguments;
    }

    /**
     * Creates a {@link Serializer} instance by finding and invoking the constructor
     * of the injected HtmlCleaner {@link Serializer} class with the given
     * object argument array.
     */
    public Serializer createSerializer(HtmlCleaner cleaner) throws Exception
    {
        if (arguments == null)
        {
            arguments = new Object [] { cleaner.getProperties() };
        }

        return (Serializer) ConstructorUtils.invokeConstructor(serializerClass, arguments);
    }
}
