/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 * 
 *      http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.apache.portals.applications.webcontent.proxy;

import java.io.StringReader;
import java.io.StringWriter;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import junit.framework.TestCase;

import org.apache.commons.beanutils.BeanUtils;
import org.apache.commons.io.IOUtils;
import org.apache.commons.lang.StringUtils;
import org.apache.portals.applications.webcontent.proxy.impl.DefaultHttpReverseProxyPathMapperImpl;
import org.apache.portals.applications.webcontent.proxy.impl.DefaultHttpReverseProxyPathMapperProviderImpl;
import org.apache.portals.applications.webcontent.proxy.impl.DefaultReverseProxyRewritingContext;
import org.apache.portals.applications.webcontent.proxy.impl.DefaultReverseProxyLinkRewritingParserAdaptor;
import org.apache.portals.applications.webcontent.rewriter.RewriterController;
import org.apache.portals.applications.webcontent.rewriter.rules.Ruleset;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * TestSimpleLinkRewritingParserAaptor
 * 
 * @version $Id: TestReverseProxyLinkRewritingParserAaptor.java 1372685 2012-08-14 00:26:42Z woonsan $
 */
public class TestReverseProxyLinkRewritingParserAaptor extends TestCase
{
    
    private static Logger log = LoggerFactory.getLogger(TestReverseProxyLinkRewritingParserAaptor.class);
    
    private String html =
        "<html>\n" +
        "<head>\n" +
        "<link type=\"text/css\" rel=\"stylesheet\" href=\"/skin/css/screen1.css\" media=\"all\">\n" +
        "<link type=\"text/css\" rel=\"stylesheet\" href=\"http://portals.apache.org/skin/css/screen2.css\" media=\"all\">\n" +
        "<style>\n" +
        "@import url(\"/skin/css/import1.css\");\n" +
        "@import url(\"http://portals.apache.org/skin/css/import2.css\");\n" +
        "</style>\n" +
        "</head>\n" +
        "<body>\n" +
        "<p><a href=\"mission.html\">Mission</a> and <a href='about.html'>About</a></p>\n" +
        "<p><a href=\"/docs/\">Docs</a> and <a href='/documents/'>Documents</a></p>\n" +
        "<p><a href=\"http://projects.apache.org\" title=\"Apache Projects\">Projects</a> <a href=\"http://portals.apache.org/apa.html\">Applications</a></p>\n" +
        "<p><a href=\"http://www.apache.org/events.html\">Events</a> <a href='http://www.apache.org/apachecon.html'>Apache Con</a></p>\n" +
        "<p><a href=\"http://www.localhost.com/manager/list\">Application Management</a> <a href='http://www.localhost.com/manager/start'>Start Application</a></p>\n" +
        "<p><a \n" +
        "href=\"/docs/\">Docs</a> and <a href='/documents/'>Documents</a></p>\n" +
        "<p><a \n" +
        "href=\"http://www.localhost.com/manager/list\">Application Management</a></p>\n" +
        "<p><a href=\"https://blogs.apache.org/index.html\">Apache Blogs</a></p>\n" +
        "<p><a href=\"http://apache.org/index.html\">Apache Software Foundation</a></p>\n" +
        "<script>new AjaxUpdate('/lazyLoader');</script>\n" +
        "</body>\n" +
        "</html>";
    
    private HttpReverseProxyPathMapperProvider proxyPathMapperProvider;
    private HttpReverseProxyPathMapper apacheProxyPathMapper;
    private HttpReverseProxyPathMapper apache0ProxyPathMapper;
    private HttpReverseProxyPathMapper securedApacheProxyPathMapper;
    private HttpReverseProxyPathMapper localhostProxyPathMapper;
    
    @Override
    public void setUp()
    {
        apacheProxyPathMapper = new DefaultHttpReverseProxyPathMapperImpl("apache", "/*_apache/", "http://$1.apache.org/");
        apache0ProxyPathMapper = new DefaultHttpReverseProxyPathMapperImpl("apache0", "/apache/", "http://apache.org/");
        securedApacheProxyPathMapper = new DefaultHttpReverseProxyPathMapperImpl("secure_apache", "/secure/*_apache/", "https://$1.apache.org/");
        localhostProxyPathMapper = new DefaultHttpReverseProxyPathMapperImpl("localhost", "/localhost/", "http://www.localhost.com/");
        
        Map<HttpReverseProxyPathMapper, RewriterController> rewriterControllerMap = new HashMap<HttpReverseProxyPathMapper, RewriterController>();
        Map<HttpReverseProxyPathMapper, Ruleset> rewriterRulesetMap = new HashMap<HttpReverseProxyPathMapper, Ruleset>();
        List<HttpReverseProxyPathMapper> proxyPathMappers = new ArrayList<HttpReverseProxyPathMapper>();
        proxyPathMappers.add(apacheProxyPathMapper);
        proxyPathMappers.add(apache0ProxyPathMapper);
        proxyPathMappers.add(securedApacheProxyPathMapper);
        proxyPathMappers.add(localhostProxyPathMapper);
        
        proxyPathMapperProvider = new DefaultHttpReverseProxyPathMapperProviderImpl(proxyPathMappers, rewriterControllerMap, rewriterRulesetMap);
    }
    
    public void testPathMappingRewriting() throws Exception
    {
        ReverseProxyRewritingContext rewritingContext = 
            new DefaultReverseProxyRewritingContext(proxyPathMapperProvider.findMapper("/www_apache/index.html"), 
                                                    proxyPathMapperProvider, 
                                                    "/webcontent/rproxy");
        
        DefaultReverseProxyLinkRewritingParserAdaptor parserAdaptor = new DefaultReverseProxyLinkRewritingParserAdaptor();
        parserAdaptor.setReverseProxyRewritingContext(rewritingContext);
        
        parserAdaptor.setLookUpAllMappings(false);
        BeanUtils.setProperty(parserAdaptor, "lookUpAllMappings", new String [] { "true" });
        assertTrue(parserAdaptor.getLookUpAllMappings());
        BeanUtils.setProperty(parserAdaptor, "customPatterns", new String [] { "\\/lazyLoader" });
        BeanUtils.setProperty(parserAdaptor, "customReplaces", new String [] { "/busyLoader" });
        
        StringReader reader = new StringReader(html);
        StringWriter writer = new StringWriter();
        parserAdaptor.rewrite(null, reader, writer);
        
        log.info("source: \n" + html);
        log.info("result: \n" + writer);
        
        List<String> lines = (List<String>) IOUtils.readLines(new StringReader(writer.toString()));
        
        int lineNum = 0;
        
        lineNum += 2;
        assertTrue("Wrong rewriting: " + lines.get(lineNum), StringUtils.contains(lines.get(lineNum), "href=\"/webcontent/rproxy/www_apache/skin/css/screen1.css\""));
        lineNum += 1;
        assertTrue("Wrong rewriting: " + lines.get(lineNum), StringUtils.contains(lines.get(lineNum), "href=\"/webcontent/rproxy/portals_apache/skin/css/screen2.css\""));
        
        lineNum += 2;
        assertTrue("Wrong rewriting: " + lines.get(lineNum), StringUtils.contains(lines.get(lineNum), "url(\"/webcontent/rproxy/www_apache/skin/css/import1.css\")"));
        lineNum += 1;
        assertTrue("Wrong rewriting: " + lines.get(lineNum), StringUtils.contains(lines.get(lineNum), "url(\"/webcontent/rproxy/portals_apache/skin/css/import2.css\")"));
        
        lineNum += 4;
        assertTrue("Wrong rewriting: " + lines.get(lineNum), StringUtils.contains(lines.get(lineNum), "href=\"mission.html\""));
        assertTrue("Wrong rewriting: " + lines.get(lineNum), StringUtils.contains(lines.get(lineNum), "href='about.html'"));
        lineNum += 1;
        assertTrue("Wrong rewriting: " + lines.get(lineNum), StringUtils.contains(lines.get(lineNum), "href=\"/webcontent/rproxy/www_apache/docs/\""));
        assertTrue("Wrong rewriting: " + lines.get(lineNum), StringUtils.contains(lines.get(lineNum), "href='/webcontent/rproxy/www_apache/documents/'"));
        lineNum += 1;
        assertTrue("Wrong rewriting: " + lines.get(lineNum), StringUtils.contains(lines.get(lineNum), "href=\"/webcontent/rproxy/projects_apache/\""));
        assertTrue("Wrong rewriting: " + lines.get(lineNum), StringUtils.contains(lines.get(lineNum), "href=\"/webcontent/rproxy/portals_apache/apa.html\""));
        lineNum += 1;
        assertTrue("Wrong rewriting: " + lines.get(lineNum), StringUtils.contains(lines.get(lineNum), "href=\"/webcontent/rproxy/www_apache/events.html\""));
        assertTrue("Wrong rewriting: " + lines.get(lineNum), StringUtils.contains(lines.get(lineNum), "href='/webcontent/rproxy/www_apache/apachecon.html'"));
        lineNum += 1;
        assertTrue("Wrong rewriting: " + lines.get(lineNum), StringUtils.contains(lines.get(lineNum), "href=\"/webcontent/rproxy/localhost/manager/list\""));
        assertTrue("Wrong rewriting: " + lines.get(lineNum), StringUtils.contains(lines.get(lineNum), "href='/webcontent/rproxy/localhost/manager/start'"));
        lineNum += 2;
        assertTrue("Wrong rewriting: " + lines.get(lineNum), StringUtils.contains(lines.get(lineNum), "href=\"/webcontent/rproxy/www_apache/docs/\""));
        lineNum += 2;
        assertTrue("Wrong rewriting: " + lines.get(lineNum), StringUtils.contains(lines.get(lineNum), "href=\"/webcontent/rproxy/localhost/manager/list\""));
        lineNum += 1;
        assertTrue("Wrong rewriting: " + lines.get(lineNum), StringUtils.contains(lines.get(lineNum), "href=\"/webcontent/rproxy/secure/blogs_apache/index.html\""));
        lineNum += 1;
        assertTrue("Wrong rewriting: " + lines.get(lineNum), StringUtils.contains(lines.get(lineNum), "href=\"/webcontent/rproxy/apache/index.html\""));
        lineNum += 1;
        assertTrue("Wrong rewriting: " + lines.get(lineNum), StringUtils.contains(lines.get(lineNum), "new AjaxUpdate('/busyLoader');"));
    }
    
}
