/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 * 
 *      http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.apache.portals.applications.webcontent.util;

import java.io.File;
import java.io.FileInputStream;
import java.io.FilenameFilter;
import java.io.IOException;
import java.io.InputStream;
import java.net.URI;
import java.net.URL;

import javax.portlet.PortletContext;
import javax.servlet.ServletContext;

import org.apache.commons.lang.StringUtils;
import org.apache.oro.io.GlobFilenameFilter;

/**
 * WebResourceUtils
 * <P>
 * This utility provides methods to retrieve web resources from classpath, servlet/portlet context or file system.
 * </P>
 * 
 * @version $Id: WebResourceUtils.java 1340195 2012-05-18 18:06:04Z woonsan $
 */
public class WebResourceUtils
{
    
    private WebResourceUtils()
    {
    }
    
    public static URL getResource(String resourcePath)
    {
        return getResource(resourcePath, null, null);
    }
    
    public static URL getResource(String resourcePath, ClassLoader classloader)
    {
        return getResource(resourcePath, classloader, null);
    }
    
    public static URL getResource(String resourcePath, Object context)
    {
        return getResource(resourcePath, null, context);
    }
    
    public static URL getResource(String resourcePath, ClassLoader classloader, Object context)
    {
        if (resourcePath == null)
        {
            return null;
        }
        
        try
        {
            if (StringUtils.startsWith(resourcePath, "file:"))
            {
                File file = new File(URI.create(resourcePath));
                
                if (file.exists())
                {
                    return file.toURL();
                }
            }
            else if (classloader != null && StringUtils.startsWith(resourcePath, "classpath:"))
            {
                return classloader.getResource(resourcePath.substring(10));
            }
            else if (context != null)
            {
                if (context instanceof ServletContext)
                {
                    return ((ServletContext) context).getResource(resourcePath);
                }
                else if (context instanceof PortletContext)
                {
                    return ((PortletContext) context).getResource(resourcePath);
                }
                else
                {
                    throw new IllegalArgumentException("The context should be either servlet context or portlet context.");
                }
            }
            else
            {
                File file = new File(resourcePath);
                
                if (file.exists())
                {
                    return file.toURL();
                }
            }
        }
        catch (Exception ignore)
        {
        }
        
        return null;
    }
    
    public static File getResourceAsFile(String resourcePath)
    {
        return getResourceAsFile(resourcePath, null, null);
    }
    
    public static File getResourceAsFile(String resourcePath, ClassLoader classloader)
    {
        return getResourceAsFile(resourcePath, classloader, null);
    }
    
    public static File getResourceAsFile(String resourcePath, Object context)
    {
        return getResourceAsFile(resourcePath, null, context);
    }
    
    public static File getResourceAsFile(String resourcePath, ClassLoader classloader, Object context)
    {
        if (resourcePath == null)
        {
            return null;
        }
        
        File file = null;
        
        try
        {
            if (StringUtils.startsWith(resourcePath, "file:"))
            {
                file = new File(URI.create(resourcePath));
            }
            else if (classloader != null && StringUtils.startsWith(resourcePath, "classpath:"))
            {
                URL resourceURL = classloader.getResource(resourcePath.substring(10));
                
                if (resourceURL != null && "file".equals(resourceURL.getProtocol()))
                {
                    file = new File(resourceURL.toURI());
                }
            }
            else if (context != null)
            {
                if (context instanceof ServletContext)
                {
                    String realPath = ((ServletContext) context).getRealPath(resourcePath);
                    
                    if (realPath != null)
                    {
                        file = new File(realPath);
                    }
                }
                else if (context instanceof PortletContext)
                {
                    String realPath = ((PortletContext) context).getRealPath(resourcePath);
                    
                    if (realPath != null)
                    {
                        file = new File(realPath);
                    }
                }
                else
                {
                    throw new IllegalArgumentException("The context should be either servlet context or portlet context.");
                }
            }
            else
            {
                file = new File(resourcePath);
            }
        }
        catch (Exception ignore)
        {
        }
        
        return file;
    }
    
    public static File [] getResourcesAsFiles(String resourcePath)
    {
        return getResourcesAsFiles(resourcePath, null, null);
    }
    
    public static File [] getResourcesAsFiles(String resourcePath, ClassLoader classloader)
    {
        return getResourcesAsFiles(resourcePath, classloader, null);
    }
    
    public static File [] getResourcesAsFiles(String resourcePath, Object context)
    {
        return getResourcesAsFiles(resourcePath, null, context);
    }
    
    public static File [] getResourcesAsFiles(String resourcePath, ClassLoader classloader, Object context)
    {
        File file = getResourceAsFile(resourcePath, classloader, context);

        if (file == null) {
        	return null;
        }

        File parent = file.getParentFile();
        FilenameFilter filter = new GlobFilenameFilter(file.getName());
        return parent.listFiles(filter);
    }
    
    public static InputStream getResourceAsStream(String resourcePath) throws IOException
    {
        return getResourceAsStream(resourcePath, null, null);
    }
    
    public static InputStream getResourceAsStream(String resourcePath, ClassLoader classloader) throws IOException
    {
        return getResourceAsStream(resourcePath, classloader, null);
    }
    
    public static InputStream getResourceAsStream(String resourcePath, Object context) throws IOException
    {
        return getResourceAsStream(resourcePath, null, context);
    }
    
    public static InputStream getResourceAsStream(String resourcePath, ClassLoader classloader, Object context) throws IOException
    {
        if (resourcePath == null)
        {
            return null;
        }
        
        if (StringUtils.startsWith(resourcePath, "file:"))
        {
            return new FileInputStream(new File(URI.create(resourcePath)));
        }
        else if (classloader != null && StringUtils.startsWith(resourcePath, "classpath:"))
        {
            return classloader.getResourceAsStream(resourcePath.substring(10));
        }
        else if (context != null)
        {
            if (context instanceof ServletContext)
            {
                return ((ServletContext) context).getResourceAsStream(resourcePath);
            }
            else if (context instanceof PortletContext)
            {
                return ((PortletContext) context).getResourceAsStream(resourcePath);
            }
            else
            {
                throw new IllegalArgumentException("The context should be either servlet context or portlet context.");
            }
        }
        else
        {
            return new FileInputStream(resourcePath);
        }
    }
    
}
