/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 * 
 *      http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.apache.portals.applications.webcontent.rewriter.html.neko;

import java.util.Stack;

import org.apache.portals.applications.webcontent.rewriter.Rewriter;
import org.apache.xerces.xni.Augmentations;
import org.apache.xerces.xni.QName;
import org.apache.xerces.xni.XMLAttributes;
import org.apache.xerces.xni.XMLString;
import org.apache.xerces.xni.XNIException;
import org.cyberneko.html.filters.ElementRemover;

/**
 * <p>
 * CallbackElementRemover
 * </p>
 * <p>
 *  Extended version of the NekoHTML ElementRemover which provides
 *  tag stripping/removal based on Rewriter settings.
 * </p>
 * 
 * @author <a href="mailto:weaver@apache.org">Scott T. Weaver </a>
 * @version $Id: CallbackElementRemover.java 891414 2009-12-16 20:19:02Z rwatler $
 *  
 */
public class CallbackElementRemover extends ElementRemover
{

    private Rewriter rewriter;
    private Stack tagStack = new Stack();

    /**
     * Construct with reference to the rewriter context to consult for rewriting advice
     */
    public CallbackElementRemover( Rewriter rewriter )
    {
        super();
        this.rewriter = rewriter;
    }
    
    
    // Base Class Protocol
    
    /**
     * <p>
     * characters
     * </p>
     * 
     * @see org.apache.xerces.xni.XMLDocumentHandler#characters(org.apache.xerces.xni.XMLString text, org.apache.xerces.xni.Augmentations augs)
     * @param text
     * @param augs
     * @throws org.apache.xerces.xni.XNIException
     */
    public void characters(XMLString text,Augmentations augs) throws XNIException
    {
        if (!tagStack.isEmpty())
        {
            String tag = (String)tagStack.peek();
            processText(tag, text);
        }
        super.characters(text, augs);
    }

    /**
     * <p>
     * comment
     * </p>
     * 
     * @see org.apache.xerces.xni.XMLDocumentHandler#comment(org.apache.xerces.xni.XMLString text, org.apache.xerces.xni.Augmentations augs)
     * @param text
     * @param augs
     * @throws org.apache.xerces.xni.XNIException
     */
    public void comment(XMLString text,Augmentations augs) throws XNIException
    {
        if (rewriter.shouldRemoveComments())
            return;
        super.comment(text,augs);
    }

    /**
     * <p>
     * emptyElement
     * </p>
     * 
     * @see org.apache.xerces.xni.XMLDocumentHandler#emptyElement(org.apache.xerces.xni.QName,
     *      org.apache.xerces.xni.XMLAttributes,
     *      org.apache.xerces.xni.Augmentations)
     * @param element
     * @param arg1
     * @param arg2
     * @throws org.apache.xerces.xni.XNIException
     */
    public void emptyElement( QName element, XMLAttributes attrs, Augmentations arg2 ) throws XNIException
    {
        String tag = element.rawname.toLowerCase();
        processTag(tag,attrs) ;
        super.emptyElement(element, attrs, arg2);
    }

    /**
     * <p>
     * startElement
     * </p>
     * 
     * @see org.apache.xerces.xni.XMLDocumentHandler#startElement(org.apache.xerces.xni.QName,
     *      org.apache.xerces.xni.XMLAttributes,
     *      org.apache.xerces.xni.Augmentations)
     * @param element
     * @param arg1
     * @param arg2
     * @throws org.apache.xerces.xni.XNIException
     */
    public void startElement( QName element, XMLAttributes attrs, Augmentations arg2 ) throws XNIException
    {
        String tag = element.rawname.toLowerCase();
        tagStack.push(tag);
        processTag(tag,attrs);
        super.startElement(element, attrs, arg2);
    }
    
    /**
     * <p>
     * endElement
     * </p>
     * 
     * @see org.apache.xerces.xni.XMLDocumentHandler#endElement(org.apache.xerces.xni.QName,
     *      org.apache.xerces.xni.Augmentations)
     * @param element
     * @param arg1
     * @throws org.apache.xerces.xni.XNIException
     */
    public void endElement( QName element, Augmentations arg1 ) throws XNIException
    {
        String tag = element.rawname.toLowerCase();
        if (!tagStack.isEmpty() && tag.equals(tagStack.peek()))
        {
            tagStack.pop();
        }
        super.endElement(element, arg1);
    }
    
    // Support Methods

    /**
     * <p>
     * processTag
     * </p>
     * 
     * @param tag
     * @param attrs
     */
    protected void processTag(String tag, XMLAttributes attrs)
    {
        if (fRemovedElements.contains(tag))
        {
            // already removed
            return ;
        }
        else if (rewriter.shouldStripTag(tag))
        {
            // first time for this tag...
            // strip - remove tag and any text associated with it
            removeElement(tag);
            return ;
        }
        else if (rewriter.shouldRemoveTag(tag))
        {
            // BOZO - block intentionally left EMPTY
            
            // first time for this tag...
            // remove - no directive necessary, the default behavior of ElementRemover is to drop tags that it does not know about (but the assocated text will remain)
            return ;
        }

        // OTHERWISE - explicitly accept (keep tag and associated text)
        // NOTE: even if fAcceptedElements contains the tag already, we need to reset the attribute names for this invocation context
        rewriter.enterConvertTagEvent(tag,new XMLAttributesWrapper(attrs));
        acceptElement(tag,getAttributeNames(attrs));
    }
    protected String[] getAttributeNames(XMLAttributes attrs)
    {
        int length = attrs != null ? attrs.getLength() : 0 ;
        String[] names = length > 0 ? new String[ length ] : null ;
        
        for( int i = 0, limit = length;i<limit;i++)
        {
            names[i] = attrs.getQName(i) ;
        }
        return names ;
    }

    /**
     * <p>
     * processText
     * </p>
     * 
     * @param tag
     * @param text
     */
    protected void processText(String tag, XMLString text)
    {
        String convertedText = rewriter.enterConvertTextEvent(tag, new String(text.ch, text.offset, text.length));
        if (convertedText != null)
        {
            char [] convertedTextChars = convertedText.toCharArray();
            text.setValues(convertedTextChars, 0, convertedTextChars.length);
        }
    }
}
