/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 * 
 *      http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.apache.portals.applications.webcontent.proxy.impl;

import java.util.Map;
import java.util.Set;

import org.apache.commons.lang.StringUtils;
import org.apache.portals.applications.webcontent.proxy.HttpReverseProxyPathMapper;

/**
 * Default implementation of <CODE>HttpProxyPathMapper</CODE> interface.
 * 
 * @version $Id: DefaultHttpReverseProxyPathMapperImpl.java 965693 2010-07-20 00:35:21Z woonsan $
 */
public class DefaultHttpReverseProxyPathMapperImpl implements HttpReverseProxyPathMapper
{
    private String name;
    private String localBasePath;
    private String remoteBaseURL;
    private boolean secured;
    private Set<String> allowedRoles;
    
    private Map<String, String> defaultRequestHeaders;
    private Map<String, String> defaultRequestCookies;
    
    private Set<String> rewriteCookiePathIncludes;
    private Set<String> rewriteCookiePathExcludes;
    
    private int hash;
    
    public DefaultHttpReverseProxyPathMapperImpl(String name, String localBasePath, String remoteBaseURL)
    {
        this(name, localBasePath, remoteBaseURL, false, null, null, null, null, null);
    }
    
    public DefaultHttpReverseProxyPathMapperImpl(String name, String localBasePath, String remoteBaseURL, 
                                                 boolean secured, Set<String> allowedRoles,
                                                 Map<String, String> defaultRequestHeaders, Map<String, String> defaultRequestCookies,
                                                 Set<String> rewriteCookiePathIncludes, Set<String> rewriteCookiePathExcludes)
    {
        this.name = name;
        this.localBasePath = localBasePath;
        this.remoteBaseURL = remoteBaseURL;
        this.secured = secured;
        this.allowedRoles = allowedRoles;
        this.defaultRequestHeaders = defaultRequestHeaders;
        this.defaultRequestCookies = defaultRequestCookies;
        this.rewriteCookiePathIncludes = rewriteCookiePathIncludes;
        this.rewriteCookiePathExcludes = rewriteCookiePathExcludes;
        
        hash = new StringBuilder(40).append(name).append(':').append(localBasePath).append(':').append(remoteBaseURL).toString().hashCode();
    }
    
    public String getName()
    {
        return name;
    }
    
    public String getLocalBasePath()
    {
        return localBasePath;
    }
    
    public String getRemoteBaseURL()
    {
        return remoteBaseURL;
    }
    
    public String getRemoteURL(String localPath)
    {
        if (StringUtils.startsWith(localPath, localBasePath))
        {
            return remoteBaseURL + localPath.substring(localBasePath.length());
        }
        
        return null;
    }
    
    public String getLocalPath(String remoteURL)
    {
        if (StringUtils.startsWith(remoteURL, remoteBaseURL))
        {
            return localBasePath + remoteURL.substring(remoteBaseURL.length());
        }
        
        return null;
    }
    
    public boolean isSecured()
    {
        return secured;
    }
    
    public Set<String> getAllowedRoles()
    {
        return allowedRoles;
    }
    
    public Map<String, String> getDefaultRequestHeaders()
    {
        return defaultRequestHeaders;
    }
    
    public Map<String, String> getDefaultRequestCookies()
    {
        return defaultRequestCookies;
    }
    
    @Override
    public int hashCode()
    {
        return hash;
    }
    
    @Override
    public boolean equals(Object o)
    {
        if (hashCode() != o.hashCode())
        {
            return false;
        }
        
        if (o instanceof HttpReverseProxyPathMapper)
        {
            HttpReverseProxyPathMapper mapper = (HttpReverseProxyPathMapper) o;
            return (StringUtils.equals(name, mapper.getName()) && 
                            StringUtils.equals(localBasePath, mapper.getLocalBasePath()) &&
                            StringUtils.equals(remoteBaseURL, mapper.getRemoteBaseURL()));
        }
        
        return false;
    }
    
    public Set<String> getRewriteCookiePathIncludes()
    {
        return rewriteCookiePathIncludes;
    }
    
    public Set<String> getRewriteCookiePathExcludes()
    {
        return rewriteCookiePathExcludes;
    }
    
}
