/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 * 
 *      http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.apache.portals.gems.dojo;

import javax.portlet.MimeResponse;
import javax.portlet.RenderResponse;

import org.apache.commons.lang.StringUtils;
import org.w3c.dom.Element;

/**
 * Help class for portlets which want to use DOJO AJAX technologies.
 * 
 * @version $Id: DojoPortletHelper.java 766781 2009-04-20 18:24:53Z woonsan $
 */
public class DojoPortletHelper
{
 
    /**
     * An attribute name of head element contributed to the portal page aggregation by a portlet.
     * The value of this attribute can be used by portal to merge contents of each contributed head element
     * into centralized element(s).
     * <BR/>
     * For example, if a contributed head element has an attribute value, 'dojo.require', with text content, 'dojo.lang.*'
     * and another contributed element has the same attribute value, 'dojo.require', with text content, 'dojo.event.*', then
     * a portal can merge those contents into a single script element for optimization.
     */
    public static final String HEAD_ELEMENT_CONTRIBUTION_MERGE_HINT_ATTRIBUTE = "org.apache.portals.portal.page.head.element.contribution.merge.hint";

    /**
     * An attribute value for key hint which can be used in head elements merging by portal.  
     */
    public static final String HEAD_ELEMENT_CONTRIBUTION_MERGE_HINT_KEY_DOJO_REQUIRES = "header.dojo.requires";

    /**
     * An ID attribute value which can be used in head elements merging by portal.  
     */
    public static final String HEAD_ELEMENT_CONTRIBUTION_ELEMENT_ID_DOJO_LIBRARY_INCLUDE = "header.dojo.library.include";
        
    private DojoPortletHelper()
    {
    }
    
    public static void contributeScript(RenderResponse response, String language, String type, String src, String textContent)
    {
        contributeScript(response, language, type, src, textContent, null);
    }
    
    public static void contributeScript(RenderResponse response, String language, String type, String src, String textContent, String id)
    {
        contributeScript(response, language, type, src, textContent, id, null);
    }
    
    public static void contributeScript(RenderResponse response, String language, String type, String src, String textContent, String id, String mergeHint)
    {
        Element headElem = response.createElement("script");
        
        if (language != null)
        {
            headElem.setAttribute("language", language);
        }
        
        if (src != null)
        {
            headElem.setAttribute("src", src);
        }
        
        if (type != null)
        {
            headElem.setAttribute("type", type);
        }
        
        if (textContent != null)
        {
            headElem.appendChild(headElem.getOwnerDocument().createTextNode(textContent));
        }
        
        if (id != null)
        {
            headElem.setAttribute("id", id);
        }
        
        if (mergeHint != null)
        {
            headElem.setAttribute(HEAD_ELEMENT_CONTRIBUTION_MERGE_HINT_ATTRIBUTE, mergeHint);
        }
        
        response.addProperty(MimeResponse.MARKUP_HEAD_ELEMENT, headElem);
    }
    
    public static void enableDojo(RenderResponse response, String dojoJsUrl)
    {
        contributeScript(response, "JavaScript", "text/javascript", dojoJsUrl == null ? "" : dojoJsUrl, null, HEAD_ELEMENT_CONTRIBUTION_ELEMENT_ID_DOJO_LIBRARY_INCLUDE, null);
    }
    
    public static void contributeDojoRequire(RenderResponse response, String dojoRequire)
    {
        String requireStatement = "dojo.require(\"" + dojoRequire + "\");";
        contributeScript(response, "JavaScript", "text/javascript", null, requireStatement, null, HEAD_ELEMENT_CONTRIBUTION_MERGE_HINT_KEY_DOJO_REQUIRES);
    }

    public static void contributeDojoRequires(RenderResponse response, String dojoRequires)
    {
        if (dojoRequires == null)
        {
            throw new IllegalArgumentException("dojoRequires is null.");
        }
        
        String [] dojoRequireArray = StringUtils.split(dojoRequires, ';');
        
        if (dojoRequireArray != null)
        {
            for (String dojoRequire : dojoRequireArray)
            {
                contributeDojoRequire(response, dojoRequire);
            }
        }
    }
    
}
