/*
 * Decompiled with CFR 0.152.
 */
package org.apache.knox.gateway.webshell;

import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.google.common.base.CharMatcher;
import java.io.IOException;
import java.nio.charset.StandardCharsets;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.atomic.AtomicInteger;
import org.apache.knox.gateway.audit.api.AuditServiceFactory;
import org.apache.knox.gateway.audit.api.Auditor;
import org.apache.knox.gateway.config.GatewayConfig;
import org.apache.knox.gateway.services.security.token.UnknownTokenException;
import org.apache.knox.gateway.webshell.ConnectionInfo;
import org.apache.knox.gateway.webshell.WebshellData;
import org.apache.knox.gateway.websockets.JWTValidator;
import org.apache.knox.gateway.websockets.ProxyWebSocketAdapter;
import org.eclipse.jetty.websocket.api.Session;

public class WebshellWebSocketAdapter
extends ProxyWebSocketAdapter {
    private Session session;
    private final ConnectionInfo connectionInfo;
    private final JWTValidator jwtValidator;
    private final StringBuilder auditBuffer;
    private final ObjectMapper objectMapper;
    private static final Auditor auditor = AuditServiceFactory.getAuditService().getAuditor("audit", "knox", "knox");

    public WebshellWebSocketAdapter(ExecutorService pool, GatewayConfig config, JWTValidator jwtValidator, AtomicInteger concurrentWebshells) {
        super(null, pool, null, config);
        this.jwtValidator = jwtValidator;
        this.auditBuffer = new StringBuilder();
        if (jwtValidator.getUsername() == null) {
            throw new RuntimeException("Needs user name in JWT to use WebShell");
        }
        this.connectionInfo = new ConnectionInfo(jwtValidator.getUsername(), config.getGatewayPIDDir(), concurrentWebshells);
        this.objectMapper = new ObjectMapper();
    }

    @Override
    public void onWebSocketConnect(Session session) {
        this.session = session;
        this.connectionInfo.connect();
        this.pool.execute(this::blockingReadFromHost);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void blockingReadFromHost() {
        byte[] buffer = new byte[this.config.getWebShellReadBufferSize()];
        try {
            int bytesRead;
            while ((bytesRead = this.connectionInfo.getInputStream().read(buffer)) != -1) {
                this.transToClient(new String(buffer, 0, bytesRead, StandardCharsets.UTF_8));
            }
        }
        catch (IOException e) {
            LOG.onError(e.toString());
        }
        finally {
            this.cleanup();
        }
    }

    @Override
    public void onWebSocketText(String message) {
        try {
            if (!this.jwtValidator.tokenIsStillValid()) {
                throw new RuntimeException("Token expired");
            }
            WebshellData webshellData = (WebshellData)this.objectMapper.readValue(message, WebshellData.class);
            this.transToHost(webshellData.getUserInput());
        }
        catch (JsonProcessingException | RuntimeException | UnknownTokenException e) {
            LOG.onError(e.toString());
            this.cleanup();
        }
    }

    private void transToHost(String userInput) {
        try {
            this.connectionInfo.getOutputStream().write(userInput.getBytes(StandardCharsets.UTF_8));
            this.connectionInfo.getOutputStream().flush();
            if (this.config.isWebShellAuditLoggingEnabled()) {
                this.audit(userInput);
            }
        }
        catch (IOException e) {
            LOG.onError("Error sending message to host");
            this.cleanup();
        }
    }

    private void transToClient(String message) {
        try {
            this.session.getRemote().sendString(message);
        }
        catch (IOException e) {
            LOG.onError("Error sending message to client");
            this.cleanup();
        }
    }

    @Override
    public void onWebSocketBinary(byte[] payload, int offset, int length) {
        throw new UnsupportedOperationException("Websocket for binary messages is not supported at this time.");
    }

    @Override
    public void onWebSocketClose(int statusCode, String reason) {
        LOG.debugLog("Closing websocket connection");
        this.cleanup();
    }

    @Override
    public void onWebSocketError(Throwable t) {
        LOG.onError(t.toString());
        this.cleanup();
    }

    private String cleanText(String text) {
        text = text.replaceAll("[\\^\\[OA|\\^\\[OB]", "");
        String noControl = CharMatcher.javaIsoControl().removeFrom((CharSequence)text);
        String printable = CharMatcher.invisible().removeFrom((CharSequence)noControl);
        String clean = CharMatcher.ascii().retainFrom((CharSequence)printable);
        return clean;
    }

    private void audit(String userInput) {
        String[] commands;
        this.auditBuffer.append(userInput);
        if ((userInput.contains("\r") || userInput.contains("\n")) && (commands = this.auditBuffer.toString().trim().split("\\s+")).length > 0) {
            auditor.audit("webshell", this.connectionInfo.getUsername() + ':' + this.connectionInfo.getPid(), "process", "success", this.cleanText(commands[0]));
            this.auditBuffer.setLength(0);
        }
    }

    private void cleanup() {
        if (this.session != null && this.session.isOpen()) {
            this.session.close();
            this.session = null;
        }
        this.connectionInfo.disconnect();
    }
}

