/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.hbase.backup.impl;

import java.io.Closeable;
import java.io.IOException;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.hbase.TableName;
import org.apache.hadoop.hbase.backup.BackupHFileCleaner;
import org.apache.hadoop.hbase.backup.BackupInfo;
import org.apache.hadoop.hbase.backup.BackupMasterObserver;
import org.apache.hadoop.hbase.backup.BackupObserver;
import org.apache.hadoop.hbase.backup.BackupType;
import org.apache.hadoop.hbase.backup.impl.BackupException;
import org.apache.hadoop.hbase.backup.impl.BackupSystemTable;
import org.apache.hadoop.hbase.backup.impl.BulkLoad;
import org.apache.hadoop.hbase.backup.impl.ExclusiveOperationException;
import org.apache.hadoop.hbase.backup.master.BackupLogCleaner;
import org.apache.hadoop.hbase.backup.master.LogRollMasterProcedureManager;
import org.apache.hadoop.hbase.backup.regionserver.LogRollRegionServerProcedureManager;
import org.apache.hadoop.hbase.client.Admin;
import org.apache.hadoop.hbase.client.Connection;
import org.apache.hadoop.hbase.client.TableDescriptor;
import org.apache.hadoop.hbase.util.EnvironmentEdgeManager;
import org.apache.yetus.audience.InterfaceAudience;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@InterfaceAudience.Private
public class BackupManager
implements Closeable {
    public static final String BACKUP_EXCLUSIVE_OPERATION_TIMEOUT_SECONDS_KEY = "hbase.backup.exclusive.op.timeout.seconds";
    private static final int DEFAULT_BACKUP_EXCLUSIVE_OPERATION_TIMEOUT = 3600;
    private static final Logger LOG = LoggerFactory.getLogger(BackupManager.class);
    protected Configuration conf = null;
    protected BackupInfo backupInfo = null;
    protected BackupSystemTable systemTable;
    protected final Connection conn;

    public BackupManager(Connection conn, Configuration conf) throws IOException {
        if (!conf.getBoolean("hbase.backup.enable", false)) {
            throw new BackupException("HBase backup is not enabled. Check your hbase.backup.enable setting.");
        }
        this.conf = conf;
        this.conn = conn;
        this.systemTable = new BackupSystemTable(conn);
    }

    protected BackupInfo getBackupInfo() {
        return this.backupInfo;
    }

    public static void decorateMasterConfiguration(Configuration conf) {
        String cleanerClass;
        if (!BackupManager.isBackupEnabled(conf)) {
            return;
        }
        String plugins = conf.get("hbase.master.logcleaner.plugins");
        if (!plugins.contains(cleanerClass = BackupLogCleaner.class.getCanonicalName())) {
            conf.set("hbase.master.logcleaner.plugins", plugins + "," + cleanerClass);
        }
        String classes = conf.get("hbase.procedure.master.classes");
        String masterProcedureClass = LogRollMasterProcedureManager.class.getName();
        if (classes == null) {
            conf.set("hbase.procedure.master.classes", masterProcedureClass);
        } else if (!classes.contains(masterProcedureClass)) {
            conf.set("hbase.procedure.master.classes", classes + "," + masterProcedureClass);
        }
        plugins = conf.get("hbase.master.hfilecleaner.plugins");
        conf.set("hbase.master.hfilecleaner.plugins", (plugins == null ? "" : plugins + ",") + BackupHFileCleaner.class.getName());
        String observerClass = BackupMasterObserver.class.getName();
        String masterCoProc = conf.get("hbase.coprocessor.master.classes");
        conf.set("hbase.coprocessor.master.classes", (masterCoProc == null ? "" : masterCoProc + ",") + observerClass);
        if (LOG.isDebugEnabled()) {
            LOG.debug("Added log cleaner: {}. Added master procedure manager: {}. Added master procedure manager: {}. Added master observer: {}", new Object[]{cleanerClass, masterProcedureClass, BackupHFileCleaner.class.getName(), observerClass});
        }
    }

    public static void decorateRegionServerConfiguration(Configuration conf) {
        if (!BackupManager.isBackupEnabled(conf)) {
            return;
        }
        String classes = conf.get("hbase.procedure.regionserver.classes");
        String regionProcedureClass = LogRollRegionServerProcedureManager.class.getName();
        if (classes == null) {
            conf.set("hbase.procedure.regionserver.classes", regionProcedureClass);
        } else if (!classes.contains(regionProcedureClass)) {
            conf.set("hbase.procedure.regionserver.classes", classes + "," + regionProcedureClass);
        }
        String coproc = conf.get("hbase.coprocessor.region.classes");
        String regionObserverClass = BackupObserver.class.getName();
        conf.set("hbase.coprocessor.region.classes", (coproc == null ? "" : coproc + ",") + regionObserverClass);
        if (LOG.isDebugEnabled()) {
            LOG.debug("Added region procedure manager: {}. Added region observer: {}", (Object)regionProcedureClass, (Object)regionObserverClass);
        }
    }

    public static boolean isBackupEnabled(Configuration conf) {
        return conf.getBoolean("hbase.backup.enable", false);
    }

    Configuration getConf() {
        return this.conf;
    }

    @Override
    public void close() {
        if (this.systemTable != null) {
            try {
                this.systemTable.close();
            }
            catch (Exception e) {
                LOG.error(e.toString(), (Throwable)e);
            }
        }
    }

    public BackupInfo createBackupInfo(String backupId, BackupType type, List<TableName> tableList, String targetRootDir, int workers, long bandwidth, boolean noChecksumVerify) throws BackupException {
        if (targetRootDir == null) {
            throw new BackupException("Wrong backup request parameter: target backup root directory");
        }
        if (type == BackupType.FULL && (tableList == null || tableList.isEmpty())) {
            List htds = null;
            try (Admin admin = this.conn.getAdmin();){
                htds = admin.listTableDescriptors();
            }
            catch (Exception e) {
                throw new BackupException(e);
            }
            if (htds == null) {
                throw new BackupException("No table exists for full backup of all tables.");
            }
            tableList = new ArrayList<TableName>();
            for (TableDescriptor hTableDescriptor : htds) {
                TableName tn = hTableDescriptor.getTableName();
                if (tn.equals((Object)BackupSystemTable.getTableName(this.conf))) continue;
                tableList.add(hTableDescriptor.getTableName());
            }
            LOG.info("Full backup all the tables available in the cluster: {}", tableList);
        }
        this.backupInfo = new BackupInfo(backupId, type, tableList.toArray(new TableName[tableList.size()]), targetRootDir);
        this.backupInfo.setBandwidth(bandwidth);
        this.backupInfo.setWorkers(workers);
        this.backupInfo.setNoChecksumVerify(noChecksumVerify);
        return this.backupInfo;
    }

    private String getOngoingBackupId() throws IOException {
        ArrayList<BackupInfo> sessions = this.systemTable.getBackupInfos(BackupInfo.BackupState.RUNNING);
        if (sessions.size() == 0) {
            return null;
        }
        return sessions.get(0).getBackupId();
    }

    public void initialize() throws IOException {
        String ongoingBackupId = this.getOngoingBackupId();
        if (ongoingBackupId != null) {
            LOG.info("There is a ongoing backup {}. Can not launch new backup until no ongoing backup remains.", (Object)ongoingBackupId);
            throw new BackupException("There is ongoing backup seesion.");
        }
    }

    public void setBackupInfo(BackupInfo backupInfo) {
        this.backupInfo = backupInfo;
    }

    public void updateBackupInfo(BackupInfo context) throws IOException {
        this.systemTable.updateBackupInfo(context);
    }

    public void startBackupSession() throws IOException {
        long startTime = EnvironmentEdgeManager.currentTime();
        long timeout = (long)this.conf.getInt(BACKUP_EXCLUSIVE_OPERATION_TIMEOUT_SECONDS_KEY, 3600) * 1000L;
        long lastWarningOutputTime = 0L;
        while (EnvironmentEdgeManager.currentTime() - startTime < timeout) {
            try {
                this.systemTable.startBackupExclusiveOperation();
                return;
            }
            catch (IOException e) {
                if (e instanceof ExclusiveOperationException) {
                    try {
                        Thread.sleep(1000L);
                    }
                    catch (InterruptedException e1) {
                        Thread.currentThread().interrupt();
                    }
                    if (lastWarningOutputTime != 0L && EnvironmentEdgeManager.currentTime() - lastWarningOutputTime <= 60000L) continue;
                    lastWarningOutputTime = EnvironmentEdgeManager.currentTime();
                    LOG.warn("Waiting to acquire backup exclusive lock for {}s", (Object)((lastWarningOutputTime - startTime) / 1000L));
                    continue;
                }
                throw e;
            }
        }
        throw new IOException("Failed to acquire backup system table exclusive lock after " + timeout / 1000L + "s");
    }

    public void finishBackupSession() throws IOException {
        this.systemTable.finishBackupExclusiveOperation();
    }

    public String readBackupStartCode() throws IOException {
        return this.systemTable.readBackupStartCode(this.backupInfo.getBackupRootDir());
    }

    public void writeBackupStartCode(Long startCode) throws IOException {
        this.systemTable.writeBackupStartCode(startCode, this.backupInfo.getBackupRootDir());
    }

    public HashMap<String, Long> readRegionServerLastLogRollResult() throws IOException {
        return this.systemTable.readRegionServerLastLogRollResult(this.backupInfo.getBackupRootDir());
    }

    public List<BulkLoad> readBulkloadRows(List<TableName> tableList) throws IOException {
        return this.systemTable.readBulkloadRows(tableList);
    }

    public void deleteBulkLoadedRows(List<byte[]> rows) throws IOException {
        this.systemTable.deleteBulkLoadedRows(rows);
    }

    public List<BackupInfo> getBackupHistory() throws IOException {
        return this.systemTable.getBackupHistory();
    }

    public ArrayList<BackupInfo> getBackupHistory(boolean completed) throws IOException {
        return this.systemTable.getBackupHistory(completed);
    }

    public void writeRegionServerLogTimestamp(Set<TableName> tables, Map<String, Long> newTimestamps) throws IOException {
        this.systemTable.writeRegionServerLogTimestamp(tables, newTimestamps, this.backupInfo.getBackupRootDir());
    }

    public Map<TableName, Map<String, Long>> readLogTimestampMap() throws IOException {
        return this.systemTable.readLogTimestampMap(this.backupInfo.getBackupRootDir());
    }

    public Set<TableName> getIncrementalBackupTableSet() throws IOException {
        return this.systemTable.getIncrementalBackupTableSet(this.backupInfo.getBackupRootDir());
    }

    public void addIncrementalBackupTableSet(Set<TableName> tables) throws IOException {
        this.systemTable.addIncrementalBackupTableSet(tables, this.backupInfo.getBackupRootDir());
    }

    public Connection getConnection() {
        return this.conn;
    }
}

