/*
 * Decompiled with CFR 0.152.
 */
package org.apache.commons.statistics.descriptive;

import java.util.Arrays;
import java.util.Objects;
import java.util.function.IntToDoubleFunction;
import org.apache.commons.numbers.arrays.Selection;
import org.apache.commons.statistics.descriptive.Interpolation;
import org.apache.commons.statistics.descriptive.NaNPolicy;
import org.apache.commons.statistics.descriptive.NaNTransformer;
import org.apache.commons.statistics.descriptive.NaNTransformers;
import org.apache.commons.statistics.descriptive.Statistics;

public final class Quantile {
    private static final String INVALID_PROBABILITY = "Invalid probability: ";
    private static final String NO_PROBABILITIES_SPECIFIED = "No probabilities specified";
    private static final String INVALID_SIZE = "Invalid size: ";
    private static final String INVALID_NUMBER_OF_PROBABILITIES = "Invalid number of probabilities: ";
    private static final Quantile DEFAULT = new Quantile(false, NaNPolicy.INCLUDE, EstimationMethod.HF8);
    private final boolean copy;
    private final NaNPolicy nanPolicy;
    private final NaNTransformer nanTransformer;
    private final EstimationMethod estimationType;

    private Quantile(boolean copy, NaNPolicy nanPolicy, EstimationMethod estimationType) {
        this.copy = copy;
        this.nanPolicy = nanPolicy;
        this.estimationType = estimationType;
        this.nanTransformer = NaNTransformers.createNaNTransformer(nanPolicy, copy);
    }

    public static Quantile withDefaults() {
        return DEFAULT;
    }

    public Quantile withCopy(boolean v) {
        return new Quantile(v, this.nanPolicy, this.estimationType);
    }

    public Quantile with(NaNPolicy v) {
        return new Quantile(this.copy, Objects.requireNonNull(v), this.estimationType);
    }

    public Quantile with(EstimationMethod v) {
        return new Quantile(this.copy, this.nanPolicy, Objects.requireNonNull(v));
    }

    public static double[] probabilities(int n) {
        Quantile.checkNumberOfProbabilities(n);
        double c1 = (double)n + 1.0;
        double[] p = new double[n];
        for (int i = 0; i < n; ++i) {
            p[i] = ((double)i + 1.0) / c1;
        }
        return p;
    }

    public static double[] probabilities(int n, double p1, double p2) {
        Quantile.checkProbability(p1);
        Quantile.checkProbability(p2);
        if (p2 <= p1) {
            throw new IllegalArgumentException("Invalid range: [" + p1 + ", " + p2 + "]");
        }
        double[] p = Quantile.probabilities(n);
        for (int i = 0; i < n; ++i) {
            p[i] = (1.0 - p[i]) * p1 + p[i] * p2;
        }
        return p;
    }

    public double evaluate(double[] values, double p) {
        return this.compute(values, 0, values.length, p);
    }

    public double evaluateRange(double[] values, int from, int to, double p) {
        Statistics.checkFromToIndex(from, to, values.length);
        return this.compute(values, from, to, p);
    }

    private double compute(double[] values, int from, int to, double p) {
        Quantile.checkProbability(p);
        int[] bounds = new int[2];
        double[] x = this.nanTransformer.apply(values, from, to, bounds);
        int start = bounds[0];
        int end = bounds[1];
        int n = end - start;
        if (n <= 1) {
            return n == 0 ? Double.NaN : x[start];
        }
        double pos = this.estimationType.index(p, n);
        int ip = (int)pos;
        int i = start + ip;
        if (pos > (double)ip) {
            Selection.select((double[])x, (int)start, (int)end, (int[])new int[]{i, i + 1});
            return Interpolation.interpolate(x[i], x[i + 1], pos - (double)ip);
        }
        Selection.select((double[])x, (int)start, (int)end, (int)i);
        return x[i];
    }

    public double[] evaluate(double[] values, double ... p) {
        return this.compute(values, 0, values.length, p);
    }

    public double[] evaluateRange(double[] values, int from, int to, double ... p) {
        Statistics.checkFromToIndex(from, to, values.length);
        return this.compute(values, from, to, p);
    }

    private double[] compute(double[] values, int from, int to, double ... p) {
        Quantile.checkProbabilities(p);
        int[] bounds = new int[2];
        double[] x = this.nanTransformer.apply(values, from, to, bounds);
        int start = bounds[0];
        int end = bounds[1];
        int n = end - start;
        double[] q = new double[p.length];
        if (n <= 1) {
            Arrays.fill(q, n == 0 ? Double.NaN : x[start]);
            return q;
        }
        int[] indices = this.computeIndices(n, p, q, start);
        Selection.select((double[])x, (int)start, (int)end, (int[])indices);
        for (int k = 0; k < p.length; ++k) {
            int ip = (int)q[k];
            int i = start + ip;
            q[k] = q[k] > (double)ip ? Interpolation.interpolate(x[i], x[i + 1], q[k] - (double)ip) : x[i];
        }
        return q;
    }

    public double evaluate(int[] values, double p) {
        return this.compute(values, 0, values.length, p);
    }

    public double evaluateRange(int[] values, int from, int to, double p) {
        Statistics.checkFromToIndex(from, to, values.length);
        return this.compute(values, from, to, p);
    }

    private double compute(int[] values, int from, int to, double p) {
        int end;
        int start;
        int[] x;
        Quantile.checkProbability(p);
        int n = to - from;
        if (n <= 1) {
            return n == 0 ? Double.NaN : (double)values[from];
        }
        if (this.copy) {
            x = Statistics.copy(values, from, to);
            start = 0;
            end = n;
        } else {
            x = values;
            start = from;
            end = to;
        }
        double pos = this.estimationType.index(p, n);
        int ip = (int)pos;
        int i = start + ip;
        if (pos > (double)ip) {
            Selection.select((int[])x, (int)start, (int)end, (int[])new int[]{i, i + 1});
            return Interpolation.interpolate(x[i], x[i + 1], pos - (double)ip);
        }
        Selection.select((int[])x, (int)start, (int)end, (int)i);
        return x[i];
    }

    public double[] evaluate(int[] values, double ... p) {
        return this.compute(values, 0, values.length, p);
    }

    public double[] evaluateRange(int[] values, int from, int to, double ... p) {
        Statistics.checkFromToIndex(from, to, values.length);
        return this.compute(values, from, to, p);
    }

    private double[] compute(int[] values, int from, int to, double ... p) {
        int end;
        int start;
        int[] x;
        Quantile.checkProbabilities(p);
        int n = to - from;
        double[] q = new double[p.length];
        if (n <= 1) {
            Arrays.fill(q, n == 0 ? Double.NaN : (double)values[from]);
            return q;
        }
        if (this.copy) {
            x = Statistics.copy(values, from, to);
            start = 0;
            end = n;
        } else {
            x = values;
            start = from;
            end = to;
        }
        int[] indices = this.computeIndices(n, p, q, start);
        Selection.select((int[])x, (int)start, (int)end, (int[])indices);
        for (int k = 0; k < p.length; ++k) {
            int ip = (int)q[k];
            int i = start + ip;
            q[k] = q[k] > (double)ip ? Interpolation.interpolate(x[i], x[i + 1], q[k] - (double)ip) : (double)x[i];
        }
        return q;
    }

    public double evaluate(int n, IntToDoubleFunction values, double p) {
        Quantile.checkSize(n);
        Quantile.checkProbability(p);
        if (n <= 1) {
            return n == 0 ? Double.NaN : values.applyAsDouble(0);
        }
        double pos = this.estimationType.index(p, n);
        int i = (int)pos;
        double v1 = values.applyAsDouble(i);
        if (pos > (double)i) {
            double v2 = values.applyAsDouble(i + 1);
            return Interpolation.interpolate(v1, v2, pos - (double)i);
        }
        return v1;
    }

    public double[] evaluate(int n, IntToDoubleFunction values, double ... p) {
        Quantile.checkSize(n);
        Quantile.checkProbabilities(p);
        double[] q = new double[p.length];
        if (n <= 1) {
            Arrays.fill(q, n == 0 ? Double.NaN : values.applyAsDouble(0));
            return q;
        }
        for (int k = 0; k < p.length; ++k) {
            double pos = this.estimationType.index(p[k], n);
            int i = (int)pos;
            double v1 = values.applyAsDouble(i);
            if (pos > (double)i) {
                double v2 = values.applyAsDouble(i + 1);
                q[k] = Interpolation.interpolate(v1, v2, pos - (double)i);
                continue;
            }
            q[k] = v1;
        }
        return q;
    }

    private static void checkProbability(double p) {
        if (!(p >= 0.0) || !(p <= 1.0)) {
            throw new IllegalArgumentException(INVALID_PROBABILITY + p);
        }
    }

    private static void checkProbabilities(double ... p) {
        if (p.length == 0) {
            throw new IllegalArgumentException(NO_PROBABILITIES_SPECIFIED);
        }
        for (double pp : p) {
            Quantile.checkProbability(pp);
        }
    }

    private static void checkSize(int n) {
        if (n < 0) {
            throw new IllegalArgumentException(INVALID_SIZE + n);
        }
    }

    private static void checkNumberOfProbabilities(int n) {
        if (n < 1) {
            throw new IllegalArgumentException(INVALID_NUMBER_OF_PROBABILITIES + n);
        }
    }

    private int[] computeIndices(int n, double[] p, double[] q, int offset) {
        int[] indices = new int[p.length << 1];
        int count = 0;
        for (int k = 0; k < p.length; ++k) {
            double pos;
            q[k] = pos = this.estimationType.index(p[k], n);
            int i = (int)pos;
            indices[count++] = offset + i;
            if (!(pos > (double)i)) continue;
            indices[count++] = offset + i + 1;
        }
        if (count < indices.length) {
            return Arrays.copyOf(indices, count);
        }
        return indices;
    }

    public static enum EstimationMethod {
        HF1{

            @Override
            double position0(double p, int n) {
                return Math.ceil((double)n * p) - 1.0;
            }
        }
        ,
        HF2{

            @Override
            double position0(double p, int n) {
                double pos = (double)n * p;
                int j = (int)pos;
                double g = pos - (double)j;
                if (g == 0.0) {
                    return (double)j - 0.5;
                }
                return j;
            }
        }
        ,
        HF3{

            @Override
            double position0(double p, int n) {
                return Math.rint((double)n * p) - 1.0;
            }
        }
        ,
        HF4{

            @Override
            double position0(double p, int n) {
                return (double)n * p - 1.0;
            }
        }
        ,
        HF5{

            @Override
            double position0(double p, int n) {
                return (double)n * p - 0.5;
            }
        }
        ,
        HF6{

            @Override
            double position0(double p, int n) {
                return (double)(n + 1) * p - 1.0;
            }
        }
        ,
        HF7{

            @Override
            double position0(double p, int n) {
                return (double)(n - 1) * p;
            }
        }
        ,
        HF8{

            @Override
            double position0(double p, int n) {
                return (double)n * p + (p + 1.0) / 3.0 - 1.0;
            }
        }
        ,
        HF9{

            @Override
            double position0(double p, int n) {
                return ((double)n + 0.25) * p - 0.625;
            }
        };


        abstract double position0(double var1, int var3);

        final double index(double p, int n) {
            double pos = this.position0(p, n);
            if (pos < 0.0) {
                return 0.0;
            }
            if (pos > (double)n - 1.0) {
                return (double)n - 1.0;
            }
            return pos;
        }
    }
}

