/*$$
 * packages uchicago.src.*
 * Copyright (c) 1999, Trustees of the University of Chicago
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with 
 * or without modification, are permitted provided that the following 
 * conditions are met:
 *
 *	 Redistributions of source code must retain the above copyright notice,
 *	 this list of conditions and the following disclaimer.
 *
 *	 Redistributions in binary form must reproduce the above copyright notice,
 *	 this list of conditions and the following disclaimer in the documentation
 *	 and/or other materials provided with the distribution.
 *
 *	 Neither the name of the University of Chicago nor the names of its
 *   contributors may be used to endorse or promote products derived from
 *   this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 * ``AS IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A
 * PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE TRUSTEES OR
 * CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
 * PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
 * LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
 * NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE,
 * EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * Nick Collier
 * nick@src.uchicago.edu
 *
 * packages cern.jet.random.*
 * Copyright (c) 1999 CERN - European Laboratory for Particle
 * Physics. Permission to use, copy, modify, distribute and sell this
 * software and its documentation for any purpose is hereby granted without
 * fee, provided that the above copyright notice appear in all copies
 * and that both that copyright notice and this permission notice appear in
 * supporting documentation. CERN makes no representations about the 
 * suitability of this software for any purpose. It is provided "as is" 
 * without expressed or implied warranty. 
 *
 * Wolfgang Hoschek
 * wolfgang.hoschek@cern.ch
 *$$*/


package collection;

import java.util.*;
import java.io.Serializable;


/**
 * A matrix (lattice) container. Expands to accomodate new elements
 *
 * @version 1.01 2/25/99
 * @author Nick Collier
 */

public class Matrix implements BaseMatrix, List, Cloneable, Serializable {

  ArrayList matrix;
  int num_rows;
  int num_cols;
  Object null_obj = null;
  static final int DEFAULT_SIZE = 10;

  public Matrix() {
    this(DEFAULT_SIZE, DEFAULT_SIZE);
  }

  /**
   * Constructs a matrix whose dimensions are rows by cols.
   * @param rows Number of rows in the matrix.
   * @param cols Number of cols in the matrix.
   */
  public Matrix(int cols, int rows) {
    int size = rows * cols;
    matrix = new ArrayList(size);
    for (int i = 0; i < size; i++) {
      matrix.add(null);
    }
    num_rows = rows;
    num_cols = cols;
  }

  /**
   * Constructs a matrix whose dimensions are rows by cols, using list
   * as the collection object.
   * @param rows Number of rows in the matrix.
   * @param cols Number of cols in the matrix.
   * @param list ArrayList to use as the collection object.
   */

  public Matrix(int cols, int rows, ArrayList list ) {
    if (list.size() != rows * cols)
      throw new IllegalArgumentException("Matrix Constructor: illegal list size");
    matrix = list;
    num_rows = rows;
    num_cols = cols;
  }


  /**
   * Gets the object at the cell intersected by row and col.
   * @param row The row dimension
   * @param col The column dimension.
   * @return The object at col, row.
   */
  public Object at(int col, int row) {
    //System.out.println("Matrix get: " + row + " " + col);
    checkIndex(row, col, num_rows, num_cols);
    return matrix.get(row * num_cols + col);
  }

  public Object get(int col, int row) {
    return at(col, row);
  }

   /**
   * Put an object into the matrix, replacing the current object
   * @param index where to put the object
   * @param value the object to put into the matrix
   */

  public void put(int index, Object value) {
    //checkIndex(index, num_rows, num_cols);
    matrix.set(index, value);
  }

  /**
   * Put an object into the matrix, replacing the current object
   * @param row the row to put the object
   * @param col the col to put the object
   * @param value the object to put into the matrix
   */
  public void put(int col, int row, Object value) {
    checkIndex(row, col, num_rows, num_cols);
    matrix.set(row * num_cols + col, value);
  }

  /**
   * Returns the object at the specified coordinates, setting the value of
   * those coordinates to null.
   * @param row the row coordinate.
   * @param col the column coordinate.
   * @return the object removed.
   */
  public Object remove(int col, int row) {
    checkIndex(row, col, num_rows, num_cols);
    return matrix.set(row * num_cols + col, null);
  }

  public int getNumRows() {
    return num_rows;
  }

  public int getNumCols() {
    return num_cols;
  }

  public int size() {
    return num_cols * num_rows;
  }

  public synchronized MatrixIterator begin() {
    return new MatrixIterator(this, 0);
  }

  public synchronized MatrixIterator end() {
    return new MatrixIterator(this, num_rows * num_cols);
  }

  public MatrixIterator matrixIterator(int col, int row) {
    checkIndex(row, col, num_rows, num_cols);
    return new MatrixIterator(this, row * num_cols + col);
  }

  /**
   * Returns a RowSlice for row n
   * @parameter n The row to get
   * @return The RowSlice for that row
   * @exception IndexOutOfBoundsException if the specified row is invalid
   */
  public RowSlice getRow(int n) {
    return getRows(n , n);
  }

  /**
   * Get rows start through end.
   * @parameter start The beginning row.
   * @parameter end The ending row.
   * @exception IndexOutOfBoundsException if the rows are invalid
   */
  public RowSlice getRows(int start, int end) {
    if (end < start)
      throw new IndexOutOfBoundsException("End col greater than start col");
    if (start != end) {
      checkIndex(start, num_cols - 1, num_cols, num_cols);
      checkIndex(end, num_cols - 1, num_cols, num_cols);
    } else {
      checkIndex(start, num_cols - 1, num_cols, num_cols);
    }
    return new RowSlice(this, start, end);
  }

  /**
   * Returns a ColSlice for col n
   * @parameter n The column to get
   * @return The ColSlice for that column
   * @exception IndexOutOfBoundsException if the specified column is invalid
   */
  public ColSlice getCol(int n) {
    return getCols(n , n);
  }

  /**
   * Get cols start through end.
   * @parameter start The beginning col.
   * @parameter end The ending col.
   * @exception IndexOutOfBoundsException if the cols are invalid
   */
  public ColSlice getCols(int start, int end) {
    if (end < start)
      throw new IllegalArgumentException("End col greater than start col");
    if (start != end) {
      checkIndex(num_rows - 1, start, num_rows, num_cols);
      checkIndex(num_rows -1, end, num_rows, num_cols);
    } else {
      checkIndex(num_rows - 1, start, num_rows, num_cols);
    }
    return new ColSlice(this, start, end);
  }


  public Enumeration elements() {
    return new MatrixIterator(this, 0);
  }

  // List interface.

  /**
   * Add an object to the matrix. Creates new row, and adds object to
   * column 0 of that row, filling the remaining columns with
   * null.
   * @param object The object to add
   * @return true as per the contract in java.util.Collection
   */

  public boolean add(Object object) {
    matrix.add(object);
    num_rows++;
    for (int i = 1; i < num_cols; i++) {
      matrix.add(null);
    }
    return true;
  }

  /**
   * Adds an object to the matrix at the specified index.
   *
   */
  public void add(int index, Object element) {
    throw new UnsupportedOperationException("Operation Unsupported");
  }

  public boolean addAll(Collection c) {
    throw new UnsupportedOperationException("Operation Unsupported");
  }

  public boolean addAll(int index, Collection c) {
    throw new UnsupportedOperationException("Operation Unsupported");
  }

  /**
   * Clears the matrix, setting it to its default size, and filling it with
   * null.
   */
  public void clear() {
    matrix.clear();
    matrix.ensureCapacity(DEFAULT_SIZE * DEFAULT_SIZE);
    for (int i = 0; i < DEFAULT_SIZE * DEFAULT_SIZE; i++) {
      matrix.add(null);
    }
  }

  /**
   * Returns true if the matrix contains the specified object.
   * @param o object whose presence in this matrix is to be tested.
   */
  public boolean contains(Object o) {
    return matrix.contains(o);
  }

  /**
   * Returns true if this matrix contains all of the elements of the
   * specified collection.
   * @param c collection to be checked for containment in this matrix.
   */
  public boolean containsAll(Collection c) {
    return matrix.containsAll(c);
  }

  /**
   * Compares the specified object with this matrix for equality. Returns true
   * if and only if the specified object is also a matrix, both
   * matrices have the same size, and all corresponding pairs of elements in
   * the two matrices are equal.
   * @param o the object to compare with this matrix.
   */
  public boolean equals(Object o) {
    if (o instanceof Matrix) {
      Matrix otherMatrix = (Matrix)o;
      return matrix.equals(otherMatrix.matrix);
    }
    return false;
  }

  /**
   * Gets the object at the specified index. Index = row * col.
   *
   *@param index the index of the object to get.
   *@return the object at the specified index.
   */
  public Object get(int index) {
    return matrix.get(index);
  }

  public int hashCode() {
    return matrix.hashCode();
  }

  /**
   * Returns the first index of the specified object.
   * @param o the object whose index to get.
   * @return the first index of the specified object or -1 if object is not found.
   */
  public int indexOf(Object o) {
    return matrix.indexOf(o);
  }

  /**
   * Returns true if this matrix contains only null elements
   */
  public boolean isEmpty() {
    ListIterator li = matrix.listIterator();
    while (li.hasNext()) {
      if (li.next() != null)
        return false;
    }
    return true;
  }

  /**
   * Return an iterator over this matrix. Iterates by row.
   */
  public Iterator iterator() {
    return begin();
  }

  /**
   * Returns the index in this list of the last occurrence of the specified
   * element, or -1 if this matrix does not contain this element.
   * @param o The object to find the last index of
   * @return The last index of the object or -1 if object is not found.
   */
  public int lastIndexOf(Object o) {
    return matrix.lastIndexOf(o);
  }


  /**
   * Return a ListIterator over this matrix. Iterates by row.
   */
  public ListIterator listIterator() {
    return begin();
  }

  /**
   * Returns a list iterator of the elements in this matrix, starting at the
   * specified position in this matrix. The specified index indicates the
   * first element that would be returned by an initial call to the nextElement
   * method. An initial call to the previousElement method would return the
   * element with the specified index minus one.
   * @param index index of first element to be returned from the
   *   list iterator (by a call to the next method).
   * @return a list iterator of the elements in this matrix,
   * starting at the specified position in this list.
   */
  public ListIterator listIterator(int index) {
    return new MatrixIterator(this, index);
  }

  /**
   * Remove the element at a particular position, and set the value
   * of that cell to null.
   * @param index the index of the element to be removed.
   * @return The object removed.
   * @exception IllegalArgumentException if the Enumeration isn't a
   * MatrixIterator for this Matrix object.
   */

  public Object remove(int index) {
    return matrix.set(index, null);
  }

  /**
   * Removes the first occurrence in this matrix of the specified element.
   * Setting the value of the removed object index to null.
   * @param element the object to be removed.
   * @return true if the list contained the specified element.
   */
  public boolean remove(Object element) {
    int index = matrix.indexOf(element);
    if (index == -1)
      return false;
    matrix.set(index, null);
    return true;
  }


  public boolean removeAll(Collection c) {
    throw new UnsupportedOperationException("Operation Unsupported");
  }

  public boolean retainAll(Collection c) {
    throw new UnsupportedOperationException("Operation Unsupported");
  }


  public Object set(int index, Object element) {
    return matrix.set(index, element);
  }

  public Object[] toArray() {
    return matrix.toArray();
  }

  public Object[] toArray(Object[] array) {
    return matrix.toArray(array);
  }

  public List subList(int fromIndex, int toIndex) {
    return matrix.subList(fromIndex, toIndex);
  }

  public Object clone() {
    try {
      return super.clone();
    } catch (Exception e) {}
    return null;
  }

  public void print() {
    String value = "";
    for (int i = 0; i < num_rows; i++) {
      for (int j = 0; j < num_cols; j++) {
        Object o = get(i, j);
        if (o == null)
          value = "Null";
        else
          value = o.toString();
        System.out.print(value + ",  ");
      }
      System.out.print("\n");
    }
  }

  public void printToFile(String fileName) {
    try {
      java.io.BufferedWriter out = new java.io.BufferedWriter(new java.io.FileWriter(fileName));

      String value = "";
      for (int i = 0; i < num_rows; i++) {
        for (int j = 0; j < num_cols; j++) {
          Object o = get(i, j);
          if (o == null)
            value = "Null";
          else
            value = o.toString();
          out.write(value + " ");
        }
        out.write("\n");
      }
      out.flush();
      out.close();
    } catch (java.io.IOException ex) {
      ex.printStackTrace();
    }
  }

  final protected static void checkIndex(int row, int col, int nrows, int ncols) {
    if (row >= nrows || row < 0|| col >= ncols || col < 0) {
      throw new IndexOutOfBoundsException ("Attempt to access row " + row +
        ", col " + col + ". Valid range is (0..." + (nrows - 1) + ", 0..." +
        (ncols - 1) + ").");
    }
  }

  final protected static void checkIndex(int index, int nrows, int ncols) {
    int length = nrows * ncols;
    if (index < 0 || index >= length) {
      int row;
      int col;
      if (index < ncols) {
        row = 0;
        col = index;
      } else {
        col = index % ncols;
        row = (index - col) / ncols;
      }
      throw new IndexOutOfBoundsException ("Attempt to access row " + row +
        ", col " + col + ". Valid range is (0..." + (nrows - 1) + ", 0..." +
        (ncols - 1) + ").");
    }
  }
}
