/*$$
 * packages uchicago.src.*
 * Copyright (c) 1999, Trustees of the University of Chicago
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with 
 * or without modification, are permitted provided that the following 
 * conditions are met:
 *
 *	 Redistributions of source code must retain the above copyright notice,
 *	 this list of conditions and the following disclaimer.
 *
 *	 Redistributions in binary form must reproduce the above copyright notice,
 *	 this list of conditions and the following disclaimer in the documentation
 *	 and/or other materials provided with the distribution.
 *
 *	 Neither the name of the University of Chicago nor the names of its
 *   contributors may be used to endorse or promote products derived from
 *   this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 * ``AS IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A
 * PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE TRUSTEES OR
 * CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
 * PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
 * LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
 * NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE,
 * EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * Nick Collier
 * nick@src.uchicago.edu
 *
 * packages cern.jet.random.*
 * Copyright (c) 1999 CERN - European Laboratory for Particle
 * Physics. Permission to use, copy, modify, distribute and sell this
 * software and its documentation for any purpose is hereby granted without
 * fee, provided that the above copyright notice appear in all copies
 * and that both that copyright notice and this permission notice appear in
 * supporting documentation. CERN makes no representations about the 
 * suitability of this software for any purpose. It is provided "as is" 
 * without expressed or implied warranty. 
 *
 * Wolfgang Hoschek
 * wolfgang.hoschek@cern.ch
 *$$*/
package collection;

/**
 * DoubleMatrix: a matrix of doubles (and any other number)
 *
 * @version 1.0 6/29/99
 * @author Nick Collier
 */

public class DoubleMatrix implements BaseMatrix {

  private double[][] matrix;
  private int sizeX;
  private int sizeY;

  public DoubleMatrix(int sizeX, int sizeY) {
    matrix = new double[sizeX][sizeY];
    this.sizeX = sizeX;
    this.sizeY = sizeY;
    initialize(0);
  }

  public DoubleMatrix(double matrix[][], int sizeX, int sizeY) {
    this.matrix = matrix;
    this.sizeX = sizeX;
    this.sizeY = sizeY;
  }

  public double getDoubleAt(int x, int y) {
    return matrix[x][y];
  }

  public void putDoubleAt(int x, int y, double value) {
    matrix[x][y] = value;
  }

  public DoubleMatrix copyMatrix() {
    double[][] newMatrix = new double[sizeX][sizeY];
    System.arraycopy(matrix, 0, newMatrix, 0, sizeX);
    DoubleMatrix dm = new DoubleMatrix(sizeX, sizeY);
    dm.matrix = newMatrix;
    return dm;
  }

  public int getSizeX() {
    return sizeX;
  }

  public int getSizeY() {
    return sizeY;
  }

  /**
   * Fills the matrix with the passed in value
   *
   * @param value the value to fill the matrix with.
   */
  public void initialize(double value) {
    for (int i = 0; i < sizeX; i++) {
      for (int j = 0; j < sizeY; j++) {
        matrix[i][j] = value;
      }
    }
  }

  public Object get(int index) {
    int y = (int)Math.floor(index / sizeY);
    if (y != 0)
      y -= 1;
    int x = index % sizeY;
    return new Double(matrix[x][y]);
  }

  public Object get(int col, int row) {
    return new Double(matrix[col][row]);
  }

  public void put(int index, Object obj) {
    if (!(obj instanceof Number)) 
      throw new IllegalArgumentException("object must be a Number");

    int y = (int)Math.floor(index / sizeY);
    if (y != 0)
      y -= 1;
    int x = index % sizeY;
    matrix[x][y] = ((Number)obj).doubleValue();
  }

  public void put(int col, int row, Object obj) {
    if (!(obj instanceof Number)) 
      throw new IllegalArgumentException("object must be a Number");
    matrix[col][row] = ((Number)obj).doubleValue();
  }

  public Object remove(int col, int row) {
    Double d = new Double(matrix[col][row]);
    matrix[col][row] = 0;
    return d;
  }

  public Object remove(int index) {
    int y = (int)Math.floor(index / sizeY);
    if (y != 0)
      y -= 1;
    int x = index % sizeY;

    return remove(x, y);
  }

  public int size() {
    return sizeX * sizeY;
  }

  public int getNumRows() {
    return sizeY;
  }

  public int getNumCols() {
    return sizeX;
  }

   public void printToFile(String fileName) {
    try {
      java.io.BufferedWriter out = new java.io.BufferedWriter(new java.io.FileWriter(fileName));

      String value = "";
      for (int i = 0; i < sizeX; i++) {
        for (int j = 0; j < sizeY; j++) {
          out.write(this.getDoubleAt(i, j) + " ");
        }
        out.write("\n");
      }
      out.flush();
      out.close();
    } catch (java.io.IOException ex) {
      ex.printStackTrace();
    }
  }

}
