/*$$
 * packages uchicago.src.*
 * Copyright (c) 1999, Trustees of the University of Chicago
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with 
 * or without modification, are permitted provided that the following 
 * conditions are met:
 *
 *	 Redistributions of source code must retain the above copyright notice,
 *	 this list of conditions and the following disclaimer.
 *
 *	 Redistributions in binary form must reproduce the above copyright notice,
 *	 this list of conditions and the following disclaimer in the documentation
 *	 and/or other materials provided with the distribution.
 *
 *	 Neither the name of the University of Chicago nor the names of its
 *   contributors may be used to endorse or promote products derived from
 *   this software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
 * ``AS IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
 * LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A
 * PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE TRUSTEES OR
 * CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO,
 * PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
 * PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
 * LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
 * NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE,
 * EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * Nick Collier
 * nick@src.uchicago.edu
 *
 * packages cern.jet.random.*
 * Copyright (c) 1999 CERN - European Laboratory for Particle
 * Physics. Permission to use, copy, modify, distribute and sell this
 * software and its documentation for any purpose is hereby granted without
 * fee, provided that the above copyright notice appear in all copies
 * and that both that copyright notice and this permission notice appear in
 * supporting documentation. CERN makes no representations about the 
 * suitability of this software for any purpose. It is provided "as is" 
 * without expressed or implied warranty. 
 *
 * Wolfgang Hoschek
 * wolfgang.hoschek@cern.ch
 *$$*/
package gui;

import java.awt.Graphics;
import java.awt.Graphics2D;
import java.awt.image.BufferedImage;
import java.awt.RenderingHints;
import java.awt.Image;
import java.util.Vector;
import java.awt.Toolkit;
import java.io.DataOutputStream;
import java.awt.Color;
/**
 * Abstract class for local and remote painting objects that do the actual
 * work of getting Displayables to paint themselves. Used by DisplaySurface.
 *
 * @author Nick Collier
 * @version $Revision: 1.1.1.1 $ $Date: 2000/03/10 03:16:26 $
 * @see LocalPainter
 */

public abstract class Painter {

  protected DisplaySurface surface;
  protected BufferedImage buffImage = null;
  protected Graphics2D g2 = null;
  protected int width, height;
  protected Vector displayables = new Vector();
  protected SimGraphics simGraphics = new SimGraphics();
  protected Toolkit toolkit;
  protected Color bgColor = Color.black;

  /**
   * Creates a Painter using the specified displaySurface, and with
   * the specified width and height.
   *
   * @param s the displaySurface associated with this LocalPainter
   * @param w the width of the painter
   * @param h the height of the painter
   */
  public Painter(DisplaySurface s, int w, int h) {
    surface = s;
    width = w;
    height = h;
    toolkit = s.getToolkit();
  }

  /**
   * Creates a BufferedImage for use by sub class painters.
   */
  protected void createBufferedImage() {
    buffImage = (BufferedImage)surface.createImage(width, height);
  }

  /**
   * Creates and sets the drawing context.
   */
  protected void createGraphics2D() {
    g2 = buffImage.createGraphics();
    g2.setRenderingHint(RenderingHints.KEY_RENDERING,
                        RenderingHints.VALUE_RENDER_SPEED);
    //g2.setRenderingHint(RenderingHints.KEY_ANTIALIASING,
    //                         RenderingHints.VALUE_ANTIALIAS_ON);
  }

  /**
   * Adds a Displayable to the list of displayables to be displayed.
   *
   * @param d the displayable to add
   */
  public void addDisplayable(Displayable d) {
    displayables.addElement(d);
  }

  /**
   * Removes a Displayable to the list of displayables to be displayed.
   *
   * @param d the displayable to remove
   */
  public void removeDisplayable(Displayable d) {
    displayables.remove(d);
  }

  /**
   * Gets the graphics context on which to draw.
   */
  public Graphics2D getGraphics() {
    return g2;
  }

  /**
   * Paints the displayables.
   */
  public abstract void paint(Graphics g);


  public void finalize() {
    if (g2 != null)
      g2.dispose();
  }

  /**
   * Disposes of the graphics object
   */
  public void dispose() {
    if (g2 != null)
      g2.dispose();
  }

  /**
   * Sets the background color.
   */
  public void setBackgroundColor(Color c) {
    bgColor = c;
  }

  /**
   * Paints the background to the currently specified color.
   */
  protected void paintBackground() {
    g2.setColor(bgColor);
    g2.fillRect(0, 0, width, height);
  }

  /**
   * Takes a snapshot of the current screen image.
   */
  public abstract void takeSnapshot(DataOutputStream os);

  /**
   * Gets the current BufferedImage
   */
  public Image getCurrentImage() {
    return buffImage;
  }

}
