#
# $Id: udpcast.py 86 2009-05-02 19:02:20Z lxp $
#
# This file is part of OpenClone.
#
# Copyright (C) 2009  David Gnedt
#
# OpenClone is free software: you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation, either version 3 of the License, or
# (at your option) any later version.
#
# OpenClone is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with OpenClone.  If not, see <http://www.gnu.org/licenses/>.
#

from __future__ import with_statement
import logging
import re
import subprocess
import threading

from .. import utils

logger = logging.getLogger('udpcast')

# bytes=  1 628 854K re-xmits=1095627 ( 95.6%) slice=0112 -   0
# Transfer complete.
# Disconnecting #0 (10.10.10.254)

class UDPcastSender(threading.Thread):
    interface = None
    port = None
    file = None
    client_count = None
    proc = None
    status = None
    status_lock = None
    id = None
    manager = None
    
    def __init__(self, **kw):
        threading.Thread.__init__(self)
        
        if 'id' in kw:
            self.id = kw['id']
        
        if 'manager' in kw:
            self.manager = kw['manager']
        
        # TODO: Get path (configuration?)
        args = ['/usr/sbin/udp-sender']
        
        if 'client_count' in kw:
            self.client_count = kw['client_count']
        
        else:
            self.client_count = 1
        
        args.append('--nokbd')
        args.append('--min-receivers')
        args.append(str(self.client_count))
        
        # TODO: Configuration option for full duplex mode
        args.append('--full-duplex')
        #args.append('--half-duplex')
        
        # TODO: Add --max-bitrate option
        
        if 'interface' in kw:
            self.interface = kw['interface']
            args.append('--interface')
            args.append(self.interface)
        
        if 'port' in kw:
            self.port = kw['port']
            args.append('--portbase')
            args.append(str(self.port))
        
        else:
            self.port = 9000
        
        if 'file' in kw:
            if 'input' in kw:
                raise Exception('Unexpected mix of file and input keyword')
            
            kw['input'] = None
            self.file = kw['file']
            args.append('--file')
            args.append(self.file)
        
        elif 'input' not in kw:
            kw['input'] = subprocess.PIPE
        
        logger.debug('Exec: %s' % ' '.join(args))
        self.proc = subprocess.Popen(args, stdin=kw['input'], stderr=subprocess.PIPE, close_fds=True)
        
        self.status = {}
        self.status_lock = threading.RLock()
        
        self.start()
    
    def run(self):
        success = False
        timeouts = 0
        with self.status_lock:
            self.status['timeouts'] = timeouts
        
        for line in utils.universal_readline_iter(self.proc.stderr, seekable=False):
            line = line.rstrip('\n\r')
            
            # TODO: Check if udpcast connects to the correct server (change multicast ip?)
            
            if line == '':
                continue
            
            result = re.match('^bytes=(?P<bytes>(?: {0,3}\d{0,3} )* {0,2}\d{1,3}[ KM]) re-xmits=(?P<rexmits>\d{7} \( {0,2}\d{1,3}\.\d%\)) slice=(?P<slice>\d{4}) .*$', line)
            if result is not None:
                res = result.groupdict()
                with self.status_lock:
                    self.status['bytes'] = res['bytes']
                    self.status['rexmits'] = res['rexmits']
                    self.status['slice'] = res['slice']
                
                continue
            
            if line.startswith('Timeout notAnswered='):
                timeouts = timeouts + 1
                with self.status_lock:
                    self.status['timeouts'] = timeouts
                
                continue
            
            if line == 'Transfer complete.\x07':
                success = True
            
            result = re.match('^Starting transfer: \d{8}$', line)
            if result is not None and self.id is not None and self.manager is not None:
                self.manager.removeSender(self.id)
                self.id = None
                self.manager = None
            
            logger.debug('out: %s' % line)
        
        ret = self.proc.wait()
        if ret != 0 and success:
            success = False
        
        if self.id is not None and self.manager is not None:
            self.manager.removeSender(self.id)
            self.id = None
            self.manager = None
        
        with self.status_lock:
            self.status['return'] = ret
            self.status['success'] = success
        
        logger.debug('success %s' % success)
    
    def getStdin(self):
        return self.proc.stdin
    
    def getStatus(self):
        with self.status_lock:
            return dict(self.status)
    
    def wait(self, timeout=None):
        self.join(timeout)
        # Kill process if thread ended and process is still running
        if not self.isAlive() and self.proc.poll() is None:
            logger.warn('UDPcast thread ended without process')
            os.kill(self.proc.pid, signal.SIGTERM)

class UDPcastReceiver(threading.Thread):
    interface = None
    port = None
    file = None
    proc = None
    status = None
    status_lock = None
    id = None
    manager = None
    
    def __init__(self, **kw):
        threading.Thread.__init__(self)
        
        if 'id' in kw:
            self.id = kw['id']
        
        if 'manager' in kw:
            self.manager = kw['manager']
        
        # TODO: Get path (configuration?)
        args = ['/usr/sbin/udp-receiver']
        
        args.append('--nokbd')
        
        if 'interface' in kw:
            self.interface = kw['interface']
            args.append('--interface')
            args.append(self.interface)
        
        if 'port' in kw:
            self.port = kw['port']
            args.append('--portbase')
            args.append(str(self.port))
        
        else:
            self.port = 9000
        
        if 'file' in kw:
            if 'output' in kw:
                raise Exception('Unexpected mix of file and output keyword')
            
            kw['output'] = None
            self.file = kw['file']
            args.append('--file')
            args.append(self.file)
        
        elif 'output' not in kw:
            kw['output'] = subprocess.PIPE
        
        logger.debug('Exec: %s' % ' '.join(args))
        self.proc = subprocess.Popen(args, stdout=kw['output'], stderr=subprocess.PIPE, close_fds=True)
        
        self.status = {}
        self.status_lock = threading.RLock()
        
        self.start()
    
    def run(self):
        success = False
        for line in utils.universal_readline_iter(self.proc.stderr, seekable=False):
            line = line.rstrip('\n\r')
            
            # TODO: Check if udpcast connects to the correct server (change multicast ip?)
            
            result = re.match('^bytes=(?P<bytes>(?: {0,3}\d{0,3} )* {0,2}\d{1,3}[ KM]) \((?P<rate> {0,2}[\d*]{1,3}\.[\d*]{2} Mbps)\)$', line)
            if result is not None:
                res = result.groupdict()
                with self.status_lock:
                    self.status['bytes'] = res['bytes']
                    self.status['rate'] = res['rate']
                
                continue
            
            if line == 'Transfer complete.\x07':
                success = True
            
            result = re.match('^Starting transfer: \d{8}$', line)
            if result is not None and self.id is not None and self.manager is not None:
                self.manager.removeReceiver(self.id)
                self.id = None
                self.manager = None
            
            logger.debug('out: %s' % line)
        
        ret = self.proc.wait()
        if ret != 0 and success:
            success = False
        
        if self.id is not None and self.manager is not None:
            self.manager.removeReceiver(self.id)
            self.id = None
            self.manager = None
        
        with self.status_lock:
            self.status['return'] = ret
            self.status['success'] = success
        
        logger.debug('success %s' % success)
    
    def getStdout(self):
        return self.proc.stdout
    
    def getStatus(self):
        with self.status_lock:
            return dict(self.status)
    
    def wait(self, timeout=None):
        self.join(timeout)
        # Kill process if thread ended and process is still running
        if not self.isAlive() and self.proc.poll() is None:
            logger.warn('UDPcast thread ended without process')
            os.kill(self.proc.pid, signal.SIGTERM)

class UDPcastManager:
    receiver = None
    sender = None
    next_port = None
    lock = None
    
    # TODO: Solve next_port overflow? (Maybe check for unused ports)
    
    def __init__(self):
        self.receiver = {}
        self.sender = {}
        self.next_port = 9000
        self.lock = threading.RLock()
    
    def getSender(self, udpcast_id, file, interface, client_count):
        with self.lock:
            if udpcast_id not in self.sender:
                logger.info('Starting UDPcast in send mode (id: %d, client_count: %d, interface: %s, port: %d, file: %s)' % (udpcast_id, client_count, interface, self.next_port, file))
                sender = self.sender[udpcast_id] = UDPcastSender(port=self.next_port, file=file, interface=interface, client_count=client_count, id=udpcast_id, manager=self)
                self.next_port = self.next_port + 2
            
            else:
                logger.info('Using sending UDPcast id %d' % udpcast_id)
                sender = self.sender[udpcast_id]
        
        return sender
    
    def removeSender(self, udpcast_id):
        with self.lock:
            del self.sender[udpcast_id]
    
    def getReceiver(self, udpcast_id, file, interface):
        with self.lock:
            if udpcast_id not in self.receiver:
                logger.info('Starting UDPcast in receive mode (id: %d, interface: %s, port: %d, file: %s)' % (udpcast_id, interface, self.next_port, file))
                receiver = self.receiver[udpcast_id] = UDPcastReceiver(port=self.next_port, file=file, interface=interface, id=udpcast_id, manager=self)
                self.next_port = self.next_port + 2
            
            else:
                logger.info('Using receiving UDPcast id %d' % udpcast_id)
                receiver = self.receiver[udpcast_id]
        
        return receiver
    
    def removeReceiver(self, udpcast_id):
        with self.lock:
            del self.receiver[udpcast_id]

manager = UDPcastManager()
