/*  GNU Ocrad - Optical Character Recognition program
    Copyright (C) 2003, 2004 Antonio Diaz Diaz.

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
*/

#include <cstdio>
#include <vector>
#include "common.h"
#include "rectangle.h"
#include "ucs.h"
#include "block.h"
#include "blockmap.h"
#include "character.h"
#include "profile.h"
#include "feats.h"


// First attempt at recognition without relying on context.
void Character::recognize1( const Charset & charset, int charbox_vcenter ) throw()
  {
  if( blocks() == 1 ) recognize11( charset, charbox_vcenter );
  else if( blocks() == 2 ) recognize12( charset, charbox_vcenter );
  else if( blocks() == 3 ) recognize13( charset );
  }


// Recognizes 1 block characters.
// 0123456789ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghklmnopqrstuvwxyz
// #$&'()*+,-./<>@[\]^_`{|}~
void Character::recognize11( const Charset & charset, int charbox_vcenter ) throw()
  {
  const Block & b = block( 0 );
  if( b.blocks() == 0 ) recognize110( charset, charbox_vcenter );
  else if( b.blocks() == 1 ) recognize111( charset, charbox_vcenter );
  else if( b.blocks() == 2 ) recognize112();
  }


// Recognizes 1 block characters without holes.
// 12357CEFGHIJKLMNSTUVWXYZcfhklmnrstuvwxyz
// '()*+,-./<>@[\]^_`{|}~
void Character::recognize110( const Charset & charset, int charbox_vcenter ) throw()
  {
  const Block & b = block( 0 );
  Features f( b );
  int code = f.test_easy( charbox_vcenter );
  if( code )
    {
    if( code == '.' && b.width() > b.height() && b.v_includes( charbox_vcenter ) )
      { add_guess( code, 1 ); add_guess( '-', 0 ); return; }
    add_guess( code, 0 ); return;
    }
  if( b.height() < 5 || ( b.height() < 8 && b.width() < 6 ) ||
      ( b.height() > 10 * b.width() ) ||
      ( b.height() * 5 < b.width() ) ) return;

  code = f.test_CEFIJLlT( charset ); if( code ) { add_guess( code, 0 ); return; }
  code = f.test_frst();              if( code ) { add_guess( code, 0 ); return; }
  code = f.test_G();                 if( code ) { add_guess( code, 0 ); return; }
  code = f.test_c();                 if( code ) { add_guess( code, 0 ); return; }
  if( charset.enabled( Charset::iso_8859_9 ) )
    { code = f.test_s_cedilla(); if( code ) { add_guess( code, 0 ); return; } }
  code = f.test_235Esz( charset );   if( code ) { add_guess( code, 0 ); return; }
  code = f.test_HKMNUuvwYy();
  if( code == 'u' && f.lp.isvpit() )	// Looks for merged 'tr'
    {
    int col = b.seek_left( b.vcenter(), b.right() );
    if( col < b.hpos( 90 ) && !b.escape_top( b.vcenter(), col ) )
      {
      col = b.seek_left( b.vcenter(), col - 1, false );
      while( --col > b.hpos( 40 ) &&
             ( b.seek_top( b.vcenter(), col ) > b.top() ||
               f.hp[col-b.left()] > b.height() / 10 ) );
      if( col > b.hpos( 40 ) )
        { only_guess( 0, b.left() ); add_guess( 't', col );
        add_guess( 'r', b.right() ); return; }
      }
    }
  if( code ) { add_guess( code, 0 ); return; }
  if( f.bp.minima( b.height() / 8 ) == 2 )
    {
    code = f.test_hknwx();
    if( code == 'n' )	// Looks for '"' or merged 'rr' or 'rt' or 'fl'
      {
      if( b.bottom() <= charbox_vcenter ) { add_guess( '"', 0 ); return; }
      if( b.width() > b.height() )
        {
        int col = b.seek_left( b.vcenter(), b.right() );
        if( col < b.hpos( 90 ) && !b.escape_top( b.vcenter(), col ) )
          {
          only_guess( 0, b.left() ); add_guess( 'r', b.hcenter() );
          if( f.lp[f.lp.pos(10)] > b.width() / 2 ) add_guess( 't', b.right() );
          else add_guess( 'r', b.right() );
          return;
          }
        }
      int dmax = 0; bool bar = false;
      for( int row = b.vcenter(); row > b.vpos( 25 ); --row )
        {
        int d = b.hcenter() - b.seek_left( row, b.hcenter() );
        if( d > dmax ) dmax = d;
        else if( 2 * d < dmax && dmax > 2 ) bar = true;
        if( bar && Ocrad::similar( d, dmax, 25 ) )
          {
          int col, limit = b.seek_right( b.vcenter(), b.hcenter() );
          for( col = b.hcenter(); col <= limit; ++col )
            if( b.seek_bottom( b.vcenter(), col ) < b.bottom() ) break;
          if( b.left() < col && col < b.right() )
            { only_guess( 0, b.left() ); add_guess( 'f', col - 1 );
            add_guess( 'l', b.right() ); return; }
          }
        }
      }
    else if( code == 'h' )	// Looks for merged 'rf'
      {
      if( b.seek_right( b.top() + 1, b.left() ) > b.hcenter() )
        {
        if( f.rp[f.rp.pos(70)] >= 2 &&
            b.seek_top( b.vpos(70), b.right() ) > b.top() )
          {
          int col = 0, hmin = f.hp.range() + 1;
          for( int i = b.hpos(40); i <= b.hpos(60); ++i )
            if( f.hp[i-b.left()] < hmin )
              { hmin = f.hp[i-b.left()]; col = i; }
          if( col > b.left() && col < b.right() )
            { only_guess( 0, b.left() ); add_guess( 'r', col - 1 );
            add_guess( 'f', b.right() ); }
          }
        return;
        }
      }
    else if( code == 'k' )	// Looks for merged 'rt'
      {
      if( b.seek_right( b.top() + 1, b.left() ) > b.hcenter() )
        {
        only_guess( 0, b.left() ); add_guess( 'r', b.hcenter() );
        add_guess( 't', b.right() ); return;
        }
      }
    if( code ) { add_guess( code, 0 ); return; }
    }
  if( f.bp.minima() == 3 ) { add_guess( 'm', 0 ); return; }
  if( f.bp.minima() == 4 )
    {
    int col = b.seek_right( b.bottom() - 1, b.left() );
    col = b.seek_right( b.bottom() - 1, col + 1, false );
    col = b.seek_right( b.bottom() - 1, col + 1 );
    if( col > b.left() && col < b.right() )
      { only_guess( 0, b.left() ); add_guess( 'r', col );
      add_guess( 'm', b.right() ); return; }
    }

  if( f.tp.minima() == 3 )
    {
    int hdiff; if( !b.bottom_hook( &hdiff ) ) add_guess( 'w', 0 );
    return;
    }

  code = f.test_line( charbox_vcenter );
  if( code ) { add_guess( code, 0 ); return; }

  code = f.test_misc();
  if( code ) { add_guess( code, 0 ); return; }
  }


// Recognizes 1 block characters with 1 hole.
// 0469ADOPQRabdegopq#
void Character::recognize111( const Charset & charset, int charbox_vcenter ) throw()
  {
  const Block & b = block( 0 );
  const Block & h = b.block( 0 );			// hole
  Features f( b );
  int top_delta = h.top() - b.top(), bottom_delta = b.bottom() - h.bottom();

  if( std::abs( top_delta - bottom_delta ) <= 2 ||
      Ocrad::similar( top_delta, bottom_delta, 40 ) ) // hole is vertically centered
    {
    int code = f.test_4ADQao( charset, charbox_vcenter );
    if( code )
      {
      if( code == 'Q' ) add_guess( 'a', 1 );
      add_guess( code, 0 );
      }
    return;
    }

  if( top_delta < bottom_delta )	// hole is high
    {
    int code = f.test_49ARegpq();
    if( code ) add_guess( code, 0 );
    return;
    }

  if( top_delta > bottom_delta )	// hole is low
    {
    int code = f.test_6abd( charset );
    if( code )
      {
      add_guess( code, 0 );
      if( code == UCS::SOACUTE )
        {
        int row = h.top() - ( b.bottom() - h.bottom() ) - 1;
        if( row <= b.top() || row + 1 >= h.top() ) return;
        Block & b1 = const_cast< Block & >( b );
        Block b2( b, *b.blockmap(), b.id() );
        b1.bottom( row ); b2.top( row + 1 );
        _block_vector.push_back( b2 );
        }
      }
    }
  }


// Recognizes 1 block characters with 2 holes.
// 8BQg$&
void Character::recognize112() throw()
  {
  const Block & b = block( 0 );
  const Block & h1 = b.block( 0 );		// upper hole
  const Block & h2 = b.block( 1 );		// lower hole
  int a1 = h1.area();
  int a2 = h2.area();
//  Features f( b );

  if( h1.includes_vcenter( h2 ) && h2.includes_vcenter( h1 ) )
    { add_guess( 'm', 0 ); return; }

  Profile lp( b, Profile::left );
  Profile tp( b, Profile::top );
  Profile rp( b, Profile::right );
  Profile bp( b, Profile::bottom );
  if( Ocrad::similar( a1, a2, 50 ) &&		// I don't like this
      h1.bottom() <= h2.top() )
    {
    int hdiff;
    if( b.bottom_hook( &hdiff ) && hdiff > b.height() / 2 )
      if( b.top_hook( &hdiff ) && hdiff > b.height() / 2 )
        { add_guess( 's', 0 ); return; }
  
    if( lp.isflats() ) { add_guess( 'B', 0 ); return; }

    int col1 = h1.seek_left( h1.bottom(), h1.right() + 1 ) - 1;
    int col2 = h2.seek_right( h2.top(), h2.left() - 1 ) + 1;
    if( col1 <= col2 )
      {
      if( lp.isconvex() || lp.ispit() ) add_guess( 'e', 1 );
      else if( !rp.isctip() ) add_guess( 'a', 1 );
      add_guess( '$', 0 ); return;
      }

    if( b.hcenter() > h1.hcenter() && b.hcenter() > h2.hcenter() &&
        ( b.hcenter() >= h1.right() || b.hcenter() >= h2.right() ) )
      { add_guess( '&', 0 ); return; }
    for( int row = h1.bottom() + 1; row < h2.top(); ++row )
      if( _blockmap->id( row, hcenter() ) == 0 )
        { add_guess( 'g', 0 ); return; }
    if( bp.isconvex() )
      {
      if( lp.ispit() || lp.isconvex() ) add_guess( 'e', 1 );
      else if( rp.ispit() ) add_guess( 'a', 1 );
      add_guess( '8', 0 ); return;
      }
    if( lp.minima() == 2 && rp.minima() == 1 ) { add_guess( 'a', 0 ); return; }
    add_guess( 'B', 1 ); add_guess( 'a', 0 ); return;
    }
  if( a1 > a2 && h1.h_overlaps( h2 ) )
    {
    if( !h1.v_overlaps( h2 ) ) { add_guess( 'g', 0 ); return; }
    if( h1.h_includes( h2 ) ) { add_guess( 'Q', 0 ); return; }
    return;
    }
  if( a1 < a2 && tp.minima() == 1 )
    {
    if( h1.h_overlaps( h2 ) && rp.minima() == 1 )
      {
      if( 2 * h1.height() > h2.height() && 2 * h1.width() > h2.width() &&
          3 * h2.width() >= b.width() && !lp.isctip() )
        add_guess( 'B', 0 );
      else add_guess( 'a', 0 );
      return;
      }
    if( h1.bottom() < h2.top() ) { add_guess( '&', 0 ); return; }
    }
  }
