/*  GNU Ocrad - Optical Character Recognition program
    Copyright (C) 2003 Antonio Diaz Diaz.

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include <cstdio>
#include <vector>
#include "common.h"
#include "rectangle.h"
#include "block.h"
#include "character.h"
#include "textline.h"
#include "textblock.h"


Textblock::Textblock( const std::vector< Block > & block_vector ) throw()
  {
  // First pass. Assign blocks to characters and create lines.
  {
  const Character * prev = 0;
  int current_line = 0;
  for( std::vector< Block >::const_iterator p = block_vector.begin();
       p != block_vector.end(); ++p )
    {
    Character c( *p );
    if( c.height() * c.width() < 4 ) continue;
    if( c.height() > 35 * c.width() ) continue;
    if( c.width() > 20 * c.height() ) continue;
    const Block * next = 0;
    if( p + 1 != block_vector.end() ) next = &(*(p+1));
    if( prev && ( c.right() < prev->left() || c.top() > prev->bottom() ) &&
        ( !next || next->right() < prev->left() || next->top() > prev->bottom() ) )
      {
      int i;
      for( i = 0; i < line( current_line ).characters(); ++i )
        {
        Character & ci = line( current_line ).character(i);
        if( c.top() <= ci.bottom() &&
            ( i > 0 || c.right() < ci.left() ||
              ci.height() <= 2 * c.height() ||
              ci.height() <= 2 * line( current_line ).mean_height() ) ) break;
        }
      if( i >= line( current_line ).characters() ) prev = 0;
      }
    if( !prev ) { add_line(); current_line = lines() - 1; }
    line( current_line ).shift_character( c );
    prev = & line( current_line ).back();
    }
  }

  // Second pass. Join lines of i-dots and tildes.
  for( int current_line = 0; current_line < lines() - 1; )
    {
    bool joined = false;
    Textline & line1 = line( current_line );
    Textline & line2 = line( current_line + 1 );
    if( line1.characters() <= 2 * line2.characters() &&
        2 * line1.mean_height1() < line2.mean_height() )
      for( int i1 = 0 ; !joined && i1 < line1.characters(); ++i1 )
        {
        Character & c1 = line1.character( i1 );
        if( c1.height() * 2 >= line2.mean_height() ) continue;
        for( int i2 = 0 ; !joined && i2 < line2.characters(); ++i2 )
          {
          Character & c2 = line2.character( i2 );
          if( c2.right() < c1.left() ) continue;
          if( c2.left() > c1.right() ) break;
          if( ( c2.includes_hcenter( c1 ) || c1.includes_hcenter( c2 ) )
              && c2.top() - c1.bottom() < line2.mean_height() )
            {
            line1.join( line2 ); delete_line( current_line + 1 );
            joined = true;
            }
          }
        }
    if( !joined ) ++current_line;
    }

  // Third pass. Build the vertical composite characters.
  for( int current_line = 0; current_line < lines(); ++current_line )
    {
    Textline & line1 = line( current_line );
    for( int i = 0 ; i < line1.characters() - 1; )
      {
      Character & c1 = line1.character( i );
      bool joined = false;
      for( int j = i + 1 ; j < line1.characters(); ++j )
        {
        Character & c2 = line1.character( j );
        if( !c1.h_overlaps( c2 ) ) continue;
        Character *cup, *cdn;
        if( c1.vcenter() >= c2.vcenter() ) cup = &c2, cdn = &c1;
        else cup = &c1, cdn = &c2;
        if( cdn->includes_hcenter( *cup ) || cup->includes_hcenter( *cdn ) ||
            ( cdn->top() > cup->bottom() && cdn->hcenter() < cup->hcenter() ) )
          {
          if( cdn == &c1 ) { c1.join( c2 ); line1.delete_character( j ); }
          else { c2.join( c1 ); line1.delete_character( i ); }
          joined = true; break;
          }
        }
      if( !joined ) ++i;
      }
    }

  // Fourth pass. Add spaces between characters.
  for( int current_line = 0; current_line < lines(); ++current_line )
    {
    Textline & line1 = line( current_line );
    int mw = line1.mean_width();
    int mg = line1.mean_gap_width();
    if( mw < 2 || mg < 1 ) continue;
    for( int i = 0 ; i < line1.characters() - 1; ++i )
      {
      Character & c1 = line1.character( i );
      Character & c2 = line1.character( i + 1 );
      int dist = ( c2.left() > c1.right() ) ? c2.left() - c1.right() - 1 : 0;
      if( dist >= mw || ( dist > (mw / 3) && dist > (mg * 2) ) )
        {
        int left = c1.right() + 1;
        int top = ( c1.top() + c2.top() ) / 2;
        int right = c2.left() - 1;
        int bottom = ( c1.bottom() + c2.bottom() ) / 2;
        Character c( Rectangle( left, top, right, bottom ), *c1.blockmap() );
        c.add_guess( ' ', 0 );
        line1.insert_character( ++i, c );
        }
      }
    }

  // Fifth pass. Add blank lines.
  if( lines() >= 3 )
    {
    int mean_vdistance = (line(lines()-1).mean_vcenter() - line(0).mean_vcenter()) / (lines() - 1);
    if( mean_vdistance > 0 )
      for( int current_line = 0; current_line + 1 < lines(); ++current_line )
        {
        Textline & line1 = line( current_line );
        Textline & line2 = line( current_line + 1 );
        int vdistance = line2.mean_vcenter() - line1.mean_vcenter();
        if( vdistance <= 0 ) continue;
        int newlines = (vdistance + (mean_vdistance / 2)) / mean_vdistance;
        for( int i = 1; i < newlines; ++i ) insert_line( ++current_line );
        }
    }

  for( int i = 0; i < lines(); ++i )
    {
    // First pass. Recognize the easy characters.
    line(i).recognize1();
    // Second pass. Use context to clear up ambiguities.
    line(i).recognize2();
    }
  }


int Textblock::characters() const throw()
  {
  int total = 0;
  for( int i = 0; i < lines(); ++i ) total += line(i).characters();
  return total;
  }


void Textblock::insert_line( const int i ) throw()
  {
  if( i < 0 || i > lines() )
    Ocrad::internal_error( "insert_line, index out of bounds" );
  data.insert( data.begin() + i, Textline() );
  }


void Textblock::delete_line( const int i ) throw()
  {
  if( i < 0 || i > lines() )
    Ocrad::internal_error( "delete_line, index out of bounds" );
  data.erase( data.begin() + i );
  }


Textline & Textblock::line( const int i ) const throw()
  {
  if( i < 0 || i > lines() )
    Ocrad::internal_error( "line, index out of bounds" );
  return data[i];
  }


void Textblock::print( FILE * outfile, const bool graph, const bool recursive )
								const throw()
  {
  if( graph || recursive ) std::fprintf( outfile, "%d lines\n\n", lines() );

  for( int i = 0; i < lines(); ++i )
    {
    if( graph || recursive )
      std::fprintf( outfile, "%d characters in line %d\n", line(i).characters(), i+1 );
    line(i).print( outfile, graph, recursive );
    }
  std::fputs( "\n", outfile );
  }


void Textblock::xprint( FILE * exportfile ) const throw()
  {
  std::fprintf( exportfile, "lines %d\n", lines() );

  for( int i = 0; i < lines(); ++i )
    {
    std::fprintf( exportfile, "line %d chars %d height %d\n", i + 1,
                  line(i).characters(), line(i).mean_height() );
    line(i).xprint( exportfile );
    }
  }
