/*  GNU Ocrad - Optical Character Recognition program
    Copyright (C) 2003 Antonio Diaz Diaz.

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include <cstdio>
#include <vector>
#include "common.h"
#include "rectangle.h"
#include "block.h"
#include "blockmap.h"
#include "character.h"
#include "iso_8859_1.h"


// Return the area of the main blocks only (no recursive)
int Character::area() const throw()
  {
  int area = 0;
  for( int i = 0; i < blocks(); ++i ) area += _block_vector[i].area();
  return area;
  }


void Character::add_block( const Block & b ) throw()
  {
  if( _blockmap != b.blockmap() )
    Ocrad::internal_error( "can't add block to character of different blockmaps.\n" );
  add_rectangle( b );
  int i = 0;
  while( i < blocks() && b.vcenter() <= _block_vector[i].vcenter() ) ++i;
  _block_vector.insert( _block_vector.begin() + i, b );
  }


void Character::add_guess( unsigned char ch, int value ) throw()
  {
  _guess.push_back( Guess( ch, value ) );
  }


void Character::insert_guess( int i, unsigned char ch, int value ) throw()
  {
  if( i < 0 || i > guesses() )
    Ocrad::internal_error( "insert_guess, index out of bounds" );
  _guess.insert( _guess.begin() + i, Guess( ch, value ) );
  }


void Character::delete_guess( int i ) throw()
  {
  if( i < 0 || i >= guesses() )
    Ocrad::internal_error( "delete_guess, index out of bounds" );
  _guess.erase( _guess.begin() + i );
  }


void Character::only_guess( unsigned char ch, int value ) throw()
  { _guess.clear(); add_guess( ch, value ); }


void Character::swap_guesses( int i, int j ) throw()
  {
  if( i < 0 || i >= guesses() || j < 0 || j >= guesses() )
    Ocrad::internal_error( "swap_guesses, index out of bounds" );
  unsigned char ch = _guess[i].ch;
  _guess[i].ch = _guess[j].ch; _guess[j].ch = ch;
  }


const Character::Guess & Character::guess( int i ) const throw()
  {
  if( i < 0 || i >= guesses() )
    Ocrad::internal_error( "guess, index out of bounds" );
  return _guess[i];
  }


bool Character::isalnum() const throw()
  {
  for( int i = 0; i < guesses(); ++i )
    if( ISO_8859_1::isalnum( _guess[i].ch ) ) return true;
  return false;
  }


void Character::join( Character & c ) throw()
  {
  if( _blockmap != c._blockmap )
    Ocrad::internal_error( "can't join characters of different blockmaps.\n" );
  for( int i = 0; i < c.blocks(); ++i ) add_block( c._block_vector[i] );
  }


void Character::print( FILE * outfile, bool graph, bool recursive )
							const throw()
  {
  if( graph || recursive )
    {
    std::fprintf( outfile, "%d guesses    ", guesses() );
    for( int i = 0; i < guesses(); ++i )
      std::fprintf( outfile, "guess '%c', confidence %d    ",
                    _guess[i].ch, _guess[i].value );
    std::fputs( "\n", outfile );
    }
  if( graph )
    {
    std::fprintf( outfile, "width = %d, height = %d, hcenter = %d, vcenter = %d, black area = %d%%\n",
                  width(), height(), hcenter(), vcenter(),
                  ( area() * 100 ) / ( height() * width() ) );
    for( int row = top(); row <= bottom(); ++row)
      {
      for( int col = left(); col <= right(); ++col)
        {
        char ch = ( row == vcenter() && col == hcenter() ) ? '+' : '';
        int id = _blockmap->id( row, col );
        if( id != 0 )
          {
          std::vector< Block >::const_iterator p = _block_vector.begin();
          for( ; p != _block_vector.end(); ++p )
            if( p->compare_id( id, recursive ) )
              {
              if( id > 0 ) ch = (ch == '+') ? 'C' : 'O';
              else ch = (ch == '+') ? '=' : '-';
              break;
              }
          }
        std::fprintf( outfile, " %c", ch );
        }
      std::fputs( "\n", outfile );
      }
    std::fputs( "\n", outfile );
    }
  else
    {
    if( recursive )
      {
      for( int i = 0; i < guesses(); ++i )
        std::fprintf( outfile, "%c %4d,    ", _guess[i].ch, _guess[i].value );
      std::fputs( "\n", outfile );
      }
    else if( guesses() ) std::putc( _guess[0].ch, outfile );
    else std::putc( '_', outfile );
    }
  }


void Character::xprint( FILE * exportfile ) const throw()
  {
  std::fprintf( exportfile, "%3d %3d %2d %2d; %d",
                left(), top(), width(), height(), guesses() );

  for( int i = 0; i < guesses(); ++i )
    std::fprintf( exportfile, ", '%c'%d", _guess[i].ch, _guess[i].value );

  std::fputs( "\n", exportfile );
  }
