/*  GNU OCRAD - Optical Character Recognition program
    Copyright (C) 2003 Antonio Diaz Diaz.

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include <cstdio>
#include <list>
#include <map>
#include <vector>
#include "common.h"
#include "rectangle.h"
#include "block.h"
#include "character.h"


// Return the area of the main blocks only (no recursive)
int Character::area() const throw()
  {
  int area = 0;
  std::list< Block >::const_iterator p = _block_list.begin();
  for( ; p != _block_list.end(); ++p ) area += p->area();
  return area;
  }


void Character::add_block( const Block & b ) throw()
  {
  if( _blockmap != b.blockmap() )
    internal_error( "can't add block to character of different blockmaps.\n" );
  add_rectangle( b );
  if( _block_list.size() == 0 ) { _block_list.push_back( b ); return; }
  std::list< Block >::iterator p = _block_list.begin();
  for( ; p != _block_list.end(); ++p )
    if( b.vcenter() >= p->vcenter() ) break;
  _block_list.insert( p, b );
  }


void Character::add_guess( unsigned char ch, int value ) throw()
  {
  _guess.push_back( Guess( ch, value ) );
  }


void Character::insert_guess( int i, unsigned char ch, int value ) throw()
  {
  if( i < 0 || i > guesses() )
    internal_error( "insert_guess, index out of bounds" );
  _guess.insert( _guess.begin() + i, Guess( ch, value ) );
  }


void Character::delete_guess( int i ) throw()
  {
  if( i < 0 || i >= guesses() )
    internal_error( "delete_guess, index out of bounds" );
  _guess.erase( _guess.begin() + i );
  }


void Character::only_guess( unsigned char ch, int value ) throw()
  { _guess.clear(); add_guess( ch, value ); }


void Character::swap_guesses( int i, int j ) throw()
  {
  if( i < 0 || i >= guesses() || j < 0 || j >= guesses() )
    internal_error( "swap_guesses, index out of bounds" );
  Guess tmp = _guess[i]; _guess[i] = _guess[j]; _guess[j] = tmp;
  }


const Character::Guess & Character::guess( int i ) const throw()
  {
  if( i < 0 || i >= guesses() )
    internal_error( "guess, index out of bounds" );
  return _guess[i];
  }


void Character::join( Character & c ) throw()
  {
  if( _blockmap != c._blockmap )
    internal_error( "can't join characters of different blockmaps.\n" );
  std::list< Block >::const_iterator p = c._block_list.begin();
  for( ; p != c._block_list.end(); ++p ) add_block( *p );
  }


void Character::print( FILE * outfile, bool graph, bool recursive )
							const throw()
  {
  if( graph || recursive )
    {
    fprintf( outfile, "%d guesses    ", guesses() );
    for( int i = 0; i < guesses(); ++i )
      fprintf( outfile, "guess '%c', confidence %d    ", _guess[i].ch,
                                                         _guess[i].value );
    fprintf( outfile, "\n" );
    }
  if( graph )
    {
    const Block * b = ( _block_list.size() ) ? &_block_list.front() : 0;
    fprintf( outfile, "height = %d, width = %d, vcenter = %d, hcenter = %d, black area = %d%%\n",
		height(), width(), vcenter(), hcenter(), (area() * 100) / (height() * width()) );
    for( int row = top(); row <= bottom(); ++row)
      {
      for( int col = left(); col <= right(); ++col)
        {
        char ch = ( row == vcenter() && col == hcenter() ) ? '-' : '';
        if( b && b->id( row, col ) > 0 )
          {
          std::list< Block >::const_iterator p = _block_list.begin();
          for( ; p != _block_list.end(); ++p )
            if( p->compare_id( b->id( row, col ), recursive ) )
              { ch = (ch == '-') ? '+' : 'O'; break; }
          }
        fprintf( outfile, " %c", ch );
        }
      fprintf( outfile, "\n" );
      }
    fprintf( outfile, "\n" );
    }
  else
    {
    if( recursive )
      {
      for( int i = 0; i < guesses(); ++i )
        fprintf( outfile, "%c %4d,    ", _guess[i].ch, _guess[i].value );
      fputs( "\n", outfile );
      }
    else if( guesses() ) fputc( _guess[0].ch, outfile );
    else fputc( '_', outfile );
    }
  }


void Character::xprint( FILE * exportfile ) const throw()
  {
  fprintf( exportfile, "%d %d %d %d;%d",
           left(), top(), width(), height(), guesses() );

  for( int i = 0; i < guesses(); ++i )
    fprintf( exportfile, ",'%c'%d", _guess[i].ch, _guess[i].value );

  fprintf( exportfile, "\n" );
  }
