/*  ocrad - Optical Character Recognition program
    Copyright (C) 2003 Antonio Diaz Diaz.

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/
/*
    Return values: 0 for a normal exit, 1 for environmental problems
    (file not found, invalid flags, I/O errors, etc), 2 to indicate a
    corrupt or invalid input file, 3 for an internal consistency error
    (eg, bug) which caused 'ocrad' to panic.
*/

#include <cctype>
#include <cstdio>
#include <cstdlib>
#include <cstring>
#include <getopt.h>
#include <list>
#include <vector>
#include "common.h"
#include "rectangle.h"
#include "bitmap.h"
#include "block.h"
#include "blockmap.h"
#include "character.h"
#include "textline.h"
#include "textpage.h"


void show_version()
  {
  printf( "%s version %s\n", PROGRAM_NAME, PROGRAM_VERSION );
  printf( "Copyright (C) %s Antonio Diaz.\n", PROGRAM_YEAR );
  }

void show_error( char * msg )
  {
  if( msg && msg[0] != 0 ) fprintf( stderr,"%s: %s\n", PROGRAM_NAME, msg );
  fprintf( stderr,"Try `%s --help' for more information.\n", PROGRAM_NAME );
  }

void show_help()
  {
  printf( "Usage: %s [options] [files]\n", PROGRAM_NAME );
  printf( "Optical Character Recognition program\n" );
  printf( "Reads pbm file(s), or standard input, and sends text to standard output\n" );
  printf( "Options:\n");
  printf( "  -h, --help               display this help and exit\n");
  printf( "  -V, --version            output version information and exit\n");
  printf( "  -1, -4                   pbm output file type (debug)\n");
  printf( "  -a, --append             append text to output file\n");
  printf( "  -C, --copy               'copy' input to output (debug)\n");
  printf( "  -c, --column=<column>    process only the specified text column\n");
  printf( "  -D, --debug=<level>      (0-100) output intermediate data (debug)\n");
  printf( "  -f, --force              force overwrite of output file\n");
//  printf( "  -l, --level              level input image before doing OCR\n");
  printf( "  -o <file>                place the output into <file>\n");
//  printf( "  -r <scale>               reduce input image by <scale> (debug)\n");
  printf( "  -s <type>                make a 'special file' (debug)\n");
//  printf( "  -t <threshold>           set threshold for -r option (debug)\n");
  printf( "  -v, --verbose            be verbose\n");

  printf( "\nReport bugs to <ant_diaz@teleline.es>.\n");
  }


void ocr( const Blockmap & blockmap, FILE * outfile, int debug_level ) throw()
  {
  if( debug_level >= 90 )
    { blockmap.print( outfile, debug_level ); return; }

  Textpage textpage( blockmap.block_list() );

  // First pass. Recognize the easy characters.
  textpage.recognize1();

  // Second pass. Use context to resolve ambiguities.
  textpage.recognize2();

  if( debug_level >= 86 )
    {
    bool graph = debug_level >= 88 ? true : false;
    bool recursive = (debug_level & 1) ? true : false;
    textpage.print( outfile, graph, recursive );
    return;
    }

  textpage.print( outfile, false, false );
  }


int main(int argc, char *argv[]) 
  {
  FILE *infile = stdin, *outfile = stdout;
  char *outfile_name = 0;
  int column = -1, debug_level = 0, scale = 0, threshold = -1;
  char filetype = 0, specialtype = 0;
  bool append = false, copy = false, force = false, level = false;
  bool verbose = false;

  // scan arguments

  while( true )			// process options
    {
    int c;
    static struct option long_options[] =
      {
      {"append", no_argument, 0, 'a'},
      {"column", required_argument, 0, 'c'},
      {"copy", no_argument, 0, 'C'},
      {"debug", required_argument, 0, 'D'},
      {"force", no_argument, 0, 'f'},
      {"help", no_argument, 0, 'h'},
      {"level", no_argument, 0, 'l'},
      {"verbose", no_argument, 0, 'v'},
      {"version", no_argument, 0, 'V'},
      {0, 0, 0, 0}
      };

    c = getopt_long( argc, argv, "14aCc:D:fhlo:r:s:t:Vv", long_options, 0 );
    if( c == -1 ) break;		// all options processed

    switch( c )
      {
      case 0: break;
      case '?': return 1;  // `getopt_long' already printed an error message.
      case '1':
      case '4': filetype = c; break;
      case 'a': append = true; break;
      case 'C': copy = true; break;
      case 'c': column = strtol( optarg, 0, 0 ) - 1; break;
      case 'D': debug_level = strtol( optarg, 0, 0 ); break;
      case 'f': force = true; break;
      case 'h': show_help(); return 0;
      case 'l': level = true; break;
      case 'o':	outfile_name = optarg; break;
      case 'r': scale = strtol( optarg, 0, 0 ); break;
      case 's': specialtype = optarg[0]; break;
      case 't': threshold = strtol( optarg, 0, 0 ); break;
      case 'V':	show_version(); return 0;
      case 'v': verbose = true; break;
      default: return 1;
      }
    } // end process options

  if( outfile_name )
    {
    if( append ) outfile = fopen( outfile_name, "a" );
    else if( force ) outfile = fopen( outfile_name, "w" );
    else if( ( outfile = fopen( outfile_name, "wx" ) ) == 0 )
      {
      fprintf( stderr, "Output file %s already exists.\n", outfile_name );
      return 1;
      }
    if( !outfile )
      { fprintf( stderr, "Cannot open %s\n", outfile_name ); return 1; }
    }

// process any remaining command line arguments (input files)
  do {
    if( optind < argc )
      {
      if( infile != stdin ) fclose( infile );
      if( strcmp( argv[optind], "-" ) == 0 ) infile = stdin;
      else if( ( infile = fopen( argv[optind], "r" ) ) == 0 )
	{ fprintf(stderr,"Cannot open %s\n",argv[optind]); return 1; }
      ++optind;
      }

// 'infile' contains the scanned text image to be converted to ascii
// (in pbm format).
// 'outfile' is the destination for the ascii version of the scanned
// text image. (Or for a pbm file if debugging).

    try
      {
      Bitmap page_image( infile );

      // make a list of column coordinates (rectangles)
      page_image.find_columns( verbose );

      if( column >= static_cast< int > ( page_image.rectangle_list().size() ) )
        {
        fprintf( stderr,"This page has only %d column(s)\n",
                 page_image.rectangle_list().size());
        return 1;
        }

      if( level )
        fprintf( stderr, "\nslope_best = %d\n", page_image.horizontalify(verbose) );

      if( scale != 0 )
        {
        Bitmap reduced( page_image, threshold, scale );
        reduced.save( outfile, filetype ); return 0;
        }
      else if( specialtype != 0 )
        {
        Bitmap::type t;
        if( specialtype == 'v' ) t = Bitmap::vertical_histogram ;
        else if( specialtype == 'h' ) t = Bitmap::horizontal_histogram ;
        else if( specialtype == 'g' ) t = Bitmap::connected_ground ;
        else { show_error( "bad special type" ); return 1; }
        Bitmap sb( page_image, t );
        sb.save( outfile, filetype ); return 0;
        }
      else if( copy )
        {
        if( column < 0 ) page_image.save( outfile, filetype );
        else
          {
          Bitmap bitmap( page_image, page_image.rectangle_list()[column] );
          bitmap.save( outfile, filetype );
          }
        return 0;
        }

      // call the character recognizer for every column of text
      for( int c = 0; c < static_cast<int>(page_image.rectangle_list().size()); ++c )
        if( column < 0 || column == c )
          {
          Blockmap blockmap( page_image, c, debug_level );
          ocr( blockmap, outfile, debug_level );
          }
      }

    catch( Bitmap::Error e ) { fputs( e.s, stderr ); return 2; }
    }
  while( optind < argc );
  fclose( outfile );
  return 0;
  }
