/*  GNU Ocrad - Optical Character Recognition program
    Copyright (C) 2003, 2004 Antonio Diaz Diaz.

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
*/

#include <algorithm>
#include <cstddef>
#include <cstdio>
#include <cstdlib>
#include "common.h"
#include "rectangle.h"


namespace {

void error( const char * s ) throw() __attribute__ ((noreturn));
void error( const char * s ) throw()
  { Ocrad::internal_error( s ); }


int hypoti( int c1, int c2 )
  {
  long long temp = c1; temp *= temp;
  long long target = c2; target *= target; target += temp;
  int lower = std::max( std::abs(c1), std::abs(c2) );
  int upper = std::abs(c1) + std::abs(c2);
  while( upper - lower > 1 )
    {
    int m = ( lower + upper ) / 2;
    temp = m; temp *= temp;
    if( temp < target ) lower = m; else upper = m;
    }
  temp = lower; temp *= temp; target *= 2; target -= temp;
  temp = upper; temp *= temp;
  if( target < temp ) return lower;
  else return upper;
  }

} // end namespace


Rectangle::Rectangle( int l, int t, int r, int b ) throw()
  {
  if( r < l || b < t )
    {
    std::fprintf( stderr, "l = %d, t = %d, r = %d, b = %d\n", l, t, r, b );
    error( "bad parameter building a Rectangle" );
    }
  _left = l; _top = t; _right = r; _bottom = b;
  }


void Rectangle::left( int l ) throw()
  {
  if( l > _right ) error( "left, bad parameter resizing a Rectangle" );
  _left = l;
  }


void Rectangle::top( int t ) throw()
  {
  if( t > _bottom ) error( "top, bad parameter resizing a Rectangle" );
  _top = t;
  }


void Rectangle::right( int r ) throw()
  {
  if( r < _left ) error( "right, bad parameter resizing a Rectangle" );
  _right = r;
  }


void Rectangle::bottom( int b ) throw()
  {
  if( b < _top ) error( "bottom, bad parameter resizing a Rectangle" );
  _bottom = b;
  }


void Rectangle::add_point( int row, int col ) throw()
  {
  if( row > _bottom ) _bottom = row; else if( row < _top ) _top = row;
  if( col > _right ) _right = col;   else if( col < _left ) _left = col;
  }


void Rectangle::add_rectangle( const Rectangle & r ) throw()
  {
  if( r._left < _left ) _left = r._left;
  if( r._top < _top ) _top = r._top;
  if( r._right > _right ) _right = r._right;
  if( r._bottom > _bottom ) _bottom = r._bottom;
  }


bool Rectangle::includes( const Rectangle & r ) const throw()
  {
  return ( _left  <= r._left  && _top    <= r._top &&
           _right >= r._right && _bottom >= r._bottom );
  }


bool Rectangle::includes( int row, int col ) const throw()
  {
  return ( _left <= col && _right >= col && _top <= row && _bottom >= row );
  }


bool Rectangle::strictly_includes( const Rectangle & r ) const throw()
  {
  return ( _left  < r._left  && _top    < r._top &&
           _right > r._right && _bottom > r._bottom );
  }


bool Rectangle::strictly_includes( int row, int col ) const throw()
  {
  return ( _left < col && _right > col && _top < row && _bottom > row );
  }


bool Rectangle::includes_hcenter( const Rectangle & r ) const throw()
  { return ( _left <= r.hcenter() && _right >= r.hcenter() ); }


bool Rectangle::includes_vcenter( const Rectangle & r ) const throw()
  { return ( _top <= r.vcenter() && _bottom >= r.vcenter() ); }


bool Rectangle::h_includes( const Rectangle & r ) const throw()
  { return ( _left <= r._left && _right >= r._right ); }


bool Rectangle::v_includes( const Rectangle & r ) const throw()
  { return ( _top <= r._top && _bottom >= r._bottom ); }


bool Rectangle::h_includes( int col ) const throw()
  { return ( _left <= col && _right >= col ); }


bool Rectangle::v_includes( int row ) const throw()
  { return ( _top <= row && _bottom >= row ); }


bool Rectangle::h_overlaps( const Rectangle & r ) const throw()
  { return ( _left <= r._right && _right >= r._left ); }


bool Rectangle::v_overlaps( const Rectangle & r ) const throw()
  { return ( _top <= r._bottom && _bottom >= r._top ); }


bool Rectangle::is_hcentered_in( const Rectangle & r ) const throw()
  {
  if( this->h_includes( r.hcenter() ) ) return true;
  int w = std::min( r.height(), r.width() ) / 2;
  if( width() < w )
    {
    int d = ( w + 1 ) / 2;
    if( hcenter() - d <= r.hcenter() && hcenter() + d >= r.hcenter() )
      return true;
    }
  return false;
  }


bool Rectangle::is_vcentered_in( const Rectangle & r ) const throw()
  {
  if( this->v_includes( r.vcenter() ) ) return true;
  int h = std::min( r.height(), r.width() ) / 2;
  if( height() < h )
    {
    int d = ( h + 1 ) / 2;
    if( vcenter() - d <= r.vcenter() && vcenter() + d >= r.vcenter() )
      return true;
    }
  return false;
  }


bool Rectangle::h_precedes( const Rectangle & r ) const throw()
  { return ( hcenter() < r.hcenter() ); }


bool Rectangle::v_precedes( const Rectangle & r ) const throw()
  {
  if( _bottom < r.vcenter() || vcenter() < r._top ) return true;
  if( this->includes_vcenter( r ) && r.includes_vcenter( *this ) )
    return this->h_precedes( r );
  return false;
  }


int Rectangle::distance( const Rectangle & r ) const throw()
  { return hypoti( h_distance( r ), v_distance( r ) ); }


int Rectangle::distance( int row, int col ) const throw()
  { return hypoti( h_distance( col ), v_distance( row ) ); }


int Rectangle::h_distance( const Rectangle & r ) const throw()
  {
  if( r._right <= _left ) return _left - r._right;
  if( r._left >= _right ) return r._left - _right;
  return 0;
  }

int Rectangle::h_distance( int col ) const throw()
  {
  if( col <= _left ) return _left - col;
  if( col >= _right ) return col - _right;
  return 0;
  }

int Rectangle::v_distance( const Rectangle & r ) const throw()
  {
  if( r._bottom <= _top ) return _top - r._bottom;
  if( r._top >= _bottom ) return r._top - _bottom;
  return 0;
  }

int Rectangle::v_distance( int row ) const throw()
  {
  if( row <= _top ) return _top - row;
  if( row >= _bottom ) return row - _bottom;
  return 0;
  }
