/*  GNU Ocrad - Optical Character Recognition program
    Copyright (C) 2003, 2004 Antonio Diaz Diaz.

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
*/

#include <cstdio>
#include <vector>
#include "common.h"
#include "rectangle.h"
#include "bitmap.h"
#include "block.h"
#include "blockmap.h"


namespace {

void add_point_to_block( std::vector< Block > & block_vector,
                         int row, int col, int id, int rwidth ) throw()
  {
  static int ib = 0, iw = 0;
  int vblocks = block_vector.size();
  if( id > 0 && ib < vblocks && block_vector[ib].id() == id )
    { block_vector[ib].add_point( row, col ); return; }
  if( id < 0 && iw < vblocks && block_vector[iw].id() == id )
    { block_vector[iw].add_point( row, col ); return; }

  for( int i = vblocks - 1; i >= 0; --i )
    if( block_vector[i].id() == id )
      {
      block_vector[i].add_point( row, col );
      if( id > 0 ) ib = i; else iw = i;
      if( vblocks - i <= rwidth ) return;
      Block temp = block_vector[i];
      block_vector.erase( block_vector.begin() + i );
      block_vector.push_back( temp );
      return;
      }
  Ocrad::internal_error( "add_point_to_block, lost block" );
  }


void create_block( std::vector< Block > & block_vector,
                   const Blockmap & blockmap, int row, int col, int id ) throw()
  {
  block_vector.push_back( Block( Rectangle( col, row, col, row ), blockmap, id ) );
  }


void delete_block( std::vector< Block > & block_vector,
                   std::vector< std::vector< int > > & data, int id ) throw()
  {
  for( int i = block_vector.size() - 1; i >= 0 ; --i )
    {
    Block & b = block_vector[i];
    if( b.id() == id )
      {
      for( int row = b.top(); row <= b.bottom(); ++row )
        for( int col = b.left(); col <= b.right(); ++col )
          if( data[row][col] == id ) data[row][col] = 0;
      block_vector.erase( block_vector.begin() + i );
      return;
      }
    }
  Ocrad::internal_error( "delete_block, lost block" );
  }


void join_blocks( std::vector< Block > & block_vector,
                  std::vector< std::vector< int > > & data, int id1, int id2 ) throw()
  {
  if( std::abs( id1 ) > std::abs( id2 ) )
    { int temp = id1; id1 = id2; id2 = temp; }

  int i1 = block_vector.size();
  int i2 = i1;
  while( --i1 >= 0 && block_vector[i1].id() != id1 );
  while( --i2 >= 0 && block_vector[i2].id() != id2 );
  if( i1 < 0 || i2 < 0 ) Ocrad::internal_error( "join_blocks, lost block" );

  Block & b1 = block_vector[i1];
  Block & b2 = block_vector[i2];
  for( int row = b2.top(); row <= b2.bottom(); ++row )
    for( int col = b2.left(); col <= b2.right(); ++col )
      if( data[row][col] == id2 ) data[row][col] = id1;
  b1.add_rectangle( b2 );
  block_vector.erase( block_vector.begin() + i2 );
  }


void ignore_abnormal_blocks( std::vector< Block > & block_vector ) throw()
  {
  for( int i = block_vector.size() - 1; i >= 0; --i )
    {
    const Block & b = block_vector[i];
    if( b.height() > 35 * b.width() ||
        b.width() > 20 * b.height() ||
        b.id() < 0 )
      block_vector.erase( block_vector.begin() + i );
    }
  }


void ignore_small_blocks( std::vector< Block > & block_vector ) throw()
  {
  int to = 0, blocks = block_vector.size();
  for( int from = 0; from < blocks; ++from )
    {
    const Block & b = block_vector[from];
    if( b.id() > 0 )	// black block
      {
      if( b.height() > 4 || b.width() > 4 ||
          ( ( b.height() > 2 || b.width() > 2 ) && b.area() > 5 ) )
        block_vector[to++] = b;
      }
    else		// white block (hole)
      {
      if( b.height() > 4 || b.width() > 4 ||
          ( ( b.height() > 2 || b.width() > 2 ) && b.area() > 3 ) )
        block_vector[to++] = b;
      }
    }
  if( to < blocks )
    block_vector.erase( block_vector.begin() + to, block_vector.end() );
  }


void ignore_wide_blocks( Blockmap::Zone & zone ) throw()
  {
  const Rectangle & r = zone.rectangle;
  std::vector< std::vector< Block > > & block_matrix = zone.block_matrix;

  for( unsigned int cut = 0; cut < block_matrix.size(); ++cut )
    {
    bool frame_found = false;
    for( unsigned int i = 0; i < block_matrix[cut].size(); )
      {
      std::vector< Block > & block_vector = block_matrix[cut];
      if( 2 * block_vector[i].width() < r.width() ) { ++i; continue; }
      Block b( block_vector[i] );
      block_vector.erase( block_vector.begin() + i );
      if( b.id() > 0 && 4 * b.area() > b.size() )	// image, not frame
        {
        if( 10 * b.width() > 8 * r.width() && 10 * b.height() > 8 * r.height() )
          { block_vector.clear(); break; }
        for( int j = block_vector.size() - 1; j >= 0; --j )
          {
          if( b.includes( block_vector[j] ) )
            block_vector.erase( block_vector.begin() + j );
          else if( block_vector[j].top() < b.top() ) break;
          }
        }
      else frame_found = true;
      }
    if( frame_found )	// Make cuts from blocks inside deleted frame(s)
      {
      int bottom = 0;
      for( unsigned int i = 0; i < block_matrix[cut].size(); ++i )
        {
        std::vector< Block > & block_vector = block_matrix[cut];
        const Block & b( block_vector[i] );
        if( b.bottom() > bottom )
          {
          int old_bottom = bottom; bottom = b.bottom();
          if( b.top() > old_bottom && i > 0 )
            {
            std::vector< Block > new_block_vector( block_vector.begin() + i, block_vector.end() );
            block_vector.erase( block_vector.begin() + i, block_vector.end() );
            block_matrix.insert( block_matrix.begin() + cut + 1, new_block_vector );
            ++cut; i = 0;
            }
          }
        }
      }
    }
  }


void order_blocks( std::vector< Block > & block_vector ) throw()
  {
  int vblocks = block_vector.size();
  int top = vblocks ? block_vector[0].top() : 0;

  for( int i = 1; i < vblocks; ++i )
    {
    int old_top = top;
    top = block_vector[i].top();
    if( top < old_top )
      {
      int j = i;
      Block temp = block_vector[i];
      while( --j >= 0 && top <= block_vector[j].top() )
        block_vector[j+1] = block_vector[j];
      block_vector[j+1] = temp;
      }
    }
  }


void remove_top_bottom_noise( std::vector< Block > & block_vector ) throw()
  {
  int blocks = block_vector.size();
  for( int i = 0; i < blocks; ++i )
    {
    Block & b = block_vector[i];
    if( b.height() < 11 ) continue;
    const Blockmap & bm = *(b.blockmap());

    int c = 0;
    for( int col = b.left(); col <= b.right(); ++col )
      if( bm.id( b.top(), col ) == b.id() && ++c > 1 ) break;
    if( c <= 1 ) b.top( b.top() + 1 );

    c = 0;
    for( int col = b.left(); col <= b.right(); ++col )
      if( bm.id( b.bottom(), col ) == b.id() && ++c > 1 ) break;
    if( c <= 1 ) b.bottom( b.bottom() - 1 );
    }
  }


void remove_left_right_noise( std::vector< Block > & block_vector ) throw()
  {
  int blocks = block_vector.size();
  for( int i = 0; i < blocks; ++i )
    {
    Block & b = block_vector[i];
    if( b.width() < 6 ) continue;
    const Blockmap & bm = *(b.blockmap());

    int c = 0;
    for( int row = b.top(); row <= b.bottom(); ++row )
      if( bm.id( row, b.left() ) == b.id() && ++c > 1 ) break;
    if( c <= 1 ) b.left( b.left() + 1 );

    c = 0;
    for( int row = b.top(); row <= b.bottom(); ++row )
      if( bm.id( row, b.right() ) == b.id() && ++c > 1 ) break;
    if( c <= 1 ) b.right( b.right() - 1 );
    }
  }

} // end namespace


int Blockmap::generate_black_id() { static int id = 0; return ++id; }

int Blockmap::generate_white_id() { static int id = 0; return --id; }


Blockmap::Blockmap( const Bitmap & page_image, int debug_level ) throw()
  : _height( page_image.height() ), _width( page_image.width() )
  {
  data.resize( _height );
  for( int row = 0; row < _height; ++row ) data[row].resize( _width );

  for( int zindex = 0; zindex < page_image.rectangles(); ++zindex )
    {
    _zone_vector.push_back( Zone( page_image.rectangle_vector()[zindex] ) );
    const Rectangle & r = _zone_vector[zindex].rectangle;
    std::vector< std::vector< Block > > & block_matrix = _zone_vector[zindex].block_matrix;
    std::vector< Block > block_vector;
    for( int col = std::max( 0, r.left() ); col <= r.right(); ++col )
      { int row = std::max( 0, r.top() ); data[row][col] = 0; }

    for( int row = std::max( 1, r.top() + 1 ); row <= r.bottom(); ++row )
      {
      std::vector< int > & datarow = data[row];
      std::vector< int > & datarow1 = data[row-1];
      bool blank_row = true;
      datarow[std::max(0,r.left())] = 0;
      for( int col = std::max( 1, r.left() + 1 ); col <= r.right(); ++col )
        {
        if( page_image.get_bit( row, col ) )	// black point
          {
          blank_row = false;
          if( datarow1[col] > 0 )
            {
            int id = datarow[col] = datarow1[col];
            add_point_to_block( block_vector, row, col, id, r.width() );
            if( datarow[col-1] > 0 && datarow[col-1] != id )
              join_blocks( block_vector, data, datarow[col-1], id );
            }
          else if( datarow[col-1] > 0 )
            {
            int id = datarow[col] = datarow[col-1];
            add_point_to_block( block_vector, row, col, id, r.width() );
            }
          else if( datarow1[col-1] > 0 )
            {
            int id = datarow[col] = datarow1[col-1];
            add_point_to_block( block_vector, row, col, id, r.width() );
            }
          else
            {
            int id = datarow[col] = generate_black_id();
            create_block( block_vector, *this, row, col, id );
            }
          }
        else						// white point
          {
          if( datarow[col-1] == 0 )
            {
            datarow[col] = 0;
            if( datarow1[col] < 0 ) delete_block( block_vector, data, datarow1[col] );
            }
          else if( datarow1[col] == 0 )
            {
            datarow[col] = 0;
            if( datarow[col-1] < 0 ) delete_block( block_vector, data, datarow[col-1] );
            }
          else if( datarow1[col] < 0 )
            {
            int id = datarow[col] = datarow1[col];
            add_point_to_block( block_vector, row, col, id, r.width() );
            if( datarow[col-1] < 0 && datarow[col-1] != id )
              join_blocks( block_vector, data, datarow[col-1], id );
            }
          else if( datarow[col-1] < 0 )
            {
            int id = datarow[col] = datarow[col-1];
            add_point_to_block( block_vector, row, col, id, r.width() );
            }
          else
            {
            int id = datarow[col] = generate_white_id();
            create_block( block_vector, *this, row, col, id );
            id = datarow1[col];
            if( datarow[col-1] != id )
              join_blocks( block_vector, data, datarow[col-1], id );
            }
          }
        }
      if( blank_row && block_vector.size() )
        {
        block_matrix.push_back( std::vector< Block >() );
        swap( block_vector, block_matrix.back() );
        }
      }
    if( block_vector.size() )
      {
      block_matrix.push_back( std::vector< Block >() );
      swap( block_vector, block_matrix.back() );
      }
    }

  if( debug_level <= 99 )
    for( int zindex = 0; zindex < zones(); ++zindex )
      {
      ignore_wide_blocks( _zone_vector[zindex] );
      std::vector< std::vector< Block > > & block_matrix = _zone_vector[zindex].block_matrix;
      for( unsigned int cut = 0; cut < block_matrix.size(); ++cut )
        {
        std::vector< Block > & block_vector = block_matrix[cut];
        ignore_small_blocks( block_vector );
        order_blocks( block_vector );
        Block::hierarchize_blocks( block_vector );
        ignore_abnormal_blocks( block_vector );
        remove_top_bottom_noise( block_vector );
        remove_left_right_noise( block_vector );
        }
      }

  for( int zindex = 0; zindex < zones(); ++zindex )
    {
    std::vector< std::vector< Block > > & block_matrix = _zone_vector[zindex].block_matrix;
    for( int cut = block_matrix.size() - 1; cut >= 0; --cut )
      if( !block_matrix[cut].size() )
        block_matrix.erase( block_matrix.begin() + cut );
    }

  for( int zindex = zones() - 1; zindex >= 0; --zindex )
    if( !_zone_vector[zindex].block_matrix.size() )
      _zone_vector.erase( _zone_vector.begin() + zindex );
  }


int Blockmap::blocks_in_map() const throw()
  {
  int sum = 0;
  for( int zindex = 0; zindex < zones(); ++zindex )
    sum += blocks_in_zone( zindex );
  return sum;
  }


int Blockmap::blocks_in_zone( int zindex ) const throw()
  {
  const std::vector< std::vector< Block > > & block_matrix = _zone_vector[zindex].block_matrix;
  int sum = 0;
  for( int cut = 0; cut < (int)block_matrix.size(); ++cut )
    sum += blocks_in_cut( zindex, cut );
  return sum;
  }


void Blockmap::print( FILE * outfile, int debug_level ) const throw()
  {
  std::fprintf( outfile, "blockmap size %dw x %dh\n", _width, _height );
  std::fprintf( outfile, "total zones in map %d\n", zones() );
  std::fprintf( outfile, "total blocks in map %d\n\n", blocks_in_map() );
  for( int zindex = 0; zindex < zones(); ++zindex )
    {
    const Rectangle & r = _zone_vector[zindex].rectangle;
    const std::vector< std::vector< Block > > & block_matrix = _zone_vector[zindex].block_matrix;

    std::fprintf( outfile, "zone %d of %d\n", zindex + 1, zones() );
    std::fprintf( outfile, "zone size %dw x %dh\n", r.width(), r.height() );
    std::fprintf( outfile, "total cuts in zone %d\n", block_matrix.size() );
    std::fprintf( outfile, "total blocks in zone %d\n\n", blocks_in_zone( zindex ) );

    int sp = (debug_level & 1) ? 0 : -1;
    for( unsigned int cut = 0; cut < block_matrix.size(); ++cut )
      {
      std::fprintf( outfile, "cut %d blocks %d\n", cut + 1, block_matrix[cut].size() );
      for( unsigned int i = 0; i < block_matrix[cut].size(); ++i )
      block_matrix[cut][i].print( outfile, sp );
      }
    }
  }
