/*  ocrad - Optical Character Recognition program
    Copyright (C) 2003 Antonio Diaz Diaz.

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include <cstdio>
#include <list>
#include <map>
#include <vector>
#include "common.h"
#include "rectangle.h"
#include "block.h"
#include "blockmap.h"
#include "character.h"
#include "textline.h"
#include "textpage.h"


Textpage::Textpage( const Blockmap & blockmap ) throw()
  {
  // First pass. Assign blocks to characters and create lines.
  {
  int current_line = 0, prev_left = 0;
  bool begin_of_line = true;
  for( std::list< Block >::const_iterator p1 = blockmap.block_list().begin();
       p1 != blockmap.block_list().end(); ++p1 )
    {
    if( p1->height() * p1->width() < 4 ) continue;
    if( p1->height() / p1->width() > 20 ) continue;
    if( p1->width() / p1->height() > 20 ) continue;
    Character c( *p1 );
    if( !begin_of_line && c.right() < prev_left ) begin_of_line = true;
    if( begin_of_line )
      {
      begin_of_line = false;
      add_line();
      current_line = lines() - 1;
      }
    line( current_line ).add_character( c );
    prev_left = c.left();
    }
  }

  // Second pass. Follow every character in line with its right neighbor.
  for( int current_line = 0; current_line < lines() - 1; )
    {
    bool joined = false;
    Textline & line1 = line( current_line );
    Textline & line2 = line( current_line + 1 );
    if( line1.character( 0 ).right() < line2.character( 0 ).left() )
      {
      int i = 0, mh2 = line2.mean_height() / 2;
      for( i = 0; i < line2.characters() && line2.character( i ).height() < mh2; ++i )
        ;
      Character & c2 = line2.character( i );
      for( i = line1.characters(); i > 0; )
        {
        Character & c1 = line1.character( --i );
        if( c1.hcenter() < c2.left() )
          {
          if( c1.includes_vcenter( c2 ) || c2.includes_vcenter( c1 ) )
            {
            line1.join( line2 ); delete_line( current_line + 1 );
            joined = true;
            }
          break;
          }
        }
      }
    if( !joined ) ++current_line;
    }

  // Second pass bis. Follow every character in line with its right neighbor.
  for( int current_line = 0; current_line < lines() - 2; )
    {
    bool joined = false;
    Textline & line1 = line( current_line );
    Textline & line2 = line( current_line + 2 );
    if( line1.character( 0 ).right() < line2.character( 0 ).left() )
      {
      int i, mh2 = line2.mean_height() / 2;
      for( i = 0 ; i < line2.characters() && line2.character( i ).height() < mh2; ++i )
        ;
      Character & c2 = line2.character( i );
      for( i = line1.characters(); i > 0; )
        {
        Character & c1 = line1.character( --i );
        if( c1.hcenter() < c2.left() )
          {
          if( c1.includes_vcenter( c2 ) || c2.includes_vcenter( c1 ) )
            {
            line1.join( line( current_line + 1 ) );
            line1.join( line2 );
            delete_line( current_line + 2 );
            delete_line( current_line + 1 );
            joined = true;
            }
          break;
          }
        }
      }
    if( !joined ) ++current_line;
    }

  // Third pass. Join lines of i-dots and tildes.
  for( int current_line = 0; current_line < lines() - 1; )
    {
    bool joined = false;
    Textline & line1 = line( current_line );
    Textline & line2 = line( current_line + 1 );
    if( line1.mean_height() * 2 < line2.mean_height() )
      for( int i1 = 0 ; !joined && i1 < line1.characters(); ++i1 )
        {
        Character & c1 = line1.character( i1 );
        if( c1.height() * 2 >= line2.mean_height() ) continue;
        for( int i2 = 0 ; !joined && i2 < line2.characters(); ++i2 )
          {
          Character & c2 = line2.character( i2 );
          if( c2.right() < c1.left() ) continue;
          if( c2.left() > c1.right() ) break;
          if( ( c2.includes_hcenter( c1 ) || c1.includes_hcenter( c2 ) )
              && c2.top() - c1.bottom() < line2.mean_height() )
            {
            line1.join( line2 ); delete_line( current_line + 1 );
            joined = true;
            }
          }
        }
    if( !joined ) ++current_line;
    }

  // Fourth pass. Build the vertical composite characters.
/*  for( int current_line = 0; current_line < lines(); ++current_line )
    {
    Textline & line1 = line( current_line );
    for( int i = 0 ; i < line1.characters() - 1; )
      {
      int best = i;
      for( int j = i + 1 ; j < line1.characters(); ++j )
        if( line1.character( j ).left() <= line1.character( best ).right() )
          best = j;
      if( best != i )
        {
        Character & c1 = line1.character( i );
        Character & c2 = line1.character( best );
        if( c1.includes_hcenter( c2 ) || c2.includes_hcenter( c1 ) )
          {
          if( c1.vcenter() >= c2.vcenter() ) c1.join( c2 );
          else { c2.join( c1 ); c1 = c2; }
          line1.delete_character( best );
          }
        else
          {
          if( c2.hcenter() < c1.hcenter() ) line1.swap_characters( i, best );
          else ++i;
          }
        }
      else ++i;
      }
    } */

  // Fourth pass. Build the vertical composite characters.
  for( int current_line = 0; current_line < lines(); ++current_line )
    {
    Textline & line1 = line( current_line );
    for( int i = 0 ; i < line1.characters() - 1; )
      {
      Character & c1 = line1.character( i );
      bool joined = false;
      for( int j = i + 1 ; j < line1.characters(); ++j )
        {
        Character & c2 = line1.character( j );
        if( c1.includes_hcenter( c2 ) || c2.includes_hcenter( c1 ) )
          {
          if( c1.vcenter() >= c2.vcenter() ) c1.join( c2 );
          else { c2.join( c1 ); c1 = c2; }
          line1.delete_character( j );
          joined = true; break;
          }
        }
      if( !joined ) ++i;
      }
    }

  // Fourth pass bis. Order characters.
  for( int current_line = 0; current_line < lines(); ++current_line )
    {
    Textline & line1 = line( current_line );
    for( int i = 0 ; i < line1.characters() - 1; ++i )
      {
      int best = i;
      for( int j = i + 1 ; j < line1.characters(); ++j )
        if( line1.character( j ).hcenter() < line1.character( best ).hcenter() )
          best = j;
      if( best != i ) line1.swap_characters( i, best );
      }
    }

  // Fifth pass. Add spaces between characters.
  for( int current_line = 0; current_line < lines(); ++current_line )
    {
    Textline & line1 = line( current_line );
    int mw = line1.mean_width();
    int mg = line1.mean_gap_width();
    if( mw < 2 || mg < 1 ) continue;
    for( int i = 0 ; i < line1.characters() - 1; ++i )
      {
      Character & c1 = line1.character( i );
      Character & c2 = line1.character( i + 1 );
      int dist = c2.left() > c1.right() ? c2.left() - c1.right() - 1 : 0;
      if( dist >= mw || ( dist > (mw / 3) && dist > (mg * 2) ) )
        {
        int top = ( c1.top() + c2.top() ) / 2;
        int bottom = ( c1.bottom() + c2.bottom() ) / 2;
        for( int j = c1.right(); j + mw <= c2.left() || j == c1.right(); j += mw )
          {
          int right = j + mw - 1;
          if( right >= c2.left() ) right = c2.left() - 1;
          Character c( Rectangle( j + 1, top, right, bottom ), blockmap );
          c.add_guess( ' ', 100 );
          line1.insert_character( ++i, c );
          }
        }
      }
    }

  // Sixth pass. Add blank lines.
  if( lines() >= 3 )
    {
    int mean_vdistance = (line(lines()-1).mean_vcenter() - line(0).mean_vcenter()) / (lines() - 1);
    if( mean_vdistance > 0 )
      for( int current_line = 0; current_line + 1 < lines(); ++current_line )
        {
        Textline & line1 = line( current_line );
        Textline & line2 = line( current_line + 1 );
        int vdistance = line2.mean_vcenter() - line1.mean_vcenter();
        if( vdistance <= 0 ) continue;
        int newlines = (vdistance + (mean_vdistance / 2)) / mean_vdistance;
        for( int i = 1; i < newlines; ++i ) insert_line( ++current_line );
        }
    }
  }


void Textpage::insert_line( const int i ) throw( Internal_error )
  {
  if( i < 0 || i > lines() ) throw Internal_error( "insert_line index too big" );
  data.insert( data.begin() + i, Textline() );
  }


void Textpage::delete_line( const int i ) throw( Internal_error )
  {
  if( i < 0 || i > lines() ) throw Internal_error( "delete_line index too big" );
  data.erase( data.begin() + i );
  }


Textline & Textpage::line( const int i ) const throw( Internal_error )
  {
  if( i < 0 || i > lines() ) throw Internal_error( "line index too big" );
  return data[i];
  }


void Textpage::print( FILE * outfile, const bool graph, const bool recursive )
								const throw()
  {
  if( graph || recursive ) fprintf( outfile, "%d lines\n\n", lines() );

  for( int i = 0; i < lines(); ++i )
    {
    if( graph || recursive )
      fprintf( outfile, "%d characters in line %d\n", line(i).characters(), i+1 );
    line( i ).print( outfile, graph, recursive );
    }
  fputs( "\n", outfile );
  }
