/*  ocrad - Optical Character Recognition program
    Copyright (C) 2003 Antonio Diaz Diaz.

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include <cstdio>
#include <list>
#include <map>
#include <vector>
#include "common.h"
#include "rectangle.h"
#include "block.h"
#include "blockmap.h"
#include "profile.h"
#include "features.h"
#include "character.h"


// First attempt at recognition without relying on context.
void Character::recognize1( int charbox_vcenter ) throw()
  {
  if( _block_list.size() == 1 ) recognize11( charbox_vcenter );
  else if( _block_list.size() == 2 ) recognize12();
  else if( _block_list.size() == 3 ) recognize13();
  }


// Recognizes 1 block characters.
// 0123456789ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghklmnopqrstuvwxyz
// #$&'()*+,-./<>@[\]^_`{|}~
void Character::recognize11( int charbox_vcenter ) throw()
  {
  const Block & b = _block_list.front();
  if( b.block_list().size() == 0 ) recognize110( charbox_vcenter );
  else if( b.block_list().size() == 1 ) recognize111();
  else if( b.block_list().size() == 2 ) recognize112();
  }


// Recognizes 1 block characters without holes.
// 12357CEFGHIJKLMNSTUVWXYZcfhklmnrstuvwxyz
// '()*+,-./<>@[\]^_`{|}~
void Character::recognize110( int charbox_vcenter ) throw()
  {
  const Block & b = _block_list.front();
  char ch = solid_type( b );
  if( !ch && iscomma() ) ch = ',';
  if( ch == ',' && b.bottom() <= charbox_vcenter ) ch = '\'';
  else if( ch == '\'' )
    {
    if( b.top() >= charbox_vcenter ) ch = ',';
    else if( b.top() + b.width() <= charbox_vcenter &&
             b.bottom() - b.width() >= charbox_vcenter ) ch = 'I';
    }
  if( ch ) { add_guess( ch, 0 ); return; }

  Features f( b );

  if( f.tp.minima( b.height() / 4 ) == 1 &&
      f.bp.minima( b.height() / 4 ) == 1 &&
      f.vbars() == 1 && f.vbar(0).width() >= 2 )
    {
    if( f.vbar(0).width() * 3 < b.width() )
      {
      if( abs( f.vbar(0).hcenter() - b.hcenter() ) <= 1 )
        {
        if( f.hbars() == 1 )
          {
          if( f.hbar(0).top() <= b.top() + 1 ) { add_guess( 'T', 0 ); return; }
          if( abs( f.hbar(0).vcenter() - b.vcenter() ) <= 1 )
            { add_guess( '+', 0 ); return; }
          }
        if( f.hbars() == 2 && f.hbar(0).top() <= b.top() + 1 &&
            f.hbar(0).width() > f.hbar(1).width() )
          { add_guess( 'T', 0 ); return; }
        }
      if( f.vbar(0).right() < b.hcenter() )
        {
        if( f.hbars() == 3 && f.hbar(0).top() <= b.top() + 1 &&
            f.hbar(0).width() > f.hbar(1).width() )
          {
          if( similar( f.hbar(0).width(), f.hbar(2).width(), 10 ) &&
              f.hbar(2).width() > f.hbar(1).width() )
            { add_guess( 'E', 0 ); return; }
          if( f.hbar(0).width() > f.hbar(2).width() &&
              f.hbar(1).includes_vcenter( b ) )
            { add_guess( 'F', 0 ); return; }
          }
        if( f.hbars() == 2 )
          {
          if( f.hbar(0).top() <= b.top() + 1 &&
              f.hbar(0).width() > f.hbar(1).width() &&
              f.hbar(1).includes_vcenter( b ) )
            { add_guess( 'F', 0 ); return; }
          if( f.hbar(1).bottom() >= b.bottom() - 1 &&
              b.height() > b.width() &&
              f.hbar(1).width() > f.hbar(0).width() &&
              abs( f.vbar(0).hcenter() - f.hbar(0).hcenter() ) <= 1 )
            { add_guess( 'L', 0 ); return; }
          }
        if( f.hbars() == 1 && f.hbar(0).bottom() >= b.bottom() - 1 &&
            similar( f.hbar(0).width(), b.width(), 10 ) )
          { add_guess( 'L', 0 ); return; }
        }
      if( f.vbar(0).left() > b.hcenter() )
        {
        if( f.hbars() == 1 && f.hbar(0).top() <= b.top() + 1 &&
            f.hbar(0).width() + 1 >= b.width() )
          { add_guess( '', 0 ); return; }
        add_guess( 'J', 0 ); return;
        }
      }
    if( abs( f.vbar(0).hcenter() - b.hcenter() ) <= 1 )
      {
      if( f.rp.istip() ) { add_guess( 'I', 0 ); return; }
      if( f.lp.istip() ) { add_guess( 'l', 0 ); return; }
      for( int i = f.vbar(0).left() - 1; i > b.left(); --i )	// FIXME
        if( f.seek_bottom( (b.vcenter() + b.bottom()) / 2, i ) < b.bottom() )
          { add_guess( 'l', 0 ); return; }
//      add_guess( 'l', 0 ); return;
      }
    if( f.vbar(0).hcenter() < b.hcenter() )
      {
      if( similar( b.height(), b.width(), 40 ) )
        {
        if( f.escape_bottom( b.vcenter(), b.right() - f.rp[b.vcenter() - b.top()] + 1 ) )
          {
          if( f.rp.minima() == 3 ) { add_guess( 'f', 0 ); return; }
          add_guess( 'r', 0 ); return;
          }
        if( !f.escape_top( b.vcenter(), b.right() - f.rp[b.vcenter() - b.top()] + 1 ) )
          { add_guess( 'c', 0 ); return; }
        if( f.lp.isvpit() ) { add_guess( 't', 0 ); return; }
        }
      else if( f.vbar(0).left() > b.left() )
        {
//        ch = f.test_ft();
//        if( ch ) { add_guess( ch, 0 ); return; }
        if( f.escape_bottom( b.vcenter(), b.right() - f.rp[b.vcenter() - b.top()] + 1 ) )
          { add_guess( 'f', 0 ); return; }
//        if( ist() ) { add_guess( 't', 0 ); return; }
        else { add_guess( 't', 0 ); return; }
        }
      else if( f.seek_bottom( b.vcenter(), b.hcenter() ) >= b.bottom() )
        {
        if( f.rp.minima() == 2 ) { add_guess( 'f', 0 ); return; }
        add_guess( 'r', 0 ); return;
        }
      }
    if( f.vbar(0).left() <= b.left() + 1 && b.height() > 2 * b.width() &&
        f.rp.istip() ) { add_guess( '[', 0 ); return; }
    if( f.vbar(0).right() >= b.right() - 1 && b.height() > 2 * b.width() &&
        f.lp.istip() ) { add_guess( ']', 0 ); return; }
    }
  ch = f.test_235sz();
  if( ch ) { add_guess( ch, 0 ); return; }
  if( f.lp.isconvex() )
    {
    if( b.height() > 2 * b.width() )
      {
      if( b.bottom() <= charbox_vcenter ) { add_guess( '`', 0 ); return; }
      add_guess( '(', 0 ); return;
      }
    int row = f.seek_bottom( b.vcenter(), b.hcenter() );
    if( row < b.bottom() && f.escape_right( b.vcenter(), b.hcenter() ) &&
        !f.escape_bottom( row, b.hcenter() ) )
      {
      int ru, rd = -1;
      for( int i = row - 1; i > b.top(); --i )
        {
        if( rd < 0 )
          { if( f.seek_right( i, b.hcenter() ) >= b.right() ) rd = i + 2; }
        else if( f.seek_right( i, b.hcenter() ) < b.right() )
          { ru = i - 1;
          if( rd >= row ) break;
          if( rd - 1 <= b.vcenter() ||
              f.seek_right( ru, b.hcenter() ) - f.seek_right( rd, b.hcenter() ) > 2 )
            { add_guess( 'G', 0 ); return; }
          }
        }
      }
    if( 2 * b.height() >= 3 * b.width() && b.bottom() <= charbox_vcenter )
      { add_guess( '`', 0 ); return; }
    add_guess( 'c', 0 ); return;
    }
  if( f.tp.minima( b.height() / 4 ) == 2 )
    {
    if( f.tp.istip() )
      {
      if( f.bp.isconvex() )
        {
//        int dummy;
//        if( f.lp.straight( dummy ) ) { add_guess( 'v', 0 ); return; }
        if( f.bp.isvpit() ) { add_guess( 'v', 0 ); return; }
        add_guess( 'U', 0 ); return;
        }
      if( f.bp.ispit() )
        {
//        if( f.rp.isconvex() || ( f.lp.istip() && !f.rp.istip() ) ) // FIXME
//        if( f.rp.increasing() && !f.lp.increasing() ) // FIXME
//          { add_guess( 'y', 0 ); return; }
        int lg = f.seek_right( b.bottom() - 1, b.left() ) - b.left();
        int rg = b.right() - f.seek_left( b.bottom() - 1, b.right() );
        if( 2 * lg < rg ) { add_guess( 'y', 0 ); return; }
        if( 2 * f.tp.range() <= b.height() ||
            ( 7 * f.tp.range() < 4 * b.height() && !f.rp.increasing() ) )
          { add_guess( 'Y', 0 ); return; }
        if( f.bp.isvpit() ) { add_guess( 'v', 0 ); return; }
        add_guess( 'u', 0 ); return;
        }
      }
    if( f.tp.isctip() )
      {
      if( f.lp.isflats() && f.bp.isctip() )
        {
        if( f.vbars() >= 2 && f.rp.isflats() )
          {
          if( f.hbars() == 1 && abs( f.hbar(0).vcenter() - b.vcenter() ) <= 1 &&
              abs( f.hbar(0).hcenter() - b.hcenter() ) <= 1 )
            { add_guess( 'H', 0 ); return; }
          if( f.tp.istip() &&
              ( f.bp.minima() == 3 || f.vscan()[f.tp.range()-1] == 4 ) )
            { add_guess( 'M', 0 ); return; }
          add_guess( 'N', 0 ); return;
          }
        if( f.rp.isctip() ) { add_guess( 'K', 0 ); return; }
        }
      if( f.lp.istip() && !f.rp.istip() ) { add_guess( 'y', 0 ); return; }
      if( f.escape_top( b.vcenter(), b.hcenter() ) )
        { add_guess( 'u', 0 ); return; }
      }
    }
  if( f.bp.minima() == 2 )
    {
    if( f.lp.istip() && f.rp.istip() ) { add_guess( 'x', 0 ); return; }
    if( f.tp.minima() == 3 ) { add_guess( 'w', 0 ); return; }
    if( f.tp.minima() == 1 && f.rp.isctip() ) { add_guess( 'k', 0 ); return; }
    if( ( ch = f.test_hn() ) ) { add_guess( ch, 0 ); return; }
    }
  if( f.bp.minima() == 3 ) { add_guess( 'm', 0 ); return; }

  if( f.rp.isconvex() && b.height() > 2 * b.width() )
    { add_guess( ')', 0 ); return; }

  ch = f.test_line();
  if( ch ) { add_guess( ch, 0 ); return; }

  if( f.hbars() == 1 && f.hbar(0).top() <= b.top() + (b.height() / 10) &&
      5 * f.hbar(0).width() >= 4 * b.width() &&
      f.rp.increasing( f.hbar(0).vcenter() - b.top() ) )
    { add_guess( '7', 0 ); return; }
  if( f.rp.istip() && f.lp.ispit() ) { add_guess( '{', 0 ); return; }
  if( f.lp.istip() && f.rp.ispit() ) { add_guess( '}', 0 ); return; }
  }


bool Character::iscomma() const throw()
  {
  const Block & b = _block_list.front();
  if( b.block_list().size() > 0 ) return 0;
  if( 4 * b.height() < 5 * b.width()) return false;
  if( b.height() > 3 * b.width() ) return false;

  if( b.width() >= 3 && b.height() >= 3 )
    {
    int upper_area = 0;
    for( int row = b.top(); row < b.top() + b.width(); ++row )
      for( int col = b.left(); col <= b.right(); ++col )
        if( b.blockmap()->id( row, col ) == b.id() ) ++upper_area;
    if( upper_area < (b.width() - 2) * (b.width() - 2) ) return false;
    int count1 = 0, count2 = 0;
    for( int col = b.left(); col <= b.right(); ++col )
      { if( b.blockmap()->id( b.top() + 1, col ) == b.id() ) ++count1;
      if( b.blockmap()->id( b.bottom() - 1, col ) == b.id() ) ++count2; }
    if( count1 <= count2 ) return false;
    }
  return true;
  }


bool Character::ist() const throw()
  {
  const Block & b = _block_list.front();
  int area1 = 0, area2 = 0;

  for( int row = b.top(); row < b.vcenter(); ++row )
    for( int col = b.hcenter(); col <= b.right(); ++col )
      if( b.blockmap()->id( row, col ) == b.id() ) ++area1;

  for( int row = b.vcenter() + 1; row <= b.bottom(); ++row )
    for( int col = b.hcenter(); col <= b.right(); ++col )
      if( b.blockmap()->id( row, col ) == b.id() ) ++area2;

  if( area2 > area1 ) return true;
  return false;
  }


char Character::solid_type( const Block & b ) const throw()
  {
  if( b.block_list().size() > 0 ) return 0;
  int inner_area;

  if( b.width() >= 3 && b.height() >= 3 )
    {
    inner_area = 0;
    for( int row = b.top() + 1; row < b.bottom(); ++row )
      for( int col = b.left() + 1; col < b.right(); ++col )
        if( b.blockmap()->id( row, col ) == b.id() ) ++inner_area;
    if( inner_area * 100 < (b.width() - 2) * (b.height() - 2) * 75 ) return 0;
    }
  else inner_area = b.area();

  if( similar( b.height(), b.width(), 20 ) ) return '.';
  if( inner_area * 100 < (b.width() - 2) * (b.height() - 2) * 85 ) return 0;
  if( b.width() >= 6 * b.height() ) return '_';
  if( b.width() > b.height() ) return '-';
  if( b.height() >= 9 )
    {
    int l = 0, r = 0;
    for( int row = b.vcenter() - 1; row <= b.vcenter() + 1; ++row )
      { if( b.blockmap()->id( row, b.left() ) == b.id() ) ++l;
      if( b.blockmap()->id( row, b.left() ) == b.id() ) ++r; }
    if( l <= 1 && r <= 1 )
      {
      bool lt = false, lb = false, rt = false, rb = false;
      for( int row = b.top(); row <= b.top() + 2; ++row )
        { if( b.blockmap()->id( row, b.left() ) == b.id() ) lt = true;
        if( b.blockmap()->id( row, b.right() ) == b.id() ) rt = true; }
      for( int row = b.bottom() - 2; row <= b.bottom(); ++row )
        { if( b.blockmap()->id( row, b.left() ) == b.id() ) lb = true;
        if( b.blockmap()->id( row, b.right() ) == b.id() ) rb = true; }
      if( lt && lb && rb ) { if( rt ) return 'I'; else return 'l'; }
      }
    }
  if( b.height() >= 5 * b.width() ) return '|';
  if( b.height() > b.width() ) return '\'';
  return 0;
  }


// Recognizes 1 block characters with 1 hole.
// 0469ADOPQRabdegopq#
void Character::recognize111() throw()
  {
  const Block & b = _block_list.front();
  const Block & h = b.block_list().front();
  Features f( b );
  Features fh( h );

  if( similar( h.top() - b.top(), b.bottom() - h.bottom(), 40 ) )
    {
    if( f.lp.isflats() ) { add_guess( 'D', 0 ); return; }
    if( !f.lp.isconvex() && !f.rp.isconvex() )
      {
      if( f.tp.minima() == 2 ) { add_guess( '#', 0 ); return; }
      if( f.tp.minima() == 1 && f.bp.minima() == 1 )
        {
        if( f.bp.isconvex() ) { add_guess( '', 0 ); return; }
        else { add_guess( '4', 0 ); return; }
        }
      add_guess( 'A', 0 ); return;
      }
    if( similar( h.left() - b.left(), b.right() - h.right(), 40 ) )
      {
      if( b.width() > 3 * h.width() && f.vscan()[b.vcenter() - b.top()] == 4 )
        { add_guess( '@', 0 ); return; }
//      if( h.block_list().size() ) { add_guess( '0', 0 ); return; }
      if( b.width() >= b.height() || similar( b.height(), b.width(), 21 ) )
        { add_guess( 'o', 0 ); return; }
      add_guess( 'O', 0 ); return;
      }
    add_guess( 'a', 0 ); return;
    }
  if( h.top() - b.top() < b.bottom() - h.bottom() )
    {
    if( f.vbars() == 1 && f.vbar(0).hcenter() > b.hcenter() &&
        fh.tp.decreasing() ) { add_guess( '4', 0 ); return; }
    if( f.rp.isconvex() && f.tp.isconvex() && f.rp.minima() == 1 )
      { add_guess( '9', 0 ); return; }
    if( f.rp.range() > b.width() / 2 && ( f.lp.isconvex() || f.bp.isconvex() ) )
      { add_guess( 'e', 0 ); return; }
    if( f.bp.minima() == 2 )
      {
      if( f.tp.isvpit() ) { add_guess( 'A', 0 ); return; }
      add_guess( 'R', 0 ); return;
      }
    if( b.left() + f.bp.iminimum() < h.hcenter() ) { add_guess( 'p', 0 ); return; }
    for( int row = b.bottom(); row > b.bottom() - (b.height() / 4); --row )
      if( blockmap()->id( row, h.left() ) == b.id() )
        {
        if( f.hscan()[h.left() - b.left() + 2] == 2 )
          { add_guess( 'Q', 0 ); return; }
        add_guess( 'g', 0 ); return;
        }
    add_guess( 'q', 0 ); return;
    }
  if( h.top() - b.top() > b.bottom() - h.bottom() )
    {
    if( f.lp.isconvex() && f.bp.isconvex() ) { add_guess( '6', 0 ); return; }
    int i = f.tp.iminimum();
    if( b.left() + i < h.left() ) { add_guess( 'b', 0 ); return; }
    if( b.left() + i > h.right() )
      {
//      if( !f.vbars() ) { add_guess( '', 0 ); return; }
      if( f.seek_right( h.top() - (b.bottom() - h.bottom()) - 1, h.right() ) >= b.right() )
        { add_guess( '', 0 ); return; }
      add_guess( 'd', 0 ); return;
      }
    for( int row = b.top(); row < b.top() + (b.height() / 4); ++row )
      if( blockmap()->id( row, h.left() ) == b.id() )
        { add_guess( 'a', 0 ); return; }
    if( f.seek_right( h.top() - (b.bottom() - h.bottom()) - 1, h.right() ) >= b.right() )
      { add_guess( '', 0 ); return; }
    add_guess( 'd', 0 ); return;
    }
  }


// Recognizes 1 block characters with 2 holes.
// 8BQg$&
void Character::recognize112() throw()
  {
  const Block & b = _block_list.front();
  const Block & b1 = b.block_list().front();		// upper hole
  const Block & b2 = b.block_list().back();		// lower hole
  int a1 = b1.area();
  int a2 = b2.area();
//  Features f( b );

  Profile lp( b, Profile::left );
  if( similar( a1, a2, 50 ) )		// I don't like this
    {
    if( lp.isflats() ) { add_guess( 'B', 0 ); return; }
    if( b1.right() < b2.left() ) { add_guess( '$', 0 ); return; }
    if( b.hcenter() > b1.hcenter() && b.hcenter() > b2.hcenter() &&
        ( b.hcenter() >= b1.right() || b.hcenter() >= b2.right() ) )
      { add_guess( '&', 0 ); return; }
    for( int row = b1.bottom() + 1; row < b2.top(); ++row )
      if( blockmap()->id( row, hcenter() ) == 0 ) { add_guess( 'g', 0 ); return; }
    add_guess( '8', 0 ); return;
    }
  if( a1 > a2 ) { add_guess( 'Q', 0 ); return; }
  add_guess( '&', 0 );
  }


// Recognizes 2 block characters.
// ij!%:;=?|
void Character::recognize12() throw()
  {
  const Block & b1 = _block_list.front();		// lower block
  const Block & b2 = _block_list.back();		// upper block
  int a1 = b1.area();
  int a2 = b2.area();
  Features f1( b1 );
  Features f2( b2 );

  if( similar( a1, a2, 10 ) )
    {
    if( width() > height() || similar( width(), height(), 50 ) )
      { add_guess( '=', 0 ); return; }
    if( b2.height() >= 2 * b2.width() ) { add_guess( '|', 0 ); return; }
    add_guess( ':', 0 ); return;
    }
  if( similar( a1, a2, 60 ) )
    {
    if( solid_type( b2 ) == '.' && b1.height() > b2.height() )
      { add_guess( ';', 0 ); return; }
    char ch = solid_type( b1 );
    if( ch == '-' || ch == '_' ) { add_guess( '', 0 ); return; }
    add_guess( '%', 0 ); return;
    }
  if( a1 > a2 )
    {
    if( solid_type( b2 ) == '.' )
      {
      if( b1.hcenter() <= b2.left() )
        {
        if( f1.bp.minima( b1.height() / 4 ) == 1 ) { add_guess( 'j', 0 ); return; }
        return;
        }
      if( similar( b1.width(), b2.width(), 20 ) && f1.wp[1] < b2.width() )
        { add_guess( '', 0 ); return; }
      if( f1.vbars() == 1 && f1.vbar(0).width() >= 2 )
        { add_guess( 'i', 0 ); return; }
      add_guess( '', 0 ); return;
      }
    const char accent[] = "";
    int slope, atype = 0;
    if( f2.tp.ispit() ) atype = 2;
    else if( f2.rp.straight( slope ) && slope < 0 ) atype = 1;
    int ltype = -1;
    Character c( b1 );
    c.recognize1( c.vcenter() );
    if( c.guess_map().size() )
      switch( c.guess_map().begin()->first )
        {
        case 'A': ltype = 0; break;
        case 'E': ltype = 1; break;
        case 'I': ltype = 2; break;
        case 'O': ltype = 3; break;
        case 'V':
        case 'U': ltype = 4; break;
        case 'a': ltype = 5; break;
        case 'e': ltype = 6; break;
        case 'l':
        case 'i': ltype = 7; break;
        case 'o': ltype = 8; break;
        case 'v':
        case 'u': ltype = 9; break;
        case 'N': add_guess( '', 0 ); return;
        case 'n': add_guess( '', 0 ); return;
        default : ltype = 7; break;
        }
    if( ltype >= 0 ) { add_guess( accent[(3*ltype)+atype], 0 ); return; }
    if( atype == 0 ) add_guess( '\'', 0 );
    else if( atype == 1 ) add_guess( '`', 0 );
    else if( atype == 2 ) add_guess( '^', 0 );
    return;
    }
  char ch = solid_type( b1 );
  if( ch == '.' )
    {
    if( similar( b1.width(), b2.width(), 50 ) ) { add_guess( '!', 0 ); return; }
    add_guess( '?', 0 ); return;
    }
  if( ch == '-' || ch == '_' )
    {
    if( b2.block_list().size() == 1 )
      {
      const Block & h = b2.block_list().front();
      if( similar( h.left() - b2.left(), b2.right() - h.right(), 40 ) )
        { add_guess( '', 0 ); return; }
      add_guess( '', 0 ); return;
      }
    }
  }


// Recognizes 3 block characters.
// %
void Character::recognize13() throw()
  {
  const Block & b1 = _block_list.front();
  if( solid_type( b1 ) == '.' ) add_guess( '', 0 );
  Character c( b1 );
  c.recognize1( c.vcenter() );
  if( c.guess_map().size() )
    switch( c.guess_map().begin()->first )
      {
      case 'A': add_guess( '', 0 ); return;
      case 'E': add_guess( '', 0 ); return;
      case 'I': add_guess( '', 0 ); return;
      case 'O': add_guess( '', 0 ); return;
      case 'V':
      case 'U': add_guess( '', 0 ); return;
      case 'a': add_guess( '', 0 ); return;
      case 'e': add_guess( '', 0 ); return;
      case 'l':
      case 'i': add_guess( '', 0 ); return;
      case 'o': add_guess( '', 0 ); return;
      case 'v':
      case 'u': add_guess( '', 0 ); return;
      }
  }
