/*  ocrad - Optical Character Recognition program
    Copyright (C) 2003 Antonio Diaz Diaz.

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include <cstdio>
#include <list>
#include <map>
#include <vector>
#include "common.h"
#include "rectangle.h"
#include "block.h"
#include "blockmap.h"
#include "character.h"


// Return the area of the main blocks only (no recursive)
int Character::area() const throw()
  {
  int area = 0;
  std::list< Block >::const_iterator p = _block_list.begin();
  for( ; p != _block_list.end(); ++p ) area += p->area();
  return area;
  }


void Character::add_guess( const unsigned char ch, int confi ) throw()
  {
  if( confi < -100 ) confi = -100;
  else if( confi > 100 ) confi = 100;
  _guess_map[ ch ] = confi;
  }


void Character::only_guess( const unsigned char ch, int confi ) throw()
  { _guess_map.clear(); add_guess( ch, confi ); }


void Character::join( Character & c ) throw( Internal_error )
  {
  if( _blockmap != c._blockmap )
      throw Internal_error( "can't join characters of different blockmaps.\n" );
  std::list< Block >::const_iterator p = c._block_list.begin();
  for( ; p != c._block_list.end(); ++p )
    add_rectangle( *p );
  _block_list.splice( _block_list.end(), c._block_list );
  }


void Character::print( FILE * outfile, const bool graph, const bool recursive )
								const throw()
  {
  if( graph || recursive )
    {
    fprintf( outfile, "%d guesses    ", _guess_map.size() );
    std::map< unsigned char, int >::const_iterator p = _guess_map.begin();
    for( ; p != _guess_map.end(); ++p )
      fprintf( outfile, "guess '%c', confidence %d    ", p->first, p->second );
    fprintf( outfile, "\n" );
    }
  if( graph )
    {
    fprintf( outfile, "height = %d, width = %d, vcenter = %d, hcenter = %d, black area = %d%%\n",
		height(), width(), vcenter(), hcenter(), (area() * 100) / (height() * width()) );
    for( int row = top(); row <= bottom(); ++row)
      {
      for( int col = left(); col <= right(); ++col)
        {
        char ch = (row == vcenter() && col == hcenter()) ? '-' : '';
        if( _blockmap->id( row, col ) > 0 )
          {
          std::list< Block >::const_iterator p = _block_list.begin();
          for( ; p != _block_list.end(); ++p )
            if( p->compare_id( _blockmap->id( row, col ), recursive ) )
              { ch = (ch == '-') ? '+' : 'O'; break; }
          }
        fprintf( outfile, " %c", ch );
        }
      fprintf( outfile, "\n" );
      }
    fprintf( outfile, "\n" );
    }
  else
    {
    unsigned char ch = '_';
    int max_confidence = -101;
    std::map< unsigned char, int >::const_iterator p = _guess_map.begin();

    for( ; p != _guess_map.end(); ++p )
      {
      if( recursive ) fprintf( outfile, "%c %4d,    ", p->first, p->second );
      else if( p->second > max_confidence )
        { ch = p->first; max_confidence = p->second; }
      }
    if( recursive ) fputs( "\n", outfile );
    else fputc( ch, outfile );
    }
  }
