/*  ocrad - Optical Character Recognition program
    Copyright (C) 2003 Antonio Diaz Diaz.

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include <cctype>
#include <cstdio>
#include <stack>
#include <vector>
#include "common.h"
#include "rectangle.h"
#include "bitmap.h"


char pbm_getrawbyte( FILE * f )
  {
  int ch = getc( f );

  if( ch == EOF )
    throw Bitmap::Error( "end-of-file reading pbm file.\n" );

  return static_cast< char > (ch);
  }

char pbm_getc( FILE * f )
  {
  char ch;
  bool comment = false;

  do {
    ch = pbm_getrawbyte( f );
    if( ch == '\n' ) comment = false;
    else if( ch == '#' ) comment = true;
    }
  while( comment );
  return ch;
  }

int pbm_getint( FILE * f )
  {
  char ch;
  int i = 0;

  do ch = pbm_getc( f ); while( isspace( ch ) );
  if( !isdigit( ch ) )
    throw Bitmap::Error( "junk in pbm file where an integer should be.\n" );
  do { i = (i * 10) + (ch - '0'); ch = pbm_getc( f ); }
  while( isdigit( ch ) );
  return i;
  }

bool pbm_getbit( FILE * f )
  {
  char ch;

  do ch = pbm_getc( f ); while( isspace( ch ) );

  if( ch == '0' ) return false;
  if( ch == '1' ) return true;
  throw Bitmap::Error( "junk in pbm file where bits should be.\n" );
  }


Bitmap::Bitmap( FILE * f )
  {
  char filetype = 0;

  if( pbm_getrawbyte( f ) == 'P' )
    {
    char ch = pbm_getrawbyte( f );
    if( ch == '1' || ch == '4' ) filetype = ch;
    }
  if( filetype == 0 )
    throw Error( "bad magic number - not a pbm file.\n" );

  _width = pbm_getint( f );
  _height = pbm_getint( f );

  data.resize( _height );
  for( int row = 0; row < _height; ++row )
    {
    data[row].resize( _width );

    if( filetype == '1' )
      {
      for( int col = 0; col < _width; ++col )
        data[row][col] = pbm_getbit( f );
      }
    else
      {
      unsigned char byte = 0, mask = 0;
      for( int col = 0; col < _width; ++col )
        {
        if( mask == 0 ) { mask = 0x80; byte = pbm_getrawbyte( f ); }
        data[row][col] = ( byte & mask ) ? true : false;
        mask >>= 1;
        }
      }
    }
  }


Bitmap::Bitmap( const Bitmap & source, const Rectangle & r )
  {
  if( r.right() >= source._width || r.bottom() >= source._height )
    throw Internal_error( "bad parameter building a Bitmap from another one.\n" );

  _width = r.right() - r.left() + 1;
  _height = r.bottom() - r.top() + 1;

  data.resize( _height );
  for( int row = 0; row < _height; ++row )
    {
    data[row].resize( _width );

    for( int col = 0; col < _width; ++col )
      data[row][col] = source.data[r.top()+row][r.left()+col];
    }
  }


Bitmap::Bitmap( const Bitmap & source, int th, const int scale )
  {
  if( scale < 2 || scale > source._width || scale > source._height ||
      th >= scale * scale )
    throw Internal_error( "bad parameter building a reduced Bitmap.\n" );

  if( th < 0 )
    {
    long dots = 0;
    for( int row = 0; row < source._height; ++row )
      for( int col = 0; col < source._width; ++col )
        if( source.data[row][col] ) ++dots;
    th = (scale * scale * dots) / (source._width * source._height);
    fprintf( stderr, "automatic threshold = %d\n", th );
    }

  _width = source._width / scale;
  if( _width * scale < source._width ) ++_width;
  _height = source._height / scale;
  if( _height * scale < source._height ) ++_height;

  data.resize( _height );
  for( int row = 0; row < _height; ++row )
    {
    data[row].resize( _width );

    for( int col = 0; col < _width; ++col )
      {
      int counter = 0;
      for( int i = 0; i < scale; ++i )
        if( (row * scale) + i < source._height )
          for( int j = 0; j < scale; ++j )
            if( (col * scale) + j < source._width )
              if( source.data[(row * scale) + i][(col * scale) + j] )
                ++counter;
      data[row][col] = ( counter > th ? true : false );
      }
    }
  }


void floodfill4( const Bitmap & source, Bitmap & dest, int row, int col )
  {
  std::stack< int, std::vector< int > > st;

  if( source.data[row][col] == false && dest.data[row][col] == false )
    { dest.data[row][col] = true; st.push(row); st.push(col); }

  while( !st.empty() )
    {
    col = st.top(); st.pop();
    row = st.top(); st.pop();
    if( col > 0 && source.data[row][col-1] == false && dest.data[row][col-1] == false )
      { dest.data[row][col-1] = true; st.push(row); st.push(col-1); }
    if( col < dest._width-1 && source.data[row][col+1] == false && dest.data[row][col+1] == false )
      { dest.data[row][col+1] = true; st.push(row); st.push(col+1); }
    if( row > 0 && source.data[row-1][col] == false && dest.data[row-1][col] == false )
      { dest.data[row-1][col] = true; st.push(row-1); st.push(col); }
    if( row < dest._height-1 && source.data[row+1][col] == false && dest.data[row+1][col] == false )
      { dest.data[row+1][col] = true; st.push(row+1); st.push(col); }
    }
  }


Bitmap::Bitmap( const Bitmap & source, const type t )
  {
  _width = source._width; _height = source._height;

  data.resize( _height );
  for( int row = 0; row < _height; ++row ) data[row].resize( _width );

  if( t == vertical_histogram )
    {
    for( int col = 0; col < _width; ++col )
      for( int row = 0, y = _height; row < _height; ++row )
        if( source.data[row][col] ) data[--y][col] = true;
    }
  else if( t == horizontal_histogram )
    {
    for( int row = 0; row < _height; ++row )
      for( int col = 0, x = 0; col < _width; ++col )
        if( source.data[row][col] ) data[row][x++] = true;
    }
  else if( t == connected_ground )
    {
    int row, col;
    for( row = 0; row < _height; ++row )
      {
      for( col = 0; col < _width; ++col )
        if( source.data[row][col] ) break;
      if( col == _width ) break;
      }
    if( row == _height ) throw Error( "image is not clear enough.\n" );
    floodfill4( source, *this, row, _width/2 );
    }
  else
    throw Internal_error( "type not implemented building a 'special' Bitmap.\n" );
  }


int Bitmap::horizontalify( const bool verbose )
  {
  const int slope_max = _width / 25;
  int slope_best = 0;			// not a true slope
  long mean = 0;
  long long variance_best = 0;		// not a true variance

  if( slope_max == 0 ) return 0;

  for( int row = 0; row < _height; ++row )
    for( int col = 0; col < _width; ++col )
      if( data[row][col] ) ++mean;
  mean /= _height;

  for( int row = 0; row < _height; ++row )
    {
    long c = -mean;
    for( int col = 0; col < _width; ++col )
      if( data[row][col] ) ++c;
    variance_best += ( c * c );
    }

if( verbose )
  {
  fprintf( stderr, "mean = %ld\n", mean );
  fprintf( stderr, "slope   0   variance %Ld\n", variance_best );
  }

  bool try_inc = true, try_dec = true;
  for( int slope = 1; slope <= slope_max; ++slope )
    {
    long long variance = 0;
    int step = _width / slope; if( step * slope < _width ) ++step;

    if( try_inc )
      {
      for( int row = 0; row < _height - slope; ++row )
        {
        long c = -mean; int y = 0;
        for( int col = 0; col < _width; ++col )
          {
          if( data[row + y][col] ) ++c;
          if( col % step == step - 1 ) ++y;
          }
        variance += ( c * c );
        }
      if( variance > variance_best )
        { variance_best = variance; slope_best = slope; }
      else if( variance < variance_best - (variance_best / 10) )
        try_inc = false;
      if( verbose )
        fprintf( stderr, "slope %3d   variance %Ld\n", slope, variance );
      }

    if( try_dec )
      {
      variance = 0;
      for( int row = slope; row < _height; ++row )
        {
        long c = -mean; int y = 0;
        for( int col = 0; col < _width; ++col )
          {
          if( data[row - y][col] ) ++c;
          if( col % step == step - 1 ) ++y;
          }
        variance += ( c * c );
        }
      if( variance > variance_best )
        { variance_best = variance; slope_best = -slope; }
      else if( variance < variance_best - (variance_best / 10) )
        try_dec = false;
      if( verbose )
        fprintf( stderr, "slope %3d   variance %Ld\n", -slope, variance );
      }
    }

  if( slope_best > 0 )
    {
    int row = 0;
    int step = _width / slope_best;
    if( step * slope_best < _width ) ++step;
    for( ; row < _height - slope_best; ++row )
      {
      for( int col = 0, y = 0; col < _width; ++col )
        {
        data[row][col] = data[row + y][col];
        if( col % step == step - 1 ) ++y;
        }
      }
    for( ; row < _height; ++row )
      for( int col = 0; col < _width; ++col ) data[row][col] = false;
    }
  else if( slope_best < 0 )
    {
    int row = _height - 1;
    int step = _width / -slope_best;
    if( step * -slope_best < _width ) ++step;
    for( ; row >= -slope_best; --row )
      {
      for( int col = 0, y = 0; col < _width; ++col )
        {
        data[row][col] = data[row - y][col];
        if( col % step == step - 1 ) ++y;
        }
      }
    for( ; row >= 0; --row )
      for( int col = 0; col < _width; ++col ) data[row][col] = false;
    }
  return slope_best;
  }


int Bitmap::find_columns( const bool verbose )
  {
  const int colmin = (_width / 10) + 60;
  const int gapmin = (_width / 100) + 6;
  std::vector< int > h_outline;
  int total_dots = 0, threshold_col, threshold_gap;
  int row, col;

  if( _rectangle_list.size() != 0 )
    throw Internal_error( "find_columns called twice.\n" );

  h_outline.resize( _width );

  for( col = 0; col < _width; ++col )
    {
    h_outline[col] = 0;
    for( row = 0; row < _height; ++row )
      if( data[row][col] ) ++h_outline[col];
    total_dots += h_outline[col];
    }

  threshold_col = (total_dots / (_width * 4)) + 6;
  threshold_gap = (total_dots / (_width * 20)) + 1;

  int left = 0, right, run_length;
  bool left_border = true;
  for( col = 0;; )
    {
    if( !left_border )			// look for text column
      {
      for( run_length = 0; col < _width-1; ++col )
        {
        if( h_outline[col] > threshold_col ) ++run_length;
        else { if( run_length < colmin ) run_length = 0; else break; }
        }
      }

    // look for gap
    for( run_length = 0; col < _width-1; ++col )
      {
      if( h_outline[col] < threshold_gap ) ++run_length;
      else { if( run_length < gapmin ) run_length = 0; else break; }
      }

    if( col >= _width-1 )
      {
      _rectangle_list.push_back( Rectangle( left, 0, _width-1, _height-1 ) );
      break;
      }

    right = col - (run_length / 2);
    if( left_border ) left_border = false;
    else _rectangle_list.push_back( Rectangle( left, 0, right, _height-1 ) );
    left = right;
    }

  if( verbose )
    fprintf( stderr, "number of text blocks %d\n", _rectangle_list.size() );
  return _rectangle_list.size();
  }


void Bitmap::save( FILE * f, char filetype )
  {
  if( filetype != '1' && filetype != '4' ) filetype = '4';
  fprintf( f, "P%c\n%d %d\n", filetype, _width, _height );

  for( int row = 0; row < _height; ++row )
    {
    if( filetype == '1' )
      {
      for( int col = 0; col < _width; ++col )
        putc( data[row][col] ? '1' : '0', f );
      putc( '\n', f );
      }
    else
      {
      unsigned char byte = 0, mask = 0x80;
      for( int col = 0; col < _width; ++col )
        {
        if( data[row][col] ) byte |= mask;
        mask >>= 1;
        if( mask == 0 ) { putc( byte, f ); byte = 0; mask = 0x80; }
        }
      if( mask != 0x80 ) putc( byte, f );  // incomplete byte at end of row
      }
    }
  fclose( f );
  }
