/*
    Numdiff - compare putatively similar files, 
    ignoring small numeric differences
    Copyright (C) 2005-2007  Ivano Primi  <ivprimi@libero.it>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
*/

#ifndef _NUMDIFF_H_
#define _NUMDIFF_H_

#include"config.h"

#ifdef HAVE_LONG_LONG
typedef long long integer;
#define STR2INT strtoll
#define INTEGER_MAX 4294967295LL
#else
#include<limits.h>

typedef long integer;
#define STR2INT strtol
#define INTEGER_MAX LONG_MAX
#endif 

/* Error codes */
#define OK            0
#define LINE_INTERR   1
#define EOF_REACHED   2
#define READING_ERROR 3
#define OUT_OF_MEMORY 4
/* *** */
#define OPEN_FAILED   5
#define WRONG_USAGE   6

/*
  Begin section -- Math types
*/

#ifdef _USE_MPA

#define ACCURACY_LEVEL 3

#include"number.h"

typedef bc_num Real;
typedef struct {
  bc_num re, im;
} Complex;

#elif defined(_USE_HPA)

#define ACCURACY_LEVEL 2

#include<cxpre.h>

typedef struct xpr   Real;
typedef struct cxpr  Complex;

#elif defined(_USE_LDBL)

#define ACCURACY_LEVEL 1

typedef long double  Real;
typedef struct {
  long double re, im;
} Complex;

#else

#define ACCURACY_LEVEL 0

typedef double  Real;
typedef struct {
  double re, im;
} Complex;

#endif /* _USE_MPA */

/*
  End section -- Math types
*/

struct numfmt {
  char dp;           /* decimal point       */
  char thsep;        /* thousands separator */
  unsigned grouping; /* Number of digits in each group */
  char pos_sign;     /* positive sign */
  char neg_sign;     /* negative sign */
  char ech;          /* prefix for decimal exponent  */
  char iu;           /* symbol of the imaginary unit */ 
}; 
/* A structure of this type is used to store  */
/* information about the legal format for the */
/* numbers in input.                          */

/*
  This is the number of the fields in the 'numftm' structure
  having "char" type. They must have all different values
  (see the code of the function valid_numfmt() in the file options.c) .
*/
#define NUMFMT_CHARS 6 

typedef struct {
  /* Mask of the options */
  unsigned long optmask;

  /* Line range (specified through the -L option) */
  integer firstln, lastln;

  /* This is a mask of bits specifying the fields in the files 
     which must be compared. */
  unsigned char fieldmask[FIELDMASK_SIZE];

  /* Tolerance for absolute and relative errors */
  Real maxabserr, maxrelerr;

  /* These 4 variables are used to print statistics */
  Real Labserr,  Crelerr,  Lrelerr,  Cabserr;

  /* Flag > 0 --> If a numeric field in the first file is greater than the  */
  /*              corresponding numeric field in the second file, then the  */
  /*              related difference is ignored, i.e. it is never output    */
  /* Flag < 0 --> If a numeric field in the first file is less than the     */
  /*              corresponding numeric field in the second file, then the  */
  /*              related difference is ignored, i.e. it is never output    */
  /* Flag = 0 --> Standard behavior: the difference between 2 corresponding */
  /*              numeric fields (one in the first file, the other one in   */
  /*              the second file) is always considered and it is output    */
  /*              whenever its absolute value is greater than the given     */
  /*              tolerance thresholds for the absolute and relative errors.*/
  signed char flag;

  /* Internal fields separators (IFS) */
  char* ifs;

  /* Internal scale (number of digits of accuracy) */
  int iscale;

  /* Files to be compared */
  const char *file1, *file2;

  /* Numeric conventions for file1 (.nf1) and file2 (.nf2) */
  struct numfmt nf1, nf2;

} argslist ; /* A structure of this type is used to store the options */
/* set by the user                                                    */ 

#define _H_MASK   0x00000001 /* -h option, used to recall help */
#define _A_MASK   0x00000002 /* -a option, used to set tolerance for abs. error  */
#define _R_MASK   0x00000004 /* -r option, used to set tolerance for rel. error  */
#define _2_MASK   0x00000008 /* -2 option, used to enable the "flexible" control */
#define _S_MASK   0x00000010 /* -s option, used to explicitly set IFS  */
#define _B_MASK   0x00000020 /* -b option, used to enable the "brief" mode */
#define _Q_MASK   0x00000040 /* -q option, used to enable "quiet" mode */
#define _X_MASK   0x00000080 /* -# option, used to set the precision   */
#define _D_MASK   0x00000100 /* -d option, used to set the decimal point */
#define _T_MASK   0x00000200 /* -t option, used to set the thousands separator */
#define _G_MASK   0x00000400 /* -g option, used to set the 'grouping' */
#define _P_MASK   0x00000800 /* -p option, used to set the character 
		  	    	'positive sign' */
#define _N_MASK   0x00001000 /* -n option, used to set the character 
		  	    	'negative sign' */
#define _E_MASK   0x00002000 /* -e option, used to set prefix for 
		  	    	decimal exponent */
#define _I_MASK   0x00004000 /* -i option, used to set the symbol
		  	    	of the imaginary unit */
#define _L_MASK   0x00008000 /* -l option, used to redirect the standard
		  	    	error on a file */
#define _O_MASK   0x00010000 /* -o option, used to redirect the standard
		  	    	output on a file */
#define _SF_MASK  0x00020000 /* -F option, used to select which fields in the
		  	    	lines of the files must be compared */
#define _SL_MASK  0x00040000 /* -L option, used to select the range of lines
				which must be compared */
#define _SP_MASK  0x00080000 /* -P option, used to ignore negative errors */
#define _SN_MASK  0x00100000 /* -N option, used to ignore positive errors */
#define _SD_MASK  0x00200000 /* -D option, used to enable the "dummy" mode */
#define _SE_MASK  0x00400000 /* -E option, used to enable the "essential" 
				mode */
#define _SV_MASK  0x00800000 /* -V option, used to enable the "verbose" mode */
#define _SS_MASK  0x01000000 /* -S option, used to print statistics */
#define _SI_MASK  0x02000000 /* -I option, used to ignore case while comparing
			        non numerical fields */
#define _V_MASK   0x04000000 /* -v option, used to show version number,
			        Copyright and No-Warrany */

/* Output modes: verbose, normal, brief, and quiet.    */
/* Do not change the relative order of the values of   */
/* these macros, the code in cmp_lines()  (see file    */
/* cmpfns.c) relies on the fact that:                  */
/* OUTMODE_VERBOSE > OUTMODE_NORMAL > OUTMODE_COINCISE */
/* > OUTMODE_BRIEF > OUTMODE_QUIET  .                  */

#define OUTMODE_VERBOSE     4
#define OUTMODE_NORMAL      3
#define OUTMODE_COINCISE    2
#define OUTMODE_BRIEF       1
#define OUTMODE_QUIET       0

/* I18N and L10N support */

#ifdef ENABLE_NLS
#include <libintl.h>
#define _(String) gettext (String)
#define gettext_noop(String) String
#define N_(String) gettext_noop (String)
#else
#define _(String) (String)
#define N_(String) String
#define textdomain(Domain)
#define bindtextdomain(Package, Directory)
#endif

#ifndef PACKAGE
#define PACKAGE "numdiff"
#endif

#ifndef LOCALEDIR
#define LOCALEDIR "/usr/local/share/locale/"
#endif

/*
  Predefined values for
  .nf*.dp       (decimal point)
  .nf*.thsep    (thousands separator)
  .nf*.grouping (number of digits in each thousands group)
  .nf*.pos_sign (positive sign)
  .nf*.neg_sign (negative sign)
  .nf*.ech      (prefix for decimal exponent)
  .nf*.iu       (symbol of the imaginary unit)
  .iscale      (decimal digits of accuracy)
  .ifs         (internal fields separator)
*/
#define DP        '.'
#define THSEP     ','
#define GROUPING   3
#define POS_SIGN  '+'
#define NEG_SIGN  '-'
#define ECH       'e'
#define IU        'i'
#define ISCALE    35
#define IFS  " \t\n"


/*
  Largest possible value for .iscale
*/
#define MAX_ISCALE  180

/*
  Character classification macros
*/
#define is_digit(c) ((c) >= '0' && (c) <= '9' ? 1 : 0)
#define is_punct(c) ispunct(c)
#define is_print(c) (isgraph(c) && ((c) < '0' || (c) > '9'))

/*
  Mathematical functions
*/

int      cmp (const Real *p, const Real *q);
int      is0 (const Real *u);
int      smart_cmp (const Complex* pz1, const Complex* pz2, int flag);
void     printno (Real u, int m);

#ifdef _USE_MPA

extern Real Zero, Inf;

void     init_mpa(void);
void     mpa_define_epsilon (int iscale, const struct numfmt* pnf);

void     initR (Real* px);
void     initC (Complex* pz);
Real     copyR (Real x);
Complex  copyC (Complex z);

#ifdef _MPA_DEBUG
void     debug_printno (Real u, int m);
#endif

void     str2R (const char *q, char **endptr, int iscale,
		const struct numfmt* pnf, Real* pr);
void     str2C (const char *q, char **endptr, int iscale,
		const struct numfmt* pnf, Complex* pc);

void     divide (Real s, Real t, Real* q, int iscale);

void     Cabs (Complex z, Real* pm, int iscale);
void     Csub (Complex z1, Complex z2, Complex* pw, int iscale);

void     delR (Real* px);
void     delC (Complex* pz);

void     mpa_undef_epsilon (void);
void     end_mpa(void);

#else

extern const Real Zero, Inf;

Real     str2R (const char *q, char **endptr);
Complex  str2C (const char *q, char **endptr);

Real     divide (Real s, Real t);

Real     Cabs (Complex z);
Complex  Csub (Complex z1, Complex z2);

#endif /* _USE_MPA */

#endif /* _NUMDIFF_H_ */
