/*  MaitreTarot.
 *  (C) 2002 Yves Mettier <ymettier@libertysurf.fr
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <stdio.h>
#include <string.h>
#include <libintl.h>
#define _(a) gettext (a)
#include <stdlib.h>
#include <maitretarot.h>
#include "main.h"
#include "config_utils.h"



#define DEFAULT_CONFIG_SERVER_PORT "2551"
#define DEFAULT_CONFIG_HOST_NAME "localhost"
#define DEFAULT_CONFIG_PLAYER_NAME "no_name"
#define DEFAULT_CONFIG_GEOMETRY "400x540+0+0"
#define DEFAULT_CONFIG_BG_COLOR "#22448C172244"
#define DEFAULT_CONFIG_TAKER_BID_COLOR "red"
#define DEFAULT_CONFIG_OTHER_BID_COLOR "blue"
#define DEFAULT_CONFIG_CARD_HEART "1"
#define DEFAULT_CONFIG_CARD_CLUB "2"
#define DEFAULT_CONFIG_CARD_DIAMOND "3"
#define DEFAULT_CONFIG_CARD_SPADE "4"
#define DEFAULT_CONFIG_CARD_TRUMP "5"
#define DEFAULT_CONFIG_ASCENDANT_ORDER "TRUE"
#define DEFAULT_CONFIG_UPPER_ORDER "TRUE"
#define DEFAULT_CONFIG_MAITRETAROT_SERVER_NAME "maitretarot_server &"
#define DEFAULT_CONFIG_PROGRAM_CLIENT_1 "mt_dolphin_ia -n IA1 &"
#define DEFAULT_CONFIG_PROGRAM_CLIENT_2 "mt_dolphin_ia -n IA2 &"
#define DEFAULT_CONFIG_PROGRAM_CLIENT_3 "mt_dolphin_ia -n IA3 &"
#define DEFAULT_CONFIG_DELAY "1"

#define DEFAULT_CONFIG_STRING "<?xml version=\"1.0\" standalone='yes'?>\n" \
                              "<maitretarot>\n" \
                              "  <client name='mt_gtk_client' type='client'>\n" \
                              "    <port>" DEFAULT_CONFIG_SERVER_PORT "</port>\n" \
                              "    <host_name><![CDATA[" DEFAULT_CONFIG_HOST_NAME "]]></host_name>\n" \
                              "    <player_name><![CDATA[" DEFAULT_CONFIG_PLAYER_NAME "]]></player_name>\n" \
                              "    <geometry>" DEFAULT_CONFIG_GEOMETRY "</geometry>\n" \
                              "    <cardpics>\n" \
                              "      <images_dir><![CDATA[" CARDPICS_DIR "]]></images_dir>\n" \
                              "    </cardpics>\n" \
                              "    <bg_color>" DEFAULT_CONFIG_BG_COLOR "</bg_color>\n" \
                              "    <taker_bid_color>" DEFAULT_CONFIG_TAKER_BID_COLOR "</taker_bid_color>\n" \
                              "    <other_bid_color>" DEFAULT_CONFIG_OTHER_BID_COLOR "</other_bid_color>\n" \
                              "    <card_heart>" DEFAULT_CONFIG_CARD_HEART "</card_heart>\n" \
                              "    <card_club>" DEFAULT_CONFIG_CARD_CLUB "</card_club>\n" \
                              "    <card_diamond>" DEFAULT_CONFIG_CARD_DIAMOND "</card_diamond>\n" \
                              "    <card_spade>" DEFAULT_CONFIG_CARD_SPADE "</card_spade>\n" \
                              "    <card_trump>" DEFAULT_CONFIG_CARD_TRUMP "</card_trump>\n" \
                              "    <card_ascendant>" DEFAULT_CONFIG_ASCENDANT_ORDER "</card_ascendant>\n" \
                              "    <card_upper>" DEFAULT_CONFIG_UPPER_ORDER "</card_upper>\n" \
                              "    <maitretarot_server>\n" \
                              "      <name><![CDATA["  DEFAULT_CONFIG_MAITRETAROT_SERVER_NAME "]]></name>\n" \
                              "    </maitretarot_server>\n" \
                              "    <program_client id='1'>\n" \
                              "      <name><![CDATA["  DEFAULT_CONFIG_PROGRAM_CLIENT_1 "]]></name>\n" \
                              "    </program_client>\n" \
                              "    <program_client id='2'>\n" \
                              "      <name><![CDATA["  DEFAULT_CONFIG_PROGRAM_CLIENT_2 "]]></name>\n" \
                              "    </program_client>\n" \
                              "    <program_client id='3'>\n" \
                              "      <name><![CDATA["  DEFAULT_CONFIG_PROGRAM_CLIENT_3 "]]></name>\n" \
                              "    </program_client>\n" \
                              "    <program_server_delay>" DEFAULT_CONFIG_DELAY "</program_server_delay>\n" \
                              "  </client>\n" \
                              "</maitretarot>"






gint
conf_update_with_command_line (config_t * conf, int argc, char **argv)
{
  int i;
  args_e prev_arg;

  prev_arg = ARGS_UNSET;
  for (i = 0; i < argc; i++)
    {
      if (prev_arg == ARGS_UNSET)
	{
	  if (!strcmp (argv[i], "-h"))
	    {
	      show_usage (argv[0]);
	      return (-1);
	    }
	  else if (!strcmp (argv[i], "--help"))
	    {
	      show_usage (argv[0]);
	      return (-1);
	    }
	  else if (!strcmp (argv[i], "--version"))
	    {
	      g_print (VERSION);
	      return (-2);
	    }
	  else if (!strcmp (argv[i], "-v"))
	    {
	      g_print (VERSION);
	      return (-2);
	    }
	  else if (!strncmp
		   (argv[i], "--no-start-screen",
		    sizeof ("--no-start-screen")))
	    {
	      conf->gui_start_screen = FALSE;
	    }
	  else
	    if (!strncmp
		(argv[i], "--cardpics-dir=", sizeof ("--cardpics-dir")))
	    {
	      conf->cardpics_dir =
		g_string_new (&(argv[i][sizeof ("--cardpics-dir")]));
	    }
	  else if (!strncmp (argv[i], "--port=", sizeof ("--port")))
	    {
	      conf->gui_port = atoi (&(argv[i][sizeof ("--port")]));
	    }
	  else if (!strncmp
		   (argv[i], "--player-name=", sizeof ("--player-name")))
	    {
	      if (conf->gui_player_name)
		g_string_free (conf->gui_player_name, TRUE);
	      conf->gui_player_name =
		g_string_new (&(argv[i][sizeof ("--player-name")]));
	    }
	  else if (!strncmp (argv[i], "--geometry=", sizeof ("--geometry")))
	    {
	      if (conf->gui_geometry)
		g_string_free (conf->gui_geometry, TRUE);
	      conf->gui_geometry =
		g_string_new (&(argv[i][sizeof ("--geometry")]));
	    }
	  else if (!strncmp (argv[i], "--server=", sizeof ("--server")))
	    {
	      if (conf->gui_host_name)
		g_string_free (conf->gui_host_name, TRUE);
	      conf->gui_host_name =
		g_string_new (&(argv[i][sizeof ("--server")]));
	    }
	  else if (!strcmp (argv[i], "-p"))
	    {
	      prev_arg = ARGS_PORT;
	    }
	  else if (!strcmp (argv[i], "-c"))
	    {
	      prev_arg = ARGS_CONFIG_FILE_NAME;
	    }
	  else if (!strcmp (argv[i], "-n"))
	    {
	      prev_arg = ARGS_PLAYERNAME;
	    }
	  else if (!strcmp (argv[i], "-g"))
	    {
	      prev_arg = ARGS_GEOMETRY;
	    }
	  else if (!strcmp (argv[i], "-s"))
	    {
	      prev_arg = ARGS_SERVER;
	    }
	}
      else if (prev_arg == ARGS_PORT)
	{
	  conf->gui_port = atoi (argv[i]);
	  prev_arg = ARGS_UNSET;
	}
      else if (prev_arg == ARGS_PLAYERNAME)
	{
	  if (conf->gui_player_name)
	    g_string_free (conf->gui_player_name, TRUE);
	  conf->gui_player_name = g_string_new (argv[i]);
	  prev_arg = ARGS_UNSET;
	}
      else if (prev_arg == ARGS_GEOMETRY)
	{
	  if (conf->gui_geometry)
	    g_string_free (conf->gui_geometry, TRUE);
	  conf->gui_geometry = g_string_new (argv[i]);
	  prev_arg = ARGS_UNSET;
	}
      else if (prev_arg == ARGS_SERVER)
	{
	  if (conf->gui_host_name)
	    g_string_free (conf->gui_host_name, TRUE);
	  conf->gui_host_name = g_string_new (argv[i]);
	  prev_arg = ARGS_UNSET;
	}
      else if (prev_arg == ARGS_CONFIG_FILE_NAME)
	{
	/* This one was already done before */
	  prev_arg = ARGS_UNSET;
	}
      else
	{
	  g_printerr ("This case should never appear (prev_arg=%d)",prev_arg);
	}
    }

  return (0);
}

void
conf_free (config_t * conf)
{
  int i;
  if (conf->cardpics_dir)
    g_string_free (conf->cardpics_dir, TRUE);
  if (conf->gui_host_name)
    g_string_free (conf->gui_host_name, TRUE);
  if (conf->gui_player_name)
    g_string_free (conf->gui_player_name, TRUE);
  if (conf->gui_geometry)
    g_string_free (conf->gui_geometry, TRUE);
  if (conf->gui_layout_bg_color)
    g_string_free (conf->gui_layout_bg_color, TRUE);
  for (i = 0; i < 4; i++)
    {
      if (conf->gui_command[i])
	g_string_free (conf->gui_command[i], TRUE);
    }
  if (config->config_file_name)
    g_free (config->config_file_name);
  g_free (conf);
}

config_t *
config_read (int argc, char **argv)
{
  config_t *conf;
  const gchar *key;
  int i;
  const gchar default_config_file_name[] = CONFIGFILENAME;
  const gchar *config_file_name = NULL;

/* Find if a config file name was specified in the command line */
  for (i = 0; i < argc; i++)
    {
      if (!strcmp (argv[i], "-c"))
	{
	  i++;
	  if (i < argc)
	    {
	      config_file_name = argv[i];
	      break;
	    }
	  else
	    {
	      show_usage (argv[0]);
	      return (NULL);
	    }
	}
      else if (!strncmp
	       (argv[i], "--config-file-name=",
		sizeof ("--config-file-name")))
	{
	  config_file_name = &(argv[i][sizeof ("--config-file-name")]);
	  break;
	}
    }
  if (!config_file_name)
    config_file_name = default_config_file_name;

/* Init the config */

  if (!libmt_xmlconfig_init_from_file (config_file_name))
    {
      g_message (_("Could not load config file. Using the default one. "));
      if (!libmt_xmlconfig_init_from_string (DEFAULT_CONFIG_STRING))
	{
	  g_error (_("Could use the default config. Aborting. "));
	  return (NULL);
	}
    }
  conf = g_malloc (sizeof (config_t));
  if (!conf)
    g_error (_("Not enough memory. Aborting."));

  conf->config_file_name = g_strdup (config_file_name);

  /* Start Screen  */
  conf->gui_start_screen = TRUE;

  /* Init gui_info_geometry */
  conf->gui_info_geometry = g_string_new (DEFAULT_CONFIG_GEOMETRY);

  /* Port */
  conf->gui_port = 0;
  key = libmt_xmlconfig_get_value
    ("/maitretarot/client[@name='mt_gtk_client']/port/text()");
  if (key)
    conf->gui_port = atoi (key);
  if (conf->gui_port == 0)
    conf->gui_port = atoi (DEFAULT_CONFIG_SERVER_PORT);

  /* host name */
  conf->gui_host_name = NULL;
  key = libmt_xmlconfig_get_value
    ("/maitretarot/client[@name='mt_gtk_client']/host_name/text()");
  if (key)
    conf->gui_host_name = g_string_new (key);
  if (!conf->gui_host_name)
    conf->gui_host_name = g_string_new (DEFAULT_CONFIG_HOST_NAME);

  /* player name */
  conf->gui_player_name = NULL;
  key = libmt_xmlconfig_get_value
    ("/maitretarot/client[@name='mt_gtk_client']/player_name/text()");
  if (key)
    conf->gui_player_name = g_string_new (key);
  if (!conf->gui_player_name)
    conf->gui_player_name = g_string_new (DEFAULT_CONFIG_PLAYER_NAME);

  /* cardpics dir */
  conf->cardpics_dir = NULL;
  key = libmt_xmlconfig_get_value
    ("/maitretarot/client[@name='mt_gtk_client']/cardpics/images_dir/text()");
  if (key)
    conf->cardpics_dir = g_string_new (key);
  if (!conf->cardpics_dir)
    conf->cardpics_dir = g_string_new (CARDPICS_DIR);

  /* geometry */
  conf->gui_geometry = NULL;
  key = libmt_xmlconfig_get_value
    ("/maitretarot/client[@name='mt_gtk_client']/geometry/text()");
  if (key)
    conf->gui_geometry = g_string_new (key);
  if (!conf->gui_geometry)
    conf->gui_geometry = g_string_new (DEFAULT_CONFIG_GEOMETRY);

  /* layout bg_color */
  conf->gui_layout_bg_color = NULL;
  key = libmt_xmlconfig_get_value
    ("/maitretarot/client[@name='mt_gtk_client']/bg_color/text()");
  if (key)
    conf->gui_layout_bg_color = g_string_new (key);
  if (!conf->gui_layout_bg_color)
    conf->gui_layout_bg_color = g_string_new (DEFAULT_CONFIG_BG_COLOR);

  /* layout taker_bid_color */
  conf->gui_taker_bid_color = NULL;
  key = libmt_xmlconfig_get_value
    ("/maitretarot/client[@name='mt_gtk_client']/taker_bid_color/text()");
  if (key)
    conf->gui_taker_bid_color = g_string_new (key);
  if (!conf->gui_taker_bid_color)
    conf->gui_taker_bid_color = g_string_new (DEFAULT_CONFIG_TAKER_BID_COLOR);

  /* layout other_bid_color */
  conf->gui_other_bid_color = NULL;
  key = libmt_xmlconfig_get_value
    ("/maitretarot/client[@name='mt_gtk_client']/other_bid_color/text()");
  if (key)
    conf->gui_other_bid_color = g_string_new (key);
  if (!conf->gui_other_bid_color)
    conf->gui_other_bid_color = g_string_new (DEFAULT_CONFIG_OTHER_BID_COLOR);

  /* Card heart */
  conf->gui_card_sort_value[0] = GUI_CARD_SORT_UNDEF;
  key = libmt_xmlconfig_get_value
    ("/maitretarot/client[@name='mt_gtk_client']/card_heart/text()");
  if (key)
    conf->gui_card_sort_value[0] = atoi (key);
  if (conf->gui_card_sort_value[0] == GUI_CARD_SORT_UNDEF)
    conf->gui_card_sort_value[0] = atoi (DEFAULT_CONFIG_CARD_HEART);

  /* Card club */
  conf->gui_card_sort_value[1] = GUI_CARD_SORT_UNDEF;
  key = libmt_xmlconfig_get_value
    ("/maitretarot/client[@name='mt_gtk_client']/card_club/text()");
  if (key)
    conf->gui_card_sort_value[1] = atoi (key);
  if (conf->gui_card_sort_value[1] == GUI_CARD_SORT_UNDEF)
    conf->gui_card_sort_value[1] = atoi (DEFAULT_CONFIG_CARD_CLUB);

  /* Card diamond */
  conf->gui_card_sort_value[2] = GUI_CARD_SORT_UNDEF;
  key = libmt_xmlconfig_get_value
    ("/maitretarot/client[@name='mt_gtk_client']/card_diamond/text()");
  if (key)
    conf->gui_card_sort_value[2] = atoi (key);
  if (conf->gui_card_sort_value[2] == GUI_CARD_SORT_UNDEF)
    conf->gui_card_sort_value[2] = atoi (DEFAULT_CONFIG_CARD_DIAMOND);

  /* Card spade */
  conf->gui_card_sort_value[3] = GUI_CARD_SORT_UNDEF;
  key = libmt_xmlconfig_get_value
    ("/maitretarot/client[@name='mt_gtk_client']/card_spade/text()");
  if (key)
    conf->gui_card_sort_value[3] = atoi (key);
  if (conf->gui_card_sort_value[3] == GUI_CARD_SORT_UNDEF)
    conf->gui_card_sort_value[3] = atoi (DEFAULT_CONFIG_CARD_SPADE);

  /* Card trump */
  conf->gui_card_sort_value[4] = GUI_CARD_SORT_UNDEF;
  key = libmt_xmlconfig_get_value
    ("/maitretarot/client[@name='mt_gtk_client']/card_trump/text()");
  if (key)
    conf->gui_card_sort_value[4] = atoi (key);
  if (conf->gui_card_sort_value[4] == GUI_CARD_SORT_UNDEF)
    conf->gui_card_sort_value[4] = atoi (DEFAULT_CONFIG_CARD_TRUMP);


  /* Ascendent/descendent sorting */
  if (g_ascii_strcasecmp (DEFAULT_CONFIG_ASCENDANT_ORDER, "FALSE") == 0)
    conf->gui_card_ascendant_order = FALSE;
  else
    conf->gui_card_ascendant_order = TRUE;
  key = libmt_xmlconfig_get_value
    ("/maitretarot/client[@name='mt_gtk_client']/card_ascendant/text()");
  if (key)
    {
      if (g_ascii_strcasecmp (key, "FALSE") == 0)
	conf->gui_card_ascendant_order = FALSE;
      else
	conf->gui_card_ascendant_order = TRUE;
    }

  /* upper/downer sorting */
  if (g_ascii_strcasecmp (DEFAULT_CONFIG_UPPER_ORDER, "FALSE") == 0)
    conf->gui_card_upper_order = FALSE;
  else
    conf->gui_card_upper_order = TRUE;
  key = libmt_xmlconfig_get_value
    ("/maitretarot/client[@name='mt_gtk_client']/card_upper/text()");
  if (key)
    {
      if (g_ascii_strcasecmp (key, "FALSE") == 0)
	conf->gui_card_upper_order = FALSE;
      else
	conf->gui_card_upper_order = TRUE;
    }

  conf->gui_command[0] = NULL;
  conf->gui_command[1] = NULL;
  conf->gui_command[2] = NULL;
  conf->gui_command[3] = NULL;


  /* maitretarot server program name */
  key = libmt_xmlconfig_get_value
    ("/maitretarot/client[@name='mt_gtk_client']/maitretarot_server/name/text()");
  if (key)
    conf->gui_command[0] = g_string_new (key);
  if (!conf->gui_command[0])
    conf->gui_command[0] = g_string_new (DEFAULT_CONFIG_MAITRETAROT_SERVER_NAME);

  /* client 1 program name */
  key = libmt_xmlconfig_get_value
    ("/maitretarot/client[@name='mt_gtk_client']/program_client[@id=1]/name/text()");
  if (key)
    conf->gui_command[1] = g_string_new (key);
  if (!conf->gui_command[1])
    conf->gui_command[1] = g_string_new (DEFAULT_CONFIG_PROGRAM_CLIENT_1);

  /* client 2 program name */
  key = libmt_xmlconfig_get_value
    ("/maitretarot/client[@name='mt_gtk_client']/program_client[@id=2]/name/text()");
  if (key)
    conf->gui_command[2] = g_string_new (key);
  if (!conf->gui_command[2])
    conf->gui_command[2] = g_string_new (DEFAULT_CONFIG_PROGRAM_CLIENT_2);

  /* client 3 program name */
  key = libmt_xmlconfig_get_value
    ("/maitretarot/client[@name='mt_gtk_client']/program_client[@id=3]/name/text()");
  if (key)
    conf->gui_command[3] = g_string_new (key);
  if (!conf->gui_command[3])
    conf->gui_command[3] = g_string_new (DEFAULT_CONFIG_PROGRAM_CLIENT_3);


  conf->gui_command_server_delay = GUI_COMMAND_SERVER_DELAY_UNDEF;
  key = libmt_xmlconfig_get_value
    ("/maitretarot/client[@name='mt_gtk_client']/program_server_delay/text()");
  if (key)
    conf->gui_command_server_delay = atoi (key);
  if (conf->gui_command_server_delay == GUI_CARD_SORT_UNDEF)
    conf->gui_command_server_delay = atoi (DEFAULT_CONFIG_DELAY);

  i = conf_update_with_command_line (conf, argc, argv);
  if (i != 0)
    {
      conf_free (conf);
      return (NULL);
    }
  return (conf);
}

static gboolean
write_channel (GIOChannel * chan, GString * st, GError ** error)
{
  GError *local_error=NULL;
  gsize blen;


  g_io_channel_write_chars (chan, st->str, -1, &blen, &local_error);
  if (local_error )
    {
      if (error)
	error[0] = g_error_copy (local_error);
      return (TRUE);
    }
  return (FALSE);
}

gint
config_write (GError ** error)
{
  GIOChannel *chan;
  GString *st;
  GError *local_error;
  gint i;

  local_error = NULL;

  chan = g_io_channel_new_file (config->config_file_name, "w", &local_error);

  if (local_error != NULL)
    {
      if (error)
	error[0] = g_error_copy (local_error);
      return (-1);
    }

  st = g_string_new ("");

  g_string_printf (st,
		   "<?xml version=\"1.0\" standalone='yes'?>\n"
		   "<maitretarot>\n"
		   "  <client name='mt_gtk_client' type='client'>\n");

  if (write_channel (chan, st, error))
    return (-1);

  g_string_printf (st, "    <port>%d</port>\n", config->gui_port);
  if (write_channel (chan, st, error))
    return (-1);


  g_string_printf (st, "    <host_name><![CDATA[%s]]></host_name>\n",
		   config->gui_host_name->str);
  if (write_channel (chan, st, error))
    return (-1);

  g_string_printf (st, "    <player_name><![CDATA[%s]]></player_name>\n",
		   config->gui_player_name->str);
  if (write_channel (chan, st, error))
    return (-1);

  g_string_printf (st, "    <geometry>%s</geometry>\n",
		   config->gui_geometry->str);
  if (write_channel (chan, st, error))
    return (-1);

  g_string_printf (st, "    <cardpics>\n");
  if (write_channel (chan, st, error))
    return (-1);

  g_string_printf (st, "      <images_dir><![CDATA[%s]]></images_dir>\n",
		   config->cardpics_dir->str);
  if (write_channel (chan, st, error))
    return (-1);

  g_string_printf (st, "    </cardpics>\n");
  if (write_channel (chan, st, error))
    return (-1);

  g_string_printf (st, "    <bg_color>%s</bg_color>\n",
		   config->gui_layout_bg_color->str);
  if (write_channel (chan, st, error))
    return (-1);

  g_string_printf (st, "    <taker_bid_color>%s</taker_bid_color>\n",
		   config->gui_taker_bid_color->str);
  if (write_channel (chan, st, error))
    return (-1);

  g_string_printf (st, "    <other_bid_color>%s</other_bid_color>\n",
		   config->gui_other_bid_color->str);
  if (write_channel (chan, st, error))
    return (-1);

  g_string_printf (st, "    <card_heart>%d</card_heart>\n",
		   config->gui_card_sort_value[0]);
  if (write_channel (chan, st, error))
    return (-1);

  g_string_printf (st, "    <card_club>%d</card_club>\n",
		   config->gui_card_sort_value[1]);
  if (write_channel (chan, st, error))
    return (-1);

  g_string_printf (st, "    <card_diamond>%d</card_diamond>\n",
		   config->gui_card_sort_value[2]);
  if (write_channel (chan, st, error))
    return (-1);

  g_string_printf (st, "    <card_spade>%d</card_spade>\n",
		   config->gui_card_sort_value[3]);
  if (write_channel (chan, st, error))
    return (-1);

  g_string_printf (st, "    <card_trump>%d</card_trump>\n",
		   config->gui_card_sort_value[4]);
  if (write_channel (chan, st, error))
    return (-1);

  g_string_printf (st, "    <card_ascendant>%s</card_ascendant>\n",
		   (config->gui_card_ascendant_order ? "TRUE" : "FALSE"));
  if (write_channel (chan, st, error))
    return (-1);

  g_string_printf (st, "    <card_upper>%s</card_upper>\n",
		   (config->gui_card_upper_order ? "TRUE" : "FALSE"));
  if (write_channel (chan, st, error))
    return (-1);

  g_string_printf (st, "    <maitretarot_server>\n");
  if (write_channel (chan, st, error))
    return (-1);

  g_string_printf (st, "      <name><![CDATA[%s]]></name>\n",
		   config->gui_command[0]->str);
  if (write_channel (chan, st, error))
    return (-1);

  g_string_printf (st, "    </maitretarot_server>\n");
  if (write_channel (chan, st, error))
    return (-1);

  for (i = 1; i < 4; i++)
    {
      g_string_printf (st, "    <program_client id='%d'>\n", i);
      if (write_channel (chan, st, error))
	return (-1);

      g_string_printf (st, "      <name><![CDATA[%s]]></name>\n",
		       config->gui_command[i]->str);
      if (write_channel (chan, st, error))
	return (-1);

      g_string_printf (st, "    </program_client>\n");
      if (write_channel (chan, st, error))
	return (-1);
    }
  g_string_printf (st,
		   "    <program_server_delay>%d</program_server_delay>\n",
		   config->gui_command_server_delay);
  if (write_channel (chan, st, error))
    return (-1);

  g_string_printf (st, "  </client>\n");
  if (write_channel (chan, st, error))
    return (-1);

  g_string_printf (st, "</maitretarot>");
  if (write_channel (chan, st, error))
    return (-1);


  g_io_channel_shutdown (chan, TRUE, &local_error);
  if (local_error != NULL)
    {
      if (error)
	error[0] = g_error_copy (local_error);
      g_string_free (st, TRUE);
      return (-1);
    }

  g_string_free (st, TRUE);
printf("Saved '%s'\n", config->config_file_name);
  return (0);
}



void
config_print (config_t * config)
{
  int i;
  printf ("\n==========================================\n"
	  "Dump of config_t structure\n"
	  "==========================================\n");
  printf ("gui_host_name = '%s'\n", config->gui_host_name->str);
  printf ("gui_port = '%d'\n", config->gui_port);
  printf ("gui_player_name = '%s'\n", config->gui_player_name->str);
  printf ("gui_geometry = '%s'\n", config->gui_geometry->str);
  printf ("gui_layout_bg_color = '%s'\n", config->gui_layout_bg_color->str);
  printf ("gui_taker_bid_color = '%s'\n", config->gui_taker_bid_color->str);
  printf ("gui_other_bid_color = '%s'\n", config->gui_other_bid_color->str);
  printf ("gui_command_server_delay = '%d'\n",
	  config->gui_command_server_delay);

  printf ("gui_card_ascendant_order = '%s'\n",
	  config->gui_card_ascendant_order ? "TRUE" : "FALSE");
  printf ("gui_card_upper_order = '%s'\n",
	  config->gui_card_upper_order ? "TRUE" : "FALSE");
  printf ("\n");
  for (i = 0; i < 4; i++)
    {
      printf ("gui_command[%d].command = '%s'\n", i,
	      config->gui_command[i]->str);
    }
  printf ("\n");
  for (i = 0; i < 5; i++)
    {
      printf ("gui_card_sort_value[%d] = '%d'\n", i,
	      config->gui_card_sort_value[i]);
    }
  printf ("==========================================\n");
}
