/*===========================================================================*/
/*
 * This file is part of libogg++ - a c++ library for transport of the Ogg format
 *
 * Copyright (C) 2006, 2007, 2008 Elaine Tsiang YueLien
 *
 * libogg++ is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc.
 * 51 Franklin Street, Fifth Floor
 * Boston, MA  02110-1301, USA
 *
 *===========================================================================*/
/** @file allPassThrough.C                                                   *
 *                                                                           */
/* allPassThrough - passing through all logical streams                      */
/*                                                                           */
/*===========================================================================*/
#include	<Ogg/Ogg.H>
#include	<Ogg/Debug.H>
#include	<Thread.H>

#include	<fstream>

extern "C" 
{
#include	<stdlib.h>
#include	<string.h>
#include	<unistd.h>
}

namespace	Examples
{
  using namespace	Ogg;

  class Demuxer: public Transport, public Thread
  {
  public:
    Demuxer()
      : Transport(false)
    {}

    ~Demuxer() throw(Ogg::Transport::NoDeletion())
    {}

    void
    run()
    {
      loop();
    }

    size_t
    recvCallback(
		 Page &	pg
		 )
    {
      size_t count = Transport::recvCallback(pg);

      if ( std::cin.eof()
	   ||
	   (count <= 0 )
	   )
	{
	  terminate();
	}

      return(count);
    }
  }
    ;

  class Muxer: public Transport, public Thread
  {
    std::ofstream	f;

  public:
    Muxer()
      : Transport(true)
    {}

    ~Muxer() throw(Ogg::Transport::NoDeletion())
    {}

    void
    run()
    {
      loop();
    }

    void
    sendCallback(
		 Page &	pg
		 )
    {
      if ( std::cout.write(static_cast<char *>(pg.data())
			   , pg.size()
			   )
	   )
	{
	  debug(true,
		"Serial No. ", pg.serialNo()
		," sent page ",pg.pageNo()
		," with granule position ", pg.granulePosition()
		);
	}
      else
	{
	  debug(true,
		"Serial No. ", pg.serialNo()
		," error occured sending page ",pg.pageNo()
		," with granule position ", pg.granulePosition()
		);
	}

      if ( pg.ending() )
	{
	  debug(true
		,"Serial No. ", pg.serialNo()
		," wrote ending page ",pg.pageNo()
		);
	  terminate();
	}
    }

    bool
    laterThan (
	       const Page &		p1
	       ,const Page &		p2
	       );
  }
    ;

  class Parser : public Logical, public virtual Thread
  {
  public:
    Parser(
	   Demuxer &	demuxer
	   )
      : Logical(demuxer)
    {}

  };

  class Writer : public Logical, public virtual Thread
  {
  public:
    Writer(
	   Muxer &	muxer
	   ,long	serialNo
	   )
      : Logical(muxer
		,serialNo
		)
    {}
  };

  class Passthrough : public Writer, public Parser
  {
  protected:
    const char * 	codecName;

  public:
    Passthrough(
		Muxer &	muxer
		,long	serialNo
		,Demuxer &	demuxer
		,const char *name
		)
      : Examples::Writer(muxer
			 ,serialNo
			 )
      , Parser(demuxer)
      , codecName(name)
    {}

    const
    char *
    name() const
    {
      return(codecName);
    }

    virtual
    Position
    time(Position	granulePosition) const = 0;


    void
    run()
    {
      try
	{
	  Examples::Writer::PageWriter & w = Examples::Writer::pageWriter();
	  Parser::PageReader & r = Parser::pageReader();

	  Page * page = &r.doGet();
	  while ( !page->ending() )
	    {
	      debug(true, codecName,
		    " serial ", page->serialNo()," : writing ",page->pageNo(),"th page, "
		    ,"granule position = ",page->granulePosition()
		    ,", pageOrderNo = ", page->pageOrderNo()
		    );
	      w.doPut(*page);
	      page = &r.doGet();
	    }
	  debug(true, codecName,
		" serial ", page->serialNo()," : writing page ",page->pageNo(),", last page, "
		,"granule position = ",page->granulePosition()
		,", pageOrderNo = ", page->pageOrderNo()
		);
	  w.doPut(*page);
	}
      catch ( exception ex
	      )
	{
	  debug(true, codecName,
		ex.what()
		);
	  throw;
	}

      return;
    }
  }
    ;

  
  bool
  Muxer::laterThan (
		    const Page &		p1
		    ,const Page &		p2
		    )
  {
    Passthrough * rw1 = static_cast<Passthrough *>
      (static_cast<Writer *>(logical(p1.serialNo())));
    Passthrough * rw2 = static_cast<Passthrough *>
      (static_cast<Writer *>(logical(p2.serialNo())));
      
    Position time1 = rw1->time(p1.granulePosition());
    Position time2 = rw2->time(p2.granulePosition());

    debug(true,
	  rw1->name()," vs ",rw2->name()
	  ,", granulee positions "
	  ,p1.granulePosition()
	  ," vs "
	  ,p2.granulePosition()
	  ,", times "
	  ,time1
	  ," vs "
	  ,time2
	  );
    debug(
	  true,
	  rw1->name()," vs ",rw2->name()
	  ,", pageOrderNo "
	  , p1.pageOrderNo()
	  ," vs "
	  , p2.pageOrderNo()
	  );


    if ( (p1.granulePosition() == 0) && (p2.granulePosition() == 0) )
      {
	if ( p1.pageNo() != p2.pageNo() )
	  return( p1.pageNo() >= p2.pageNo() );
	  else
	  return( p1.serialNo() < p2.serialNo() );
      }

    return( time1 >= time2 );
    /*
    return( p1.pageOrderNo() >= p2.pageOrderNo() );
    */
  }


  class TheoraPassthrough : public Passthrough
  {
    Position		granuleNumer;
    Position		granuleDenom;
    unsigned int	granuleShift;

  public:
    TheoraPassthrough(
		      Muxer &	muxer
		      ,long	serialNo
		      ,Demuxer &	demuxer
		      )
      : Passthrough(muxer
		    ,serialNo
		    ,demuxer
		    ,"theora"
		    )
    {}

    bool
    selectCallback(
		   Packet &	pkt
		   )
    { 
      const  char * header = static_cast<const  char *>(pkt.data());
      const unsigned char * uheader = static_cast<const unsigned char *>(pkt.data());

      if ( (pkt.size() < 41)
	   ||
	   (uheader[0] != 0x80)
	   ||
	   (strncmp (header+1, "theora", 6))
	   )
	return(false);

      unsigned int pos = (uheader[22] |
			  (uheader[23]<<8) |
			  (uheader[24]<<16) |
			  (uheader[25]<<24)
			  );
      granuleNumer = Position(pos);
      pos = (uheader[26] |
	     (uheader[27]<<8) |
	     (uheader[28]<<16) |
	     (uheader[29]<<24)
	     );
      granuleDenom = Position(pos);

      granuleNumer = granuleDenom/granuleNumer;

      if ( (granuleNumer < 25)
	   ||
	   (granuleDenom > 31)
	   )
	{
	  pos = (uheader[25] |
		 (uheader[24]<<8) |
		 (uheader[23]<<16) |
		 (uheader[22]<<24)
		 );
	  granuleNumer = Position(pos);
	  pos = (uheader[29] |
		 (uheader[28]<<8) |
		 (uheader[27]<<16) |
		 (uheader[26]<<24)
		 );
	  granuleDenom = Position(pos);
	}
      
      char
	shift  = ((header[40] & 0x03) << 3);
      shift |= (header[41] & 0xe0) >> 5;
      granuleShift = Position(shift);

      debug(true,codecName,
	    " numerator = ", granuleNumer
	    ," denominator = ", granuleDenom
	    ," shift = ", granuleShift
	    );

      return(true);
    }


    Position
    time(Position	granulePosition) const
    {
      Position iframe = granulePosition >> granuleShift;
      Position pframe = granulePosition - (iframe << granuleShift);
      return( (iframe + pframe) * granuleDenom * 1000/ granuleNumer );
    }
  }
    ;
  
  class VorbisPassthrough : public Passthrough
  {
    Position		granuleRate;

  public:
    VorbisPassthrough(
		      Muxer &	muxer
		      ,long	serialNo
		      ,Demuxer &	demuxer
		      )
      : Passthrough(muxer
		    ,serialNo
		    ,demuxer
		    ,"vorbis"
		    )
    {}

    bool
    selectCallback(
		   Packet &	pkt
		   )
    {
      const  char * header = static_cast<const  char *>(pkt.data());
      const unsigned char * uheader = static_cast<const unsigned char *>(pkt.data());

      if ( (pkt.size() < 30)
	   ||
	   (uheader[0] != 0x01)
	   ||
	   (strncmp (header+1, "vorbis", 6))
	   )
	return(false);

      unsigned int pos = (uheader[12] |
			  (uheader[13]<<8) |
			  (uheader[14]<<16) |
			  (uheader[15]<<24)
			  );
      granuleRate = Position(pos);

      debug(true,codecName,
	    " rate ", granuleRate
	    );
      return(true);
    }

    Position
    time(Position	granulePosition) const
    {
      return(granulePosition*1000/granuleRate);
    }
  };

}

int main(int, const char**)
{
  using namespace	Examples;
  using namespace	Ogg;

  Demuxer & demuxer = *(new Demuxer);
  Muxer & muxer	= *(new Muxer);
  VorbisPassthrough & vorwriter = *(new VorbisPassthrough(muxer
							  ,450116321
							  ,demuxer
							  ));
  TheoraPassthrough& theowriter = *(new TheoraPassthrough(muxer
							  ,658032240
							  ,demuxer
							  ));

  muxer.start();
  demuxer.start();
  vorwriter.start();
  theowriter.start();

  vorwriter.finish();
  theowriter.finish();
  demuxer.finish();
  muxer.finish();

  exit(0);
}
